﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_SOUND_ARCHIVE_FILE_EDITOR_H_
#define NW_SND_EDIT_SOUND_ARCHIVE_FILE_EDITOR_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/snd_SoundArchiveFilesHook.h>
#include <nw/snd/edit/sndedit_Result.h>
#include <nw/snd/edit/res/sndedit_ResItemInfo.h>

namespace nw {

#ifdef NW_PLATFORM_CTR
namespace io {
#else
namespace ut {
#endif
class FileStream;
}

namespace snd {
namespace edit {
namespace internal {

class SoundArchiveEditController;

//---------------------------------------------------------------------------
//! @brief    サウンドアーカイブのファイル編集機能を提供するクラスです。
//---------------------------------------------------------------------------
class SoundArchiveFileEditor
{
    NW_DISALLOW_COPY_AND_ASSIGN(SoundArchiveFileEditor);

public:
    //---------------------------------------------------------------------------
    //! @brief  サウンドアーカイブのパラメータアクセスをフックし、パラメータを指し替えます。
    //---------------------------------------------------------------------------
    class Hook : public snd::internal::SoundArchiveFilesHook
    {
        NW_DISALLOW_COPY_AND_ASSIGN(Hook);

    public:
        Hook() : m_Owner(NULL) { }

    public:
        void Initialize(SoundArchiveFileEditor& owner)
        {
            m_Owner = &owner;
        }

    protected:
        //---------------------------------------------------------------------------
        //! @brief         指定アイテムラベルがフック対象かどうかを調べます。
        //!
        //! @param[in]     itemLabel     アイテムラベルを指定します。
        //!
        //! @return        対象の場合は true、対象でない場合は false を返します。
        //---------------------------------------------------------------------------
        virtual bool IsTargetItemImpl(const char* itemLabel);

        //---------------------------------------------------------------------------
        //! @brief         フックの有無が変化しないようにロックします。
        //---------------------------------------------------------------------------
        virtual void LockImpl();

        //---------------------------------------------------------------------------
        //! @brief         フックのロックを解除します。
        //---------------------------------------------------------------------------
        virtual void UnlockImpl();

        //---------------------------------------------------------------------------
        //! @brief         指定バッファを使って指定アイテムラベルに関連するファイルを開きます。
        //!
        //! @param[in]     buffer        ファイルを開くのに利用するバッファを指定します。
        //! @param[in]     bufferLength  バッファの長さを指定します。
        //! @param         cacheBuffer   TBD
        //! @param         cacheBufferLength  TBD
        //! @param[in]     itemLabel     アイテムラベルを指定します。
        //! @param[in]     fileType      取得するファイルタイプを指定します。
        //!
        //! @return        FileStream を返します。
        //---------------------------------------------------------------------------
#ifdef NW_PLATFORM_CTR
        virtual io::FileStream* OpenFileImpl(void* buffer, u32 bufferLength, void* cacheBuffer, u32 cacheBufferLength, const char* itemLabel, const char* fileType);
#else
        virtual ut::FileStream* OpenFileImpl(void* buffer, u32 bufferLength, void* cacheBuffer, u32 cacheBufferLength, const char* itemLabel, const char* fileType);
#endif

        //---------------------------------------------------------------------------
        //! @brief         指定アイテムラベルに関連するファイルデータへのアドレスを取得します。
        //!
        //! @param[in]     itemLabel    アイテムラベルを指定します。
        //! @param[in]     itemType     アイテム種別を指定します。
        //! @param[in]     fileType     取得するファイルタイプを指定します。
        //! @param[in]     fileIndex    取得するファイルのインデックスがあれば指定します。
        //!
        //! @return        ファイルデータへのアドレスを返します。
        //---------------------------------------------------------------------------
        virtual const void* GetFileAddressImpl(const char* itemLabel, const char* itemType, const char* fileType, u32 fileIndex = 0);

    private:
        const ResName* GetBankFilePath(const char* itemLabel, ResDataType fileDataType, u32 bankIndex);

        ResDataType StringToResDataType(const char* value) const;

    private:
        SoundArchiveFileEditor* m_Owner;
    };

public:
    /* ctor */ SoundArchiveFileEditor();
    /* dtor */ ~SoundArchiveFileEditor() {}

    //! @brief  初期化します。
    //! @param editController TBD
    //! @param soundArchivePlayer TBD
    //! @return TBD
    Result Initialize(
        SoundArchiveEditController* editController,
        SoundArchivePlayer* soundArchivePlayer);

    //! @brief  終了処理を行います。
    void Finalize();

    bool IsInitialized() const { return m_SoundArchivePlayer != NULL; }

    //! @brief  ファイル編集を開始します。
    void Start();

    //! @brief  ファイル編集を終了します。
    void Stop();

private:
    Hook                        m_SoundArchiveHook;
    SoundArchiveEditController* m_EditController;
    SoundArchivePlayer*         m_SoundArchivePlayer;
};

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_SOUND_ARCHIVE_FILE_EDITOR_H_

