﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_RESULT_H_
#define NW_SND_EDIT_RESULT_H_

namespace nw {
namespace snd {
namespace edit {

//---------------------------------------------------------------------------
//! @brief  【β版】sndedit の処理結果です。
//---------------------------------------------------------------------------
enum ResultType
{
    //! @briefprivate
    SNDEDIT_RESULT_ERROR_FLAG_MASK     = 0x80000000,
    //! @briefprivate
    SNDEDIT_RESULT_CATEGORY_MASK       = 0x7f000000,
    //! @briefprivate
    SNDEDIT_RESULT_CODE_MASK           = 0x00ffffff,
    //! @briefprivate
    SNDEDIT_RESULT_CATEGORY_BIT_OFFSET = 24,

    //! @briefprivate
    SNDEDIT_RESULT_ERROR_FLAG          = SNDEDIT_RESULT_ERROR_FLAG_MASK,
    //! @briefprivate
    SNDEDIT_RESULT_CATEGORY_SYSTEM     = 0x00 << SNDEDIT_RESULT_CATEGORY_BIT_OFFSET,
    //! @briefprivate
    SNDEDIT_RESULT_CATEGORY_CONNECTION = 0x01 << SNDEDIT_RESULT_CATEGORY_BIT_OFFSET,

    //---------------------------------------------------------------------------
    // 成功コード（システム）
    //---------------------------------------------------------------------------

    SNDEDIT_RESULT_TRUE  = 0x000000 + SNDEDIT_RESULT_CATEGORY_SYSTEM,   //!< 成功しました。
    SNDEDIT_RESULT_FALSE = 0x000001 + SNDEDIT_RESULT_CATEGORY_SYSTEM,   //!< 処理を実行しませんでした。

    //---------------------------------------------------------------------------
    // 失敗コード（システム）
    //---------------------------------------------------------------------------

    SNDEDIT_RESULT_FAILED            = 0x000000 + SNDEDIT_RESULT_CATEGORY_SYSTEM + SNDEDIT_RESULT_ERROR_FLAG,    //!< 失敗しました。
    SNDEDIT_RESULT_NOT_INITIALIZED   = 0x000001 + SNDEDIT_RESULT_CATEGORY_SYSTEM + SNDEDIT_RESULT_ERROR_FLAG,    //!< 初期化されていません。
    SNDEDIT_RESULT_NOT_CONNECTED     = 0x000002 + SNDEDIT_RESULT_CATEGORY_SYSTEM + SNDEDIT_RESULT_ERROR_FLAG,    //!< 接続されていません。
    SNDEDIT_RESULT_OUT_OF_MEMORY     = 0x000003 + SNDEDIT_RESULT_CATEGORY_SYSTEM + SNDEDIT_RESULT_ERROR_FLAG,    //!< メモリ不足です。
    SNDEDIT_RESULT_CACHE_OVER_FLOW   = 0x000004 + SNDEDIT_RESULT_CATEGORY_SYSTEM + SNDEDIT_RESULT_ERROR_FLAG,    //!< キャッシュ数が上限を超えました。
    SNDEDIT_RESULT_NAME_NOT_FOUND    = 0x000005 + SNDEDIT_RESULT_CATEGORY_SYSTEM + SNDEDIT_RESULT_ERROR_FLAG,    //!< 名前が見つかりません。
    SNDEDIT_RESULT_NAME_TOO_LONG     = 0x000006 + SNDEDIT_RESULT_CATEGORY_SYSTEM + SNDEDIT_RESULT_ERROR_FLAG,    //!< 名前が長すぎます。
    SNDEDIT_RESULT_INVALID_DATA_TYPE = 0x000007 + SNDEDIT_RESULT_CATEGORY_SYSTEM + SNDEDIT_RESULT_ERROR_FLAG,    //!< データタイプが正しくありません。
    SNDEDIT_RESULT_ITEM_PREPAREING   = 0x000008 + SNDEDIT_RESULT_CATEGORY_SYSTEM + SNDEDIT_RESULT_ERROR_FLAG,    //!< アイテムは準備中です。

    //---------------------------------------------------------------------------
    // 失敗コード（通信）
    //---------------------------------------------------------------------------

    SNDEDIT_RESULT_TIMEOUT  = 0x000000 + SNDEDIT_RESULT_CATEGORY_CONNECTION + SNDEDIT_RESULT_ERROR_FLAG,   //!< タイムアウトが発生しました。
    SNDEDIT_RESULT_HIOERROR = 0x000001 + SNDEDIT_RESULT_CATEGORY_CONNECTION + SNDEDIT_RESULT_ERROR_FLAG    //!< HIO 接続に失敗しました。
};

//---------------------------------------------------------------------------
//! @brief  【β版】sndedit の処理結果を制御します。
//---------------------------------------------------------------------------
class Result
{
public:
    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    Result() : value(SNDEDIT_RESULT_TRUE) { }

    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  コンストラクタです。
    //! @param value :private
    //---------------------------------------------------------------------------
    explicit Result(u32 value) : value(value) { }

    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  コンストラクタです。
    //! @param value :private
    //---------------------------------------------------------------------------
    explicit Result(ResultType value) : value(value) { }

    //---------------------------------------------------------------------------
    //! @brief  結果が成功かどうかを調べます。
    //! @return 成功の場合は true、失敗の場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsSucceeded() const { return !IsFailed(); }

    //---------------------------------------------------------------------------
    //! @brief  結果が成功かつ TRUE かどうかを調べます。
    //! @return 成功かつ TRUE の場合は true、それ以外の場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsTrue() const { return value == SNDEDIT_RESULT_TRUE; }

    //---------------------------------------------------------------------------
    //! @brief  結果が成功かつ FALSE かどうかを調べます。
    //! @return 成功かつ FALSE の場合は true、それ以外の場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsFalse() const { return value == SNDEDIT_RESULT_FALSE; }

    //---------------------------------------------------------------------------
    //! @brief  結果が失敗かどうかを調べます。
    //! @return 失敗の場合は true、成功の場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsFailed() const { return (value & SNDEDIT_RESULT_ERROR_FLAG) != 0; }

    //---------------------------------------------------------------------------
    //! @brief  結果コードを取得します。
    //! @return 結果コードを返します。
    //---------------------------------------------------------------------------
    ResultType GetValue() const { return static_cast<ResultType>(value); }

    //---------------------------------------------------------------------------
    //! @brief  Result を文字列に変換します。
    //!
    //!         この関数は DEBUG または DEVELOP ビルド時にのみ正しい値を返します。
    //!
    //! @return DEBUG または DEVELOP ビルド時には、変換後の文字列を返します。
    //!
    //!         RELEASE ビルド時には必ず空文字を返します。
    //---------------------------------------------------------------------------
    const char* ToString() const
    {
#if !defined(NW_RELEASE)
        switch(value)
        {
        //---------------------------------------------------------------------------
        // 成功コード（システム）
        case SNDEDIT_RESULT_TRUE:
            return "SNDEDIT_RESULT_TRUE";

        case SNDEDIT_RESULT_FALSE :
            return "SNDEDIT_RESULT_FALSE";

        //---------------------------------------------------------------------------
        // 失敗コード（システム）
        case SNDEDIT_RESULT_FAILED:
            return "SNDEDIT_RESULT_FAILED";

        case SNDEDIT_RESULT_NOT_INITIALIZED:
            return "SNDEDIT_RESULT_NOT_INITIALIZED";

        case SNDEDIT_RESULT_NOT_CONNECTED:
            return "SNDEDIT_RESULT_NOT_CONNECTED";

        case SNDEDIT_RESULT_OUT_OF_MEMORY:
            return "SNDEDIT_RESULT_OUT_OF_MEMORY";

        case SNDEDIT_RESULT_CACHE_OVER_FLOW:
            return "SNDEDIT_RESULT_CACHE_OVER_FLOW";

        case SNDEDIT_RESULT_NAME_NOT_FOUND:
            return "SNDEDIT_RESULT_NAME_NOT_FOUND";

        case SNDEDIT_RESULT_NAME_TOO_LONG:
            return "SNDEDIT_RESULT_NAME_TOO_LONG";

        case SNDEDIT_RESULT_INVALID_DATA_TYPE:
            return "SNDEDIT_RESULT_INVALID_DATA_TYPE";

        case SNDEDIT_RESULT_ITEM_PREPAREING:
            return "SNDEDIT_RESULT_ITEM_PREPAREING";

        //---------------------------------------------------------------------------
        // 失敗コード（通信）
        case SNDEDIT_RESULT_TIMEOUT:
            return "SNDEDIT_RESULT_TIMEOUT";

        case SNDEDIT_RESULT_HIOERROR:
            return "SNDEDIT_RESULT_HIOERROR";
        }
#endif

        return "";
    }

    //--------------------------------------------------------------------------
    //! @briefprivate
    //! @brief   u32 型へのキャスト演算子です。
    //!
    //! @return  u32 値を返します。
    //---------------------------------------------------------------------------
    operator u32() const
    {
        return value;
    }

    //--------------------------------------------------------------------------
    //! @briefprivate
    //! @brief   ResultType 型へのキャスト演算子です。
    //!
    //! @return  ResultType で表現した値を返します。
    //---------------------------------------------------------------------------
    operator ResultType() const
    {
        return static_cast<ResultType>(value);
    }

private:
    u32 value;
};


} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_EDIT_RESULT_H_
