﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SCFONT_SCALABLE_FONT_H_
#define NW_SCFONT_SCALABLE_FONT_H_

#include <nw/scfont/scfont_TextureCache.h>

namespace nw
{
namespace scfont
{

//---------------------------------------------------------------------------
//! @brief スケーラブルフォントを表すクラスです。
//!
//! @details
//! nw::font::Fontを継承するフォント(ResFontなど)と同じように使用することが
//! できますが、クラスを作成した状態ではまだ文字を表示することができず、
//! 明示的に表示する必要がある文字を指定してテクスチャキャッシュ内にプロット
//! しなければならないという違いがあります。
//---------------------------------------------------------------------------
class ScalableFont : public font::Font
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(font::Font);

    //----------------------------------------------------
    //! @name コンストラクタ／デストラクタ/初期化/終了処理
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 実質的な初期化はInitializeで行われます。ここでは、
    //! メンバ変数の初期化のみ行っています。
    //!
    ScalableFont();

    //! @brief デストラクタです。
    //!
    virtual ~ScalableFont();

    //! @brief 初期化のパラメータを指定するための構造体です。
    //!
    struct InitializeArg {
        //! このフォントで使用するテクスチャキャッシュです。
        //! テクスチャキャッシュは予め初期化しておく必要があります。
        //! デフォルト値はありません。必ず指定してください。
        TextureCache* textureCache;
        //! このインスタンスで扱うフォントのサイズです。
        //! デフォルトでは40です。
        int fontSize;
        //! フォントの字形を指定する数値です。
    	//! TextureCacheのInitializeで与えたttf(otf)のバイナリポインタの配列のインデックスを
        //! 与えてください。
        //! デフォルトでは0です。
        u16 fontFace;
        //! このメンバをtrueにすると、フォントの代替文字を空白にし、かつその幅を元々
        //! 表示するはずだった文字と同じサイズにします。falseにしている場合の挙動は、
        //! isAlternateCharSpaceWithOriginalWidthForNotReadyCharによります。
        //! デフォルトではfalseです。
        bool isAlternateCharSpaceWithOriginalWidth;
        //! このメンバはisAlternateCharSpaceWithOriginalWidthがfalseのときのみ有効です。
        //! このメンバをtrueにすると、フォントにない文字については通常の代替文字
        //! (デフォルトでは'?')が使われますが、フォントにあってまだプロットされていない
        //! 文字については、フォントの代替文字を空白にし、かつその幅を元々表示するはず
        //! だった文字と同じサイズにします。
        //! falseにしている場合は、通常の代替文字が使われます。
        //! デフォルトではfalseです。
        bool isAlternateCharSpaceWithOriginalWidthForNotReadyChar;

        //! @brief コンストラクタです。デフォルト値で初期化します。
        InitializeArg();
    };

    //! @brief 初期化を行います。
    //!
    //! @details
    //! 内部でリソースの確保は行いませんので、Finalizeメソッドはありません。
    //!
    //! @param arg  パラメータを指定する構造体
    //!
    void Initialize(const InitializeArg& arg);
    //@}

    //----------------------------------------------------
    //! @name テクスチャキャッシュの操作
    //@{

    //! @brief 指定した文字を、描画する文字として登録します。
    //!
    //! @details
    //! 描画する文字として登録した文字は、テクスチャキャッシュになかった場合は、新たに
    //! プロットする必要がある文字として登録されます。この文字は、UpdateTextureCacheと
    //! CompleteTextureCacheを行った段階で使えるようになります。
    //!
    //! 描画する文字として登録した文字が、テクスチャキャッシュに既にあった場合は、
    //! プロット等は行われず、引き続き使用できます。また、その文字は必要な文字として
    //! 登録されますので、削除対象になりません。
    //!
    //! 文字をロックする際のグループは、0から31までの32個あります。
    //! ロックはグループ毎に行われますので、例えばグル―プ0で登録された文字がグループ1
    //! でもう一度登録された場合、その両方でロックされたことになり、両方についてロック
    //! がクリアされないと削除対象になりません。
    //!
    //! @param code         描画する文字として登録する文字
    //! @param lockGroup    文字をロックする際のグループ。0から31までの数値で指定してください。-1を指定した場合はロックしません。
    //! @return 登録した文字のプロットが必要な場合にtrueが返されます。つまり、falseが返った場合はその文字は
    //!         UpdateTextureCacheを行わなくても描画できます。
    //!
    bool RegisterGlyph(char16 code, s32 lockGroup = -1) const
    {
        return m_TextureCache->RegisterGlyph(code, m_FontSize, m_FontFace, lockGroup);
    }

    //! @brief 指定した文字列に含まれる文字を、描画する文字として登録します。
    //!
    //! @details
    //! 登録する文字を文字列として指定できる以外は @ref RegisterGlyph と同じです。
    //!
    //! @param codes        描画する文字として登録する文字列。NULL終端する必要があります。
    //! @param lockGroup    文字をロックする際のグループ。0から31までの数値で指定してください。-1を指定した場合はロックしません。
    //! @return プロットが必要な文字が何文字あったかを返します。つまり0が返された場合は、
    //!         全ての文字がUpdateTextureCacheを行わなくても描画できることを示します。
    //!
    u32 RegisterGlyphs(const char16* codes, s32 lockGroup = -1) const
    {
        return m_TextureCache->RegisterGlyphsWithLength(codes, UINT_MAX, m_FontSize, m_FontFace, lockGroup);
    }

    //! @brief 指定した文字列に含まれる指定した文字数だけの文字を、描画する文字として登録します。
    //!
    //! @details
    //! 登録する文字数を指定できる以外は @ref RegisterGlyphs と同じです。
    //!
    //! @param codes        描画する文字として登録する文字列
    //! @param codeLength   文字列の文字数。この文字数より短い文字数でNULL終端が来た場合は、登録はそこで終了します。
    //! @param lockGroup    文字をロックする際のグループ。0から31までの数値で指定してください。-1を指定した場合はロックしません。
    //! @return プロットが必要な文字が何文字あったかを返します。つまり0が返された場合は、
    //!         全ての文字がUpdateTextureCacheを行わなくても描画できることを示します。
    //!
    u32 RegisterGlyphsWithLength(const char16* codes, u32 codeLength, s32 lockGroup = -1) const
    {
        return m_TextureCache->RegisterGlyphsWithLength(codes, codeLength, m_FontSize, m_FontFace, lockGroup);
    }

    //! @brief 指定した文字列に含まれる文字のロックを解除します。
    //!
    //! @param[in] codes        ロックを解除する文字が含まれる文字列。NULL終端する必要があります。
    //! @param[in] lockGroup    ロックを解除するグループ。0から31までの数値で指定してください。
    //!
    void ClearLockGlyphs(const char16* codes, s32 lockGroup) const
    {
        m_TextureCache->ClearLockGlyphsWithLength(codes, UINT_MAX, m_FontSize, m_FontFace, lockGroup);
    }

    //! @brief 指定した長さの文字列に含まれる文字のロックを解除します。
    //!
    //! @param codes        ロックを解除する文字が含まれる文字列。
    //! @param codeLength   文字列の文字数。この文字数より短い文字数でNULL終端が来た場合は、そこで終了します。
    //! @param lockGroup    ロックを解除するグループ。0から31までの数値で指定してください。
    //!
    void ClearLockGlyphsWithLength(const char16* codes, u32 codeLength, s32 lockGroup) const
    {
        m_TextureCache->ClearLockGlyphsWithLength(codes, codeLength, m_FontSize, m_FontFace, lockGroup);
    }

    //! @brief このフォントが利用しているテクスチャキャッシュを取得します。
    //!
    //! @return テクスチャキャッシュです。
    //!
    TextureCache* GetTextureCache() const { return m_TextureCache; }

    //! @brief このフォントで使用している代替文字を登録します。
    //!
    //! @details
    //! 通常代替文字の登録はInitializeで行われますが、TextureCacheをリセットすると、
    //! 代替文字も消えてしまうため再登録が必要となります。
    //!
    void RegisterAlternateCharGlyph() const;
    //@}

    //----------------------------------------------------
    //! @name フォントに関する情報の取得
    //@{
    //! @brief 初期化時に設定したフォントの字形を指定する数値を取得します。
    //!
    //! @return フォントの字形を指定する数値
    //!
    u16 GetFontFace() const { return m_FontFace; }

    //! @brief 初期化時に設定したフォントのサイズを指定する数値を取得します。
    //!
    //! @return 初期化時に設定したフォントのサイズを指定する数値
    //!
    u32 GetFontSizeOfInitializeArg() const { return m_FontSize; }

    //! @brief このフォントを利用して、マルチコアでの描画が行えるか否かを返します。
    //!
    //! @return マルチコアでの描画が行えるか否か
    //!
    bool IsDrawMultiCore() const { return m_TextureCache->IsDrawMultiCore(); }

    //! @brief  フォントの中に指定した文字コードの文字が含まれているかを返します。
    //!
    //! @details
    //! 登録されているTextureCacheクラスのIsGlyphExistInFontメソッドを呼び出す
    //! ユーティリティです。テクスチャキャッシュの状態には関係なく、ttf(otf)の中に
    //! 文字が含まれているかを返します。
    //!
    //! @param code         幅を求めたい文字
    //! @return 含まれるか否か
    //!
    bool IsGlyphExistInFont(char16 code) const { return m_TextureCache->IsGlyphExistInFont(code, m_FontFace); }
    //@}

    //----------------------------------------------------
    //! @name フォントの設定を変更するためのインターフェース
    //@{
    /**
     *  フォントの高さを変更します。
     *
     *  通常、フォントの高さはinitializeメソッドによって自動的に設定され、それを
     *  修正する必要はありません。
     *
     *  このメソッドを使うことで、異なるフォントを同じテキストボックスで使いたい
     *  場合に、同じくらいのサイズに見えるようにフォントの縦方向のスケールを調整
     *  することができます。
     *
     *  @param height   設定したい高さ
     */
    void                    SetHeight(int height) { m_FontHeight = height; }

    /**
     *  フォントのベースライン位置を変更します。
     *
     *  通常、フォントのベースライン位置はinitializeメソッドによって自動的に設定され、
     *  それを修正する必要はありません。
     *
     *  このメソッドを使うことで、SetHeightでフォントの縦方向のスケールを調整した
     *  ときに、縦方向の表示位置がずれてしまうのを調整することができます。
     *
     *  @param[in] pos  設定したいベースライン位置
     */
    void                    SetBaselinePos(int pos) { m_FontBaselinePos = pos; }
    //@}

    //----------------------------------------------------
    //! @name Fontクラスのインターフェース
    //@{

    //! @brief      フォントの幅を取得します。
    //!
    //! @return     ピクセル単位でのフォントの幅を返します。
    //!
    virtual int             GetWidth() const;

    //! @brief      フォントの高さを取得します。
    //!
    //! @return     ピクセル単位でのフォントの高さを返します。
    //!
    virtual int             GetHeight() const;

    //! @brief      フォントのアセントを取得します。
    //!
    //! @return     ピクセル単位でのフォントのアセントを返します。
    //!
    virtual int             GetAscent() const;

    //! @brief      フォントのディセントを取得します。
    //!
    //! @return     ピクセル単位でのフォントのディセントを返します。
    //!
    virtual int             GetDescent() const;

    //! @brief      フォントに含まれる最大の文字幅を取得します。
    //!
    //! @return     フォントに含まれる最大の文字幅をピクセル単位で返します。
    //!
    virtual int             GetMaxCharWidth() const;

    //! @brief      フォントのタイプを取得します。
    //!
    //! @return     フォントのタイプを返します。
    //!
    virtual font::Font::Type GetType() const;

    //! @brief      フォントのテクスチャフォーマットを取得します。
    //!
    //! @return     グリフデータのテクスチャフォーマットを返します。
    //!
    virtual font::TexFmt    GetTextureFormat() const;

    //! @brief      フォントが持つ改行幅を取得します。
    //!
    //! @return     フォントが持つ改行幅を返します。
    //!
    virtual int             GetLineFeed() const;

    //! @brief      フォントのデフォルト文字幅情報を取得します。
    //!
    //! @return     フォントのデフォルト文字幅情報を返します。
    //!
    virtual const font::CharWidths GetDefaultCharWidths() const;

    //! @brief      フォントがもつ改行幅を設定します。
    //!
    //! @param[in]  linefeed  新しい改行幅。
    //!
    virtual void            SetLineFeed(int linefeed);

    //! @brief      フォントのデフォルト文字幅情報を設定します。
    //!
    //! @param[in]  widths  新しいデフォルトと文字幅情報。
    //!
    virtual void            SetDefaultCharWidths(
                                const font::CharWidths& widths
                            );

    //! @brief      フォントの代替文字を変更します。
    //!
    //! @param[in]  c  新しい代替文字の文字コード。
    //!
    //! @return     代替文字の変更に成功したら true 、失敗したら false を返します。@n
    //!             フォント内に文字コード c に対応するグリフが存在しない場合に失敗します。
    //!
    virtual bool            SetAlternateChar(font::CharCode c);

    //! @brief      文字の文字幅を取得します。
    //!
    //! @details
    //! このメソッドは、文字が既にプロットされていたら、その文字の幅を返します。ただし、まだ
    //! 文字がプロットされていない場合の挙動は、初期化時のInitializeArgでどのような設定に
    //! しているかによって異なります。
    //!
    //! isAlternateCharSpaceWithOriginalWidthとisAlternateCharSpaceWithOriginalWidthForNotReadyChar
    //! を何れもfalseにしている場合は、「プロットされていない文字は常に代替文字になる」設定で
    //! あるため、描画するものとGetCharWidthの結果が一致するように、GetCharWidthはプロットされて
    //! いない文字については代替文字の幅を返します。
    //!
    //! それに対して、「プロットされていない文字は常に元々表示するはずだった文字と同じ幅の空白
    //! として扱う」か、もしくは「プロットされていない文字は、それがフォント内になければ代替文字にし、
    //! フォントにあれば元々表示するはずだった文字と同じ幅の空白として扱う」設定だった場合は、
    //! GetCharWidthはプロットされていない文字についてはスケーラブルフォントエンジンから直接文字幅を
    //! 取得し、代替文字ではなくその文字の幅を返します。
    //!
    //! スケーラブルフォントエンジンから幅を取得する場合、テクスチャキャッシュ内にある場合より
    //! 遅くなります。
    //! WiiU実機で、一文字につき5マイクロ秒程度かかりますのでご注意ください。
    //!
    //! @param[in]  c  幅を求める文字の文字コード。
    //!
    //! @return     ピクセル単位での文字幅を返します。
    //!
    virtual int             GetCharWidth(font::CharCode c) const;

    //! @brief      文字の文字幅情報を取得します。
    //!
    //! @details
    //! 引数で指定した文字がまだプロットされていない場合の挙動は @ref GetCharWidth と同じです。
    //!
    //! @param[in]  c  文字幅情報を取得する文字の文字コード。
    //!
    //! @return         文字幅情報を返します。
    //!
    virtual const font::CharWidths
                            GetCharWidths(font::CharCode c) const;

    //! @brief      グリフデータを取得します。
    //!
    //! @param[out] pGlyph  グリフデータを受け取るバッファへのポインタ
    //! @param[in]  c       グリフデータを取得する文字の文字コード
    //!
    virtual void            GetGlyph(
                                font::Glyph*      pGlyph,
                                font::CharCode    c
                            ) const;

    //! @brief      グリフデータの有無を取得します。
    //!
    //! @param[in]  c  グリフデータの有無を取得する文字の文字コード
    //!
    //! @return     グリフデータが存在する場合は true 、存在しない場合は false を返します。
    //!
    virtual bool            HasGlyph(
                                font::CharCode    c
                            ) const;

    //! @brief      カーニングの値を取得します。
    //!
    //! @param[in]  c0  カーニングの値を取得する、進行方向と逆側の文字
    //! @param[in]  c1  カーニングの値を取得する、進行方向側の文字
    //!
    //! @return     カーニングの値を返します。
    //!
    virtual int            GetKerning(font::CharCode c0, font::CharCode c1) const;

    //! @brief      フォントが対応している文字コードを取得します。
    //!
    //! @return     フォントが対応している文字コードを返します。
    //!
    virtual font::CharacterCode GetCharacterCode() const;

    //! @brief      ベースライン位置を取得します。
    //!
    //! @return     セルの上端からベースラインまでのピクセル数を返します。
    //!
    virtual int             GetBaselinePos() const;

    //! セルの高さを取得します。

    //! @brief      セルの高さを取得します。
    //!
    //! @return     ピクセル単位でのセルの高さを返します。
    //!
    virtual int             GetCellHeight() const;

    //! @brief      セルの幅を取得します。
    //!
    //! @return     ピクセル単位でのセルの幅を返します。
    //!
    virtual int             GetCellWidth() const;

    //! @brief      文字描画時にテクスチャの線形補間を行うかどうかを指定します。
    //!
    //! @param[in]  atSmall 文字がオリジナルサイズより小さく表示される場合に線形補間を
    //!                     行うのであれば true を指定します。
    //! @param[in]  atLarge 文字がオリジナルサイズより大きく表示される場合に線形補間を
    //!                     行うのであれば true を指定します。
    //!
    virtual void            EnableLinearFilter(
                                bool    atSmall,
                                bool    atLarge
                            );

    //! @brief      文字が小さく表示される場合に線形補間が行われるかを取得します。
    //!
    //! @return     文字がオリジナルサイズより小さく表示される場合に線形補間が行われるのであれば
    //!             true を、行われないのであれば false を返します。
    //!
    virtual bool            IsLinearFilterEnableAtSmall() const;

    //! @brief      文字が大きく表示される場合に線形補間が行われるかを取得します。
    //!
    //! @return     文字がオリジナルサイズより大きく表示される場合に線形補間が行われるのであれば
    //!             true を、行われないのであれば false を返します。
    //!
    virtual bool            IsLinearFilterEnableAtLarge() const;

    //! @brief      テクスチャのラップ、フィルタ設定値を取得します。
    //!
    //! @return     テクスチャのラップ、フィルタ設定値を返します。
    //!
    virtual u32             GetTextureWrapFilterValue() const;

    //! @brief        カラーの白黒補間の影響の有無を取得します。
    //!
    //! @return       常にtrueを返します。
    //!
    virtual bool            IsColorBlackWhiteInterpolationEnabled() const;

    //! @brief        カラーの白黒補間の影響の有無を設定します。
    //!
    //! @details
    //! このクラスでは、白黒補完は常に有効であるため、このメソッドは何もしません。
    //!
    //! @param[in] flag true を設定すると、白黒補間の影響を受けます。
    //!
    virtual void            SetColorBlackWhiteInterpolationEnabled(bool flag);

    //! @brief        縁取り効果の有無を取得します。
    //!
    //! @return       縁取り効果が有効な場合は、true を返します。
    //!
    virtual bool            IsBorderEffectEnabled() const;
    //@}

protected:
    enum {
        CHARCODE_ALTERNATE_WHITESPACE = 0x3000
    };

    enum AlternateCharMode {
        USE_WHITESPACE,
        USE_WHITESPACE_FOR_NOT_READY_CHAR,
        USE_ALTERNATE_CHAR
    };

    TextureCache* m_TextureCache;
    u32 m_FontSize;
    int m_FontHeight;
    int m_FontAscent;
    int m_FontBaselinePos;
    int m_FontLineFeed;
    font::CharWidths m_FontDefaultCharWidth;
    u8 m_AlternateCharMode;
    u16 m_FontFace;
    font::CharCode m_FontAlternateChar;

};

} // namespace nw::scfont
} // namespace nw

#endif // NW_SCFONT_SCALABLE_FONT_H_
