﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo/HAL Laboratory, Inc. All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nw/math/math_Vector3.h>

namespace nw {
namespace math {

namespace internal { namespace standard {

NW_MATH_INLINE VEC4*
VEC3Transform(VEC4* pOut, const MTX44* pM, const VEC3* pV)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    VEC4 tmp;
    tmp.x = pM->f._00 * pV->x + pM->f._01 * pV->y + pM->f._02 * pV->z + pM->f._03;
    tmp.y = pM->f._10 * pV->x + pM->f._11 * pV->y + pM->f._12 * pV->z + pM->f._13;
    tmp.z = pM->f._20 * pV->x + pM->f._21 * pV->y + pM->f._22 * pV->z + pM->f._23;
    tmp.w = pM->f._30 * pV->x + pM->f._31 * pV->y + pM->f._32 * pV->z + pM->f._33;

    pOut->x = tmp.x;
    pOut->y = tmp.y;
    pOut->z = tmp.z;
    pOut->w = tmp.w;

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44Add(MTX44* pOut, const MTX44* p1, const MTX44* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->f._00 = p1->f._00 + p2->f._00;
    pOut->f._01 = p1->f._01 + p2->f._01;
    pOut->f._02 = p1->f._02 + p2->f._02;
    pOut->f._03 = p1->f._03 + p2->f._03;

    pOut->f._10 = p1->f._10 + p2->f._10;
    pOut->f._11 = p1->f._11 + p2->f._11;
    pOut->f._12 = p1->f._12 + p2->f._12;
    pOut->f._13 = p1->f._13 + p2->f._13;

    pOut->f._20 = p1->f._20 + p2->f._20;
    pOut->f._21 = p1->f._21 + p2->f._21;
    pOut->f._22 = p1->f._22 + p2->f._22;
    pOut->f._23 = p1->f._23 + p2->f._23;

    pOut->f._30 = p1->f._30 + p2->f._30;
    pOut->f._31 = p1->f._31 + p2->f._31;
    pOut->f._32 = p1->f._32 + p2->f._32;
    pOut->f._33 = p1->f._33 + p2->f._33;

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44Sub(MTX44* pOut, const MTX44* p1, const MTX44* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->f._00 = p1->f._00 - p2->f._00;
    pOut->f._01 = p1->f._01 - p2->f._01;
    pOut->f._02 = p1->f._02 - p2->f._02;
    pOut->f._03 = p1->f._03 - p2->f._03;

    pOut->f._10 = p1->f._10 - p2->f._10;
    pOut->f._11 = p1->f._11 - p2->f._11;
    pOut->f._12 = p1->f._12 - p2->f._12;
    pOut->f._13 = p1->f._13 - p2->f._13;

    pOut->f._20 = p1->f._20 - p2->f._20;
    pOut->f._21 = p1->f._21 - p2->f._21;
    pOut->f._22 = p1->f._22 - p2->f._22;
    pOut->f._23 = p1->f._23 - p2->f._23;

    pOut->f._30 = p1->f._30 - p2->f._30;
    pOut->f._31 = p1->f._31 - p2->f._31;
    pOut->f._32 = p1->f._32 - p2->f._32;
    pOut->f._33 = p1->f._33 - p2->f._33;

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44Mult(MTX44* pOut, const MTX44* p, f32 f)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    pOut->f._00 = p->f._00 * f;
    pOut->f._01 = p->f._01 * f;
    pOut->f._02 = p->f._02 * f;
    pOut->f._03 = p->f._03 * f;

    pOut->f._10 = p->f._10 * f;
    pOut->f._11 = p->f._11 * f;
    pOut->f._12 = p->f._12 * f;
    pOut->f._13 = p->f._13 * f;

    pOut->f._20 = p->f._20 * f;
    pOut->f._21 = p->f._21 * f;
    pOut->f._22 = p->f._22 * f;
    pOut->f._23 = p->f._23 * f;

    pOut->f._30 = p->f._30 * f;
    pOut->f._31 = p->f._31 * f;
    pOut->f._32 = p->f._32 * f;
    pOut->f._33 = p->f._33 * f;

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44Mult(MTX44* pOut, const MTX44* __restrict p1, const MTX44* __restrict p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    MTX44 mTmp;

    MTX44* __restrict pDst = ( pOut == p1 || pOut == p2 ) ? &mTmp : pOut;

    pDst->f._00 = p1->f._00 * p2->f._00 + p1->f._01 * p2->f._10 + p1->f._02 * p2->f._20 + p1->f._03 * p2->f._30;
    pDst->f._01 = p1->f._00 * p2->f._01 + p1->f._01 * p2->f._11 + p1->f._02 * p2->f._21 + p1->f._03 * p2->f._31;
    pDst->f._02 = p1->f._00 * p2->f._02 + p1->f._01 * p2->f._12 + p1->f._02 * p2->f._22 + p1->f._03 * p2->f._32;
    pDst->f._03 = p1->f._00 * p2->f._03 + p1->f._01 * p2->f._13 + p1->f._02 * p2->f._23 + p1->f._03 * p2->f._33;

    pDst->f._10 = p1->f._10 * p2->f._00 + p1->f._11 * p2->f._10 + p1->f._12 * p2->f._20 + p1->f._13 * p2->f._30;
    pDst->f._11 = p1->f._10 * p2->f._01 + p1->f._11 * p2->f._11 + p1->f._12 * p2->f._21 + p1->f._13 * p2->f._31;
    pDst->f._12 = p1->f._10 * p2->f._02 + p1->f._11 * p2->f._12 + p1->f._12 * p2->f._22 + p1->f._13 * p2->f._32;
    pDst->f._13 = p1->f._10 * p2->f._03 + p1->f._11 * p2->f._13 + p1->f._12 * p2->f._23 + p1->f._13 * p2->f._33;

    pDst->f._20 = p1->f._20 * p2->f._00 + p1->f._21 * p2->f._10 + p1->f._22 * p2->f._20 + p1->f._23 * p2->f._30;
    pDst->f._21 = p1->f._20 * p2->f._01 + p1->f._21 * p2->f._11 + p1->f._22 * p2->f._21 + p1->f._23 * p2->f._31;
    pDst->f._22 = p1->f._20 * p2->f._02 + p1->f._21 * p2->f._12 + p1->f._22 * p2->f._22 + p1->f._23 * p2->f._32;
    pDst->f._23 = p1->f._20 * p2->f._03 + p1->f._21 * p2->f._13 + p1->f._22 * p2->f._23 + p1->f._23 * p2->f._33;

    pDst->f._30 = p1->f._30 * p2->f._00 + p1->f._31 * p2->f._10 + p1->f._32 * p2->f._20 + p1->f._33 * p2->f._30;
    pDst->f._31 = p1->f._30 * p2->f._01 + p1->f._31 * p2->f._11 + p1->f._32 * p2->f._21 + p1->f._33 * p2->f._31;
    pDst->f._32 = p1->f._30 * p2->f._02 + p1->f._31 * p2->f._12 + p1->f._32 * p2->f._22 + p1->f._33 * p2->f._32;
    pDst->f._33 = p1->f._30 * p2->f._03 + p1->f._31 * p2->f._13 + p1->f._32 * p2->f._23 + p1->f._33 * p2->f._33;

    if ( pDst != pOut )
    {
        nw::math::MTX44Copy( pOut, pDst );
    }

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44MultScale(MTX44* pOut, const MTX44* pM, const VEC3* pS)
{
    // スケール行列を右からかけるバージョン
    pOut->f._00 = pM->f._00 * pS->x;
    pOut->f._10 = pM->f._10 * pS->x;
    pOut->f._20 = pM->f._20 * pS->x;
    pOut->f._30 = pM->f._30 * pS->x;

    pOut->f._01 = pM->f._01 * pS->y;
    pOut->f._11 = pM->f._11 * pS->y;
    pOut->f._21 = pM->f._21 * pS->y;
    pOut->f._31 = pM->f._31 * pS->y;

    pOut->f._02 = pM->f._02 * pS->z;
    pOut->f._12 = pM->f._12 * pS->z;
    pOut->f._22 = pM->f._22 * pS->z;
    pOut->f._32 = pM->f._32 * pS->z;

    if (pOut != pM)
    {
        pOut->f._03 = pM->f._03;
        pOut->f._13 = pM->f._13;
        pOut->f._23 = pM->f._23;
        pOut->f._33 = pM->f._33;
    }

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44MultScale(MTX44* pOut, const VEC3* pS, const MTX44* pM)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pS );
    NW_ASSERT_NOT_NULL( pM );

    const f32 (*const src)[4] = pM->m;
    f32 (*const dst)[4] = pOut->m;

    dst[0][0] = src[0][0] * pS->x;
    dst[0][1] = src[0][1] * pS->x;
    dst[0][2] = src[0][2] * pS->x;
    dst[0][3] = src[0][3] * pS->x;

    dst[1][0] = src[1][0] * pS->y;
    dst[1][1] = src[1][1] * pS->y;
    dst[1][2] = src[1][2] * pS->y;
    dst[1][3] = src[1][3] * pS->y;

    dst[2][0] = src[2][0] * pS->z;
    dst[2][1] = src[2][1] * pS->z;
    dst[2][2] = src[2][2] * pS->z;
    dst[2][3] = src[2][3] * pS->z;

    if (pOut != pM)
    {
        dst[3][0] = src[3][0];
        dst[3][1] = src[3][1];
        dst[3][2] = src[3][2];
        dst[3][3] = src[3][3];
    }

    return pOut;
}

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

NW_MATH_INLINE VEC4*
VEC3Transform(VEC4* pOut, const MTX44* pM, const VEC3* pV)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    const f32x2 m00m01 = __PSQ_LX(pM, offsetof(MTX44, m[0][0]), 0, 0);
    const f32x2 m02m03 = __PSQ_LX(pM, offsetof(MTX44, m[0][2]), 0, 0);
    const f32x2 m10m11 = __PSQ_LX(pM, offsetof(MTX44, m[1][0]), 0, 0);
    const f32x2 m12m13 = __PSQ_LX(pM, offsetof(MTX44, m[1][2]), 0, 0);
    const f32x2 m20m21 = __PSQ_LX(pM, offsetof(MTX44, m[2][0]), 0, 0);
    const f32x2 m22m23 = __PSQ_LX(pM, offsetof(MTX44, m[2][2]), 0, 0);
    const f32x2 m30m31 = __PSQ_LX(pM, offsetof(MTX44, m[3][0]), 0, 0);
    const f32x2 m32m33 = __PSQ_LX(pM, offsetof(MTX44, m[3][2]), 0, 0);

    // { xy zw } = { x y z 1 }
    const f32x2 xy = __PSQ_LX(pV, offsetof(VEC3, x), 0, 0);
    const f32x2 zw = __PSQ_LX(pV, offsetof(VEC3, z), 1, 0);

    f32x2 fp4, fp5, fp6, fp7;

    // fp4 = [ m00 m01 m02, m03 ] . T[ x y z 1 ]
    fp4 = __PS_MUL(m00m01, xy);
    fp4 = __PS_MADD(m02m03, zw, fp4);
    fp4 = __PS_SUM0(fp4, fp4, fp4);

    // fp5 = [ m10 m11 m12, m13 ] . T[ x y z 1 ]
    fp5 = __PS_MUL(m10m11, xy);
    fp5 = __PS_MADD(m12m13, zw, fp5);
    fp5 = __PS_SUM1(fp5, fp4, fp5);

    tof32x2(pOut->x) = fp5;

    // fp6 = [ m20 m21 m22, m23 ] . T[ x y z 1 ]
    fp6 = __PS_MUL(m20m21, xy);
    fp6 = __PS_MADD(m22m23, zw, fp6);
    fp6 = __PS_SUM0(fp6, fp6, fp6);

    // fp7 = [ m30 m31 m32, m33 ] . T[ x y z 1 ]
    fp7 = __PS_MUL(m30m31, xy);
    fp7 = __PS_MADD(m32m33, zw, fp7);
    fp7 = __PS_SUM1(fp7, fp6, fp7);

    tof32x2(pOut->z) = fp7;

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44Add(MTX44* pOut, const MTX44* p1, const MTX44* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    const f32x2 fp0 = __PS_ADD(tof32x2(p1->a[0] ), tof32x2(p2->a[0] ));
    const f32x2 fp1 = __PS_ADD(tof32x2(p1->a[2] ), tof32x2(p2->a[2] ));
    const f32x2 fp2 = __PS_ADD(tof32x2(p1->a[4] ), tof32x2(p2->a[4] ));
    const f32x2 fp3 = __PS_ADD(tof32x2(p1->a[6] ), tof32x2(p2->a[6] ));
    const f32x2 fp4 = __PS_ADD(tof32x2(p1->a[8] ), tof32x2(p2->a[8] ));
    const f32x2 fp5 = __PS_ADD(tof32x2(p1->a[10]), tof32x2(p2->a[10]));
    const f32x2 fp6 = __PS_ADD(tof32x2(p1->a[12]), tof32x2(p2->a[12]));
    const f32x2 fp7 = __PS_ADD(tof32x2(p1->a[14]), tof32x2(p2->a[14]));

    tof32x2(pOut->a[0] ) = fp0;
    tof32x2(pOut->a[2] ) = fp1;
    tof32x2(pOut->a[4] ) = fp2;
    tof32x2(pOut->a[6] ) = fp3;
    tof32x2(pOut->a[8] ) = fp4;
    tof32x2(pOut->a[10]) = fp5;
    tof32x2(pOut->a[12]) = fp6;
    tof32x2(pOut->a[14]) = fp7;

    return pOut;
}


NW_MATH_INLINE MTX44*
MTX44Sub(MTX44* pOut, const MTX44* p1, const MTX44* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    const f32x2 fp0 = __PS_SUB(tof32x2(p1->a[0] ), tof32x2(p2->a[0] ));
    const f32x2 fp1 = __PS_SUB(tof32x2(p1->a[2] ), tof32x2(p2->a[2] ));
    const f32x2 fp2 = __PS_SUB(tof32x2(p1->a[4] ), tof32x2(p2->a[4] ));
    const f32x2 fp3 = __PS_SUB(tof32x2(p1->a[6] ), tof32x2(p2->a[6] ));
    const f32x2 fp4 = __PS_SUB(tof32x2(p1->a[8] ), tof32x2(p2->a[8] ));
    const f32x2 fp5 = __PS_SUB(tof32x2(p1->a[10]), tof32x2(p2->a[10]));
    const f32x2 fp6 = __PS_SUB(tof32x2(p1->a[12]), tof32x2(p2->a[12]));
    const f32x2 fp7 = __PS_SUB(tof32x2(p1->a[14]), tof32x2(p2->a[14]));

    tof32x2(pOut->a[0] ) = fp0;
    tof32x2(pOut->a[2] ) = fp1;
    tof32x2(pOut->a[4] ) = fp2;
    tof32x2(pOut->a[6] ) = fp3;
    tof32x2(pOut->a[8] ) = fp4;
    tof32x2(pOut->a[10]) = fp5;
    tof32x2(pOut->a[12]) = fp6;
    tof32x2(pOut->a[14]) = fp7;

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44Mult(MTX44* pOut, const MTX44* p, f32 f)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    const f32x2 fp0 = __PS_MULS0F(tof32x2(p->a[0] ), f);
    const f32x2 fp1 = __PS_MULS0F(tof32x2(p->a[2] ), f);
    const f32x2 fp2 = __PS_MULS0F(tof32x2(p->a[4] ), f);
    const f32x2 fp3 = __PS_MULS0F(tof32x2(p->a[6] ), f);
    const f32x2 fp4 = __PS_MULS0F(tof32x2(p->a[8] ), f);
    const f32x2 fp5 = __PS_MULS0F(tof32x2(p->a[10]), f);
    const f32x2 fp6 = __PS_MULS0F(tof32x2(p->a[12]), f);
    const f32x2 fp7 = __PS_MULS0F(tof32x2(p->a[14]), f);

    tof32x2(pOut->a[0] ) = fp0;
    tof32x2(pOut->a[2] ) = fp1;
    tof32x2(pOut->a[4] ) = fp2;
    tof32x2(pOut->a[6] ) = fp3;
    tof32x2(pOut->a[8] ) = fp4;
    tof32x2(pOut->a[10]) = fp5;
    tof32x2(pOut->a[12]) = fp6;
    tof32x2(pOut->a[14]) = fp7;

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44Mult(MTX44* pOut, const MTX44* __restrict p1, const MTX44* __restrict p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    // d = a . b;
    //
    // { f0 f4 } = { b00 b01 b02 b03 }
    // { f1 f5 }   { b10 b11 b12 b13 }
    // { f2 f6 }   { b20 b21 b22 b23 }
    // { f3 f7 }   { b30 b31 b32 b33 }

    const f32x2 f0 = __PSQ_LX(p2, offsetof(MTX44, m[0][0]), 0, 0);
    const f32x2 f1 = __PSQ_LX(p2, offsetof(MTX44, m[1][0]), 0, 0);
    const f32x2 f2 = __PSQ_LX(p2, offsetof(MTX44, m[2][0]), 0, 0);
    const f32x2 f3 = __PSQ_LX(p2, offsetof(MTX44, m[3][0]), 0, 0);
    const f32x2 f4 = __PSQ_LX(p2, offsetof(MTX44, m[0][2]), 0, 0);
    const f32x2 f5 = __PSQ_LX(p2, offsetof(MTX44, m[1][2]), 0, 0);
    const f32x2 f6 = __PSQ_LX(p2, offsetof(MTX44, m[2][2]), 0, 0);
    const f32x2 f7 = __PSQ_LX(p2, offsetof(MTX44, m[3][2]), 0, 0);

    f32x2 f8, f9, f10;

    // { d00 d01 d02 d03 } = { f9 f10 } := { a00 a01 a02 a03 } . { f0 f4 }
    //                                                           { f1 f5 }
    //                                                           { f2 f6 }
    //                                                           { f3 f7 }

    f8 = __PSQ_LX(p1, offsetof(MTX44, m[0][0]), 0, 0); // { a00 a01 }
    f9 = __PS_MULS0(f0, f8);
    f10 = __PS_MULS0(f4, f8);
    f9 = __PS_MADDS1(f1, f8, f9);
    f10 = __PS_MADDS1(f5, f8, f10);
    f8 = __PSQ_LX(p1, offsetof(MTX44, m[0][2]), 0, 0); // { a02 a03 }
    f9 = __PS_MADDS0(f2, f8, f9);
    f10 = __PS_MADDS0(f6, f8, f10);
    f9 = __PS_MADDS1(f3, f8, f9);
    f10 = __PS_MADDS1(f7, f8, f10);
    __PSQ_STX(pOut, offsetof(MTX44, m[0][0]), f9, 0, 0);
    __PSQ_STX(pOut, offsetof(MTX44, m[0][2]), f10, 0, 0);

    //                                                           { f0 f4 }
    // { d10 d11 d12 d13 } = { f9 f10 } := { a10 a11 a12 a13 } . { f1 f5 }
    //                                                           { f2 f6 }
    //                                                           { f3 f7 }

    f8 = __PSQ_LX(p1, offsetof(MTX44, m[1][0]), 0, 0); // { a10 a11 }
    f9 = __PS_MULS0(f0, f8);
    f10 = __PS_MULS0(f4, f8);
    f9 = __PS_MADDS1(f1, f8, f9);
    f10 = __PS_MADDS1(f5, f8, f10);
    f8 = __PSQ_LX(p1, offsetof(MTX44, m[1][2]), 0, 0); // { a12 a13 }
    f9 = __PS_MADDS0(f2, f8, f9);
    f10 = __PS_MADDS0(f6, f8, f10);
    f9 = __PS_MADDS1(f3, f8, f9);
    f10 = __PS_MADDS1(f7, f8, f10);
    __PSQ_STX(pOut, offsetof(MTX44, m[1][0]), f9, 0, 0);
    __PSQ_STX(pOut, offsetof(MTX44, m[1][2]), f10, 0, 0);

    //                                                           { f0 f4 }
    //                                                           { f1 f5 }
    // { d20 d21 d22 d23 } = { f9 f10 } := { a20 a21 a22 a23 } . { f2 f6 }
    //                                                           { f3 f7 }

    f8 = __PSQ_LX(p1, offsetof(MTX44, m[2][0]), 0, 0); // { a20 a21 }
    f9 = __PS_MULS0(f0, f8);
    f10 = __PS_MULS0(f4, f8);
    f9 = __PS_MADDS1(f1, f8, f9);
    f10 = __PS_MADDS1(f5, f8, f10);
    f8 = __PSQ_LX(p1, offsetof(MTX44, m[2][2]), 0, 0); // { a22 a23 }
    f9 = __PS_MADDS0(f2, f8, f9);
    f10 = __PS_MADDS0(f6, f8, f10);
    f9 = __PS_MADDS1(f3, f8, f9);
    f10 = __PS_MADDS1(f7, f8, f10);
    __PSQ_STX(pOut, offsetof(MTX44, m[2][0]), f9, 0, 0);
    __PSQ_STX(pOut, offsetof(MTX44, m[2][2]), f10, 0, 0);

    //                                                           { f0 f4 }
    //                                                           { f1 f5 }
    //                                                           { f2 f6 }
    // { d30 d31 d32 d33 } = { f9 f10 } := { a30 a31 a32 a33 } . { f3 f7 }

    f8 = __PSQ_LX(p1, offsetof(MTX44, m[3][0]), 0, 0); // { a30 a31 }
    f9 = __PS_MULS0(f0, f8);
    f10 = __PS_MULS0(f4, f8);
    f9 = __PS_MADDS1(f1, f8, f9);
    f10 = __PS_MADDS1(f5, f8, f10);
    f8 = __PSQ_LX(p1, offsetof(MTX44, m[3][2]), 0, 0); // { a32 a33 }
    f9 = __PS_MADDS0(f2, f8, f9);
    f10 = __PS_MADDS0(f6, f8, f10);
    f9 = __PS_MADDS1(f3, f8, f9);
    f10 = __PS_MADDS1(f7, f8, f10);
    __PSQ_STX(pOut, offsetof(MTX44, m[3][0]), f9, 0, 0);
    __PSQ_STX(pOut, offsetof(MTX44, m[3][2]), f10, 0, 0);

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44MultScale(MTX44* pOut, const MTX44* pM, const VEC3* pS)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pS );

    // スケール行列を右からかけるバージョン
    //
    // d = m . Mscale(s)
    //
    // d = { m00 m01 m02 m03 } . { sx 0  0  0 }
    //     { m10 m11 m12 m13 }   { 0  sy 0  0 }
    //     { m20 m21 m22 m23 }   { 0  0  sz 0 }
    //     { m30 m31 m32 m33 }   { 0  0  0  1 }
    //
    //   = { sx*m00 sy*m01 sz*m02 m03 }
    //     { sx*m10 sy*m11 sz*m12 m13 }
    //     { sx*m20 sy*m21 sz*m22 m23 }
    //     { sx*m30 sy*m31 sz*m32 m33 }

    // xy = { sx, sy }
    f32x2 xy = __PSQ_LX(pS, offsetof(VEC3, x), 0, 0);

    tof32x2(pOut->f._00) = __PS_MUL(tof32x2(pM->f._00), xy);
    tof32x2(pOut->f._10) = __PS_MUL(tof32x2(pM->f._10), xy);
    tof32x2(pOut->f._20) = __PS_MUL(tof32x2(pM->f._20), xy);
    tof32x2(pOut->f._30) = __PS_MUL(tof32x2(pM->f._30), xy);

    // z1 = { sz, 1 }
    f32x2 z1 = __PSQ_LX(pS, offsetof(VEC3, z), 1, 0);

    tof32x2(pOut->f._02) = __PS_MUL(tof32x2(pM->f._02), z1);
    tof32x2(pOut->f._12) = __PS_MUL(tof32x2(pM->f._12), z1);
    tof32x2(pOut->f._22) = __PS_MUL(tof32x2(pM->f._22), z1);
    tof32x2(pOut->f._32) = __PS_MUL(tof32x2(pM->f._32), z1);

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44MultScale(MTX44* pOut, const VEC3* pS, const MTX44* pM)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pS );
    NW_ASSERT_NOT_NULL( pM );

    // スケール行列を左からかけるバージョン
    //
    // d = Mscale(s) . m
    //
    // d = { sx 0  0  0 } . { m00 m01 m02 m03 }
    //     { 0  sy 0  0 }   { m10 m11 m12 m13 }
    //     { 0  0  sz 0 }   { m20 m21 m22 m23 }
    //     { 0  0  0  1 }   { m30 m31 m32 m33 }
    //
    //   = { sx*m00 sx*m01 sx*m02 sx*m03 }
    //     { sy*m10 sy*m11 sy*m12 sy*m13 }
    //     { sz*m20 sz*m21 sz*m22 sz*m23 }
    //     {    m30    m31    m32    m33 }


    f32x2 xy = tof32x2(pS->x);

    tof32x2(pOut->f._00) = __PS_MULS0(tof32x2(pM->f._00), xy);
    tof32x2(pOut->f._02) = __PS_MULS0(tof32x2(pM->f._02), xy);

    tof32x2(pOut->f._10) = __PS_MULS1(tof32x2(pM->f._10), xy);
    tof32x2(pOut->f._12) = __PS_MULS1(tof32x2(pM->f._12), xy);

    f32x2 zz = __PS_FDUP(pS->z);

    tof32x2(pOut->f._20) = __PS_MUL(tof32x2(pM->f._20), zz);
    tof32x2(pOut->f._22) = __PS_MUL(tof32x2(pM->f._22), zz);

    if (pOut != pM)
    {
        tof32x2(pOut->f._30) = tof32x2(pM->f._30);
        tof32x2(pOut->f._32) = tof32x2(pM->f._32);
    }

    return pOut;
}

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//----------------------------------------
//! @name    ユーティリティ
//@{

//---------------------------------------------------------------------------
//! @brief        ベクトルを行列で変換します。ベクトルの 4 要素目を 1 として計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pV と同じベクトルを指していても構いません。
//! @param[in]    pM    変換行列へのポインタ。
//! @param[in]    pV    元となるベクトルへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC3Transform(VEC4* pOut, const MTX44* pM, const VEC3* pV)
{
    return NW_MATH_IMPL_NS::VEC3Transform(pOut, pM, pV);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルを行列で変換します。ベクトルの 4 要素目を 1 として計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pV と同じベクトルを指していても構いません。
//! @param[in]    pM    変換行列へのポインタ。
//! @param[in]    pV    元となるベクトルへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3Transform(VEC3* pOut, const MTX44* pM, const VEC3* pV)
{
    return VEC3Transform(pOut, reinterpret_cast<const MTX34*>(pM), pV);
}

//---------------------------------------------------------------------------
//! @brief        回転行列とみなしてRPYを計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。
//! @param[in]    pM    入力の行列へのポインタです。
//!
//! @return       RPY をラジアン単位で返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3CalcRPY(VEC3* pOut, const MTX44* pM)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(pM);

    f32 tmp = math::FAbs(pM->_20);

    if (1.0f - tmp < math::F_ULP)
    {
        pOut->x = 0.f;
        pOut->y = -math::F_PI / 2.0f * (pM->_20 / tmp);
        pOut->z = math::Atan2Rad(-pM->_01, -pM->_20 * pM->_02);
    }
    else
    {
        pOut->x = math::Atan2Rad(pM->_21, pM->_22);
        pOut->y = math::AsinRad(-pM->_20);
        pOut->z = math::Atan2Rad(pM->_10, pM->_00);
    }

    return pOut;
}

//@}

//---------------------------------------------------------------------------
//        MTX44
//---------------------------------------------------------------------------

//----------------------------------------
//! @name    行列
//@{

//---------------------------------------------------------------------------
//! @brief        零行列を作成します。
//!
//! @param[out]   pOut  零行列を格納するバッファへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44Zero(MTX44* pOut)
{
    pOut->f._00 = pOut->f._01 = pOut->f._02 = pOut->f._03 =
    pOut->f._10 = pOut->f._11 = pOut->f._12 = pOut->f._13 =
    pOut->f._20 = pOut->f._21 = pOut->f._22 = pOut->f._23 =
    pOut->f._30 = pOut->f._31 = pOut->f._32 = pOut->f._33 = 0.f;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        単位行列を作成します。
//!
//! @param[out]   pOut  単位行列を格納するバッファへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44Identity(MTX44* pOut)
{
    NW_ASSERT_NOT_NULL( pOut );

    MTX44Copy(pOut, MTX44::Identity());

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        行列をコピーします。
//!
//! @param[out]   pOut  コピー先の行列へのポインタ。
//! @param[in]    p     コピー元の行列へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44Copy(MTX44* pOut, const MTX44* p)
{
    if (pOut != p)
    {
        *pOut = *p;
    }

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        行列の和を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じ行列を指していても構いません。
//! @param[in]    p1    左辺値へのポインタ。
//! @param[in]    p2    右辺値へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44Add(MTX44* pOut, const MTX44* p1, const MTX44* p2)
{
    return NW_MATH_IMPL_NS::MTX44Add(pOut, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        行列の差を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じ行列を指していても構いません。
//! @param[in]    p1    左辺値へのポインタ。
//! @param[in]    p2    右辺値へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44Sub(MTX44* pOut, const MTX44* p1, const MTX44* p2)
{
    return NW_MATH_IMPL_NS::MTX44Sub(pOut, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        行列のスカラー積を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じ行列を指していても構いません。
//! @param[in]    p     元の行列のポインタ。
//! @param[in]    f     掛ける数
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44Mult(MTX44* pOut, const MTX44* p, f32 f)
{
    return MTX44Mult(pOut, p, f);
}

//---------------------------------------------------------------------------
//! @brief        行列が単位行列かどうか判定します。
//!
//! @param[in]    p  判定対象の行列へのポインタ。
//!
//! @return       p が単位行列であれば true そうでなければ false を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE bool
MTX44IsIdentity(const MTX44* p)
{
    return p->f._00 == 1.f && p->f._01 == 0.f && p->f._02 == 0.f && p->f._03 == 0.f &&
           p->f._10 == 0.f && p->f._11 == 1.f && p->f._12 == 0.f && p->f._13 == 0.f &&
           p->f._20 == 0.f && p->f._21 == 0.f && p->f._22 == 1.f && p->f._23 == 0.f &&
           p->f._30 == 0.f && p->f._31 == 0.f && p->f._32 == 0.f && p->f._33 == 1.f;
}

//---------------------------------------------------------------------------
//! @brief        行列の転置を取得します。
//!
//! @param[in]    pOut  計算結果を受け取るバッファへのポインタ。pSrcと同じ行列を指していても構いません。
//! @param[in]    pSrc  元となる行列へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44Transpose(MTX44* pOut, const MTX44 *pSrc)
{
    MTX44 tmp;
    const MTX44 *pMtx;

    if (pOut != pSrc)
    {
        pMtx = pSrc;
        pOut->f._00 = pSrc->f._00;
        pOut->f._11 = pSrc->f._11;
        pOut->f._22 = pSrc->f._22;
        pOut->f._33 = pSrc->f._33;
    }
    else
    {
        pMtx = &tmp;
        tmp.f._01 = pSrc->f._01;
        tmp.f._02 = pSrc->f._02;
        tmp.f._03 = pSrc->f._03;
        tmp.f._12 = pSrc->f._12;
        tmp.f._13 = pSrc->f._13;
        tmp.f._23 = pSrc->f._23;
    }

    pOut->f._01 = pSrc->f._10;
    pOut->f._02 = pSrc->f._20;
    pOut->f._03 = pSrc->f._30;
    pOut->f._12 = pSrc->f._21;
    pOut->f._13 = pSrc->f._31;
    pOut->f._23 = pSrc->f._32;

    pOut->f._10 = pMtx->f._01;
    pOut->f._20 = pMtx->f._02;
    pOut->f._30 = pMtx->f._03;
    pOut->f._21 = pMtx->f._12;
    pOut->f._31 = pMtx->f._13;
    pOut->f._32 = pMtx->f._23;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        射影行列を視野角とアスペクト比から作成します。
//!
//! @param[out]   pOut    射影行列を格納する行列へのポインタ。
//! @param[in]    fovyRad 縦方向の視野角（Radian）
//! @param[in]    aspect  視野のアスペクト比(幅/高さ)
//! @param[in]    n       ニアクリッピング面までの距離。
//! @param[in]    f       ファークリッピング面までの距離。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44PerspectiveRadNew(MTX44* pOut, f32 fovyRad, f32 aspect, f32 n, f32 f)
{
    NW_ASSERT_NOT_NULL(pOut);

    f32 (*const m)[4] = pOut->m;

    // find the cotangent of half the (YZ) field of view

    const f32 angle = fovyRad * 0.5f;

    const f32 cot = 1.0f / ::std::tanf(angle);

    m[0][0] =  cot / aspect;
    m[0][1] =  0.0f;
    m[0][2] =  0.0f;
    m[0][3] =  0.0f;

    m[1][0] =  0.0f;
    m[1][1] =   cot;
    m[1][2] =  0.0f;
    m[1][3] =  0.0f;

    m[2][0] =  0.0f;
    m[2][1] =  0.0f;

    const f32 tmp = -1.0f / (f - n);
    // (-w, w) に変換
    m[2][2] = (f + n) * tmp;
    m[2][3] = (2 * f * n) * tmp;

    m[3][0] =  0.0f;
    m[3][1] =  0.0f;
    m[3][2] = -1.0f;
    m[3][3] =  0.0f;

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44PerspectiveRadOld(MTX44* pOut, f32 fovyRad, f32 aspect, f32 n, f32 f)
{
    // 不具合修正前の、NW4F 1.4.0 以前の MTX44PerspectiveRad です。
    // 使用する場合は NW_MATH_USE_OLDMTX を定義してください。
    NW_ASSERT_NOT_NULL(pOut);

    f32 (*const m)[4] = pOut->m;

    // find the cotangent of half the (YZ) field of view

    const f32 angle = fovyRad * 0.5f;

    const f32 cot = 1.0f / ::std::tanf(angle);

    m[0][0] =  cot / aspect;
    m[0][1] =  0.0f;
    m[0][2] =  0.0f;
    m[0][3] =  0.0f;

    m[1][0] =  0.0f;
    m[1][1] =   cot;
    m[1][2] =  0.0f;
    m[1][3] =  0.0f;

    m[2][0] =  0.0f;
    m[2][1] =  0.0f;

    const f32 tmp = -1.0f / (f - n);
    m[2][2] = f * tmp;
    m[2][3] = f * n * tmp;

    m[3][0] =  0.0f;
    m[3][1] =  0.0f;
    m[3][2] = -1.0f;
    m[3][3] =  0.0f;

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        射影行列をニアクリッピング面での視錐台を元に作成します。
//!               引数の順序が OpenGL 準拠になっているので注意。
//!
//! @param[out]   pOut  射影行列を格納する行列へのポインタ。
//! @param[in]    l     ニアクリッピング面での視錐台左辺の X 座標
//! @param[in]    r     ニアクリッピング面での視錐台右辺の X 座標
//! @param[in]    b     ニアクリッピング面での視錐台下辺の Y 座標
//! @param[in]    t     ニアクリッピング面での視錐台上辺の Y 座標
//! @param[in]    n     ニアクリッピング面までの距離。
//! @param[in]    f     ファークリッピング面までの距離。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44FrustumNew(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f)
{
    NW_ASSERT_NOT_NULL( pOut );

    // NOTE: Be careful about "l" vs. "1" below!!!

    f32 (*const m)[4] = pOut->m;
    f32 tmp     =  1.0f / (r - l);
    m[0][0] =  (2*n) * tmp;
    m[0][1] =  0.0f;
    m[0][2] =  (r + l) * tmp;
    m[0][3] =  0.0f;

    tmp     =  1.0f / (t - b);
    m[1][0] =  0.0f;
    m[1][1] =  (2*n) * tmp;
    m[1][2] =  (t + b) * tmp;
    m[1][3] =  0.0f;

    m[2][0] =  0.0f;
    m[2][1] =  0.0f;

    tmp = -1.0f / (f - n);

    // (-w, w) に変換
    m[2][2] = (f + n) * tmp;
    m[2][3] = (2 * f * n) * tmp;

    m[3][0] =  0.0f;
    m[3][1] =  0.0f;
    m[3][2] = -1.0f;
    m[3][3] =  0.0f;

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44FrustumOld(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f)
{
    // 不具合修正前の、NW4F 1.4.0 以前の MTX44Frustum です。
    // 使用する場合は NW_MATH_USE_OLDMTX を定義してください。
    NW_ASSERT_NOT_NULL( pOut );

    // NOTE: Be careful about "l" vs. "1" below!!!

    f32 (*const m)[4] = pOut->m;
    f32 tmp     =  1.0f / (r - l);
    m[0][0] =  (2*n) * tmp;
    m[0][1] =  0.0f;
    m[0][2] =  (r + l) * tmp;
    m[0][3] =  0.0f;

    tmp     =  1.0f / (t - b);
    m[1][0] =  0.0f;
    m[1][1] =  (2*n) * tmp;
    m[1][2] =  (t + b) * tmp;
    m[1][3] =  0.0f;

    m[2][0] =  0.0f;
    m[2][1] =  0.0f;

    tmp = -1.0f / (f - n);

    m[2][2] = f * tmp;
    m[2][3] = f * n * tmp;

    m[3][0] =  0.0f;
    m[3][1] =  0.0f;
    m[3][2] = -1.0f;
    m[3][3] =  0.0f;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        正射影行列を作成します。
//!               引数の順序が OpenGL 準拠になっているので注意。
//!
//! @param[out]   pOut  射影行列を格納する行列へのポインタ。
//! @param[in]    l     ニアクリッピング面での視錐台左辺の X 座標
//! @param[in]    r     ニアクリッピング面での視錐台右辺の X 座標
//! @param[in]    b     ニアクリッピング面での視錐台下辺の Y 座標
//! @param[in]    t     ニアクリッピング面での視錐台上辺の Y 座標
//! @param[in]    n     ニアクリッピング面までの距離。
//! @param[in]    f     ファークリッピング面までの距離。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44OrthoNew(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f)
{
    NW_ASSERT_NOT_NULL( pOut );

    // NOTE: Be careful about "l" vs. "1" below!!!

    f32 (*const m)[4] = pOut->m;
    f32 tmp     =  1.0f / (r - l);
    m[0][0] =  2.0f * tmp;
    m[0][1] =  0.0f;
    m[0][2] =  0.0f;
    m[0][3] = -(r + l) * tmp;

    tmp     =  1.0f / (t - b);
    m[1][0] =  0.0f;
    m[1][1] =  2.0f * tmp;
    m[1][2] =  0.0f;
    m[1][3] = -(t + b) * tmp;

    m[2][0] =  0.0f;
    m[2][1] =  0.0f;

    tmp     =  -1.0f / (f - n);

    // (-1, 1) に変換
    m[2][2] = 2.0f * tmp;
    m[2][3] = (f + n) * tmp;

    m[3][0] =  0.0f;
    m[3][1] =  0.0f;
    m[3][2] =  0.0f;
    m[3][3] =  1.0f;

    return pOut;
}

NW_MATH_INLINE MTX44*
MTX44OrthoOld(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f)
{
    // 不具合修正前の、NW4F 1.4.0 以前の MTX44Ortho です。
    // 使用する場合は NW_MATH_USE_OLDMTX を定義してください。
    NW_ASSERT_NOT_NULL( pOut );

    // NOTE: Be careful about "l" vs. "1" below!!!

    f32 (*const m)[4] = pOut->m;
    f32 tmp     =  1.0f / (r - l);
    m[0][0] =  2.0f * tmp;
    m[0][1] =  0.0f;
    m[0][2] =  0.0f;
    m[0][3] = -(r + l) * tmp;

    tmp     =  1.0f / (t - b);
    m[1][0] =  0.0f;
    m[1][1] =  2.0f * tmp;
    m[1][2] =  0.0f;
    m[1][3] = -(t + b) * tmp;

    m[2][0] =  0.0f;
    m[2][1] =  0.0f;

    tmp     =  -1.0f / (f - n);

    m[2][2] = tmp;
    m[2][3] = n * tmp;

    m[3][0] =  0.0f;
    m[3][1] =  0.0f;
    m[3][2] =  0.0f;
    m[3][3] =  1.0f;

    return pOut;
}

namespace {
    //---------------------------------------------------------------------------
    //! @brief  画面方向に向けて射影行列を回転します。
    //!
    //! @param  pOut    回転をおこなう行列へのポインタ
    //! @param  pivot
    //!
    //! @return
    //---------------------------------------------------------------------------
    inline MTX44*
    MTX44Pivot( MTX44* pOut, PivotDirection pivot )
    {
        // TODO: 処理の最適化が必要。

        const f32 PIVOT_ROTATION_SIN_COS[ PIVOT_NUM ][ 2 ] =
        {
        #ifdef NW_PLATFORM_CTR
            { 0.0f,  1.0f }, // NONE
            { -1.0f, 0.0f }, // TO_UP
            { 0.0f, -1.0f }, // TO_RIGHT
            { 1.0f,  0.0f }, // TO_BOTTOM
            { 0.0f,  1.0f }, // TO_LEFT
        #else
            { 0.0f,  1.0f }, // NONE
            { 0.0f,  1.0f }, // TO_UP
            { -1.0f, 0.0f }, // TO_RIGHT
            { 0.0f, -1.0f }, // TO_BOTTOM
            { 1.0f,  0.0f }, // TO_LEFT
        #endif
        };

        if ( pivot == PIVOT_NONE )
        {
            return pOut;
        }

        f32 sin = PIVOT_ROTATION_SIN_COS[ pivot ][ 0 ];
        f32 cos = PIVOT_ROTATION_SIN_COS[ pivot ][ 1 ];

        f32 (*const m)[4] = pOut->m;

        if ( sin == 0.0f )
        {
            m[0][0] = cos * m[0][0];
            m[0][1] = cos * m[0][1];
            m[0][2] = cos * m[0][2];
            m[0][3] = cos * m[0][3];

            m[1][0] = cos * m[1][0];
            m[1][1] = cos * m[1][1];
            m[1][2] = cos * m[1][2];
            m[1][3] = cos * m[1][3];
        }
        else // if ( cos == 0.0f )
        {
            f32 tmp = m[0][0];
            m[0][0] = -sin * m[1][0];
            m[1][0] = sin * tmp;

            tmp = m[0][1];
            m[0][1] = -sin * m[1][1];
            m[1][1] = sin * tmp;

            tmp = m[0][2];
            m[0][2] = -sin * m[1][2];
            m[1][2] = sin * tmp;

            tmp = m[0][3];
            m[0][3] = -sin * m[1][3];
            m[1][3] = sin * tmp;
        }

        return pOut;
    }

} // namespace

//---------------------------------------------------------------------------
//! @brief        画面の回転を掛けた射影行列を作成します。
//!               射影行列をニアクリッピング面での視錐台を元に作成します。
//!
//! @param[out]   pOut  射影行列を格納する行列へのポインタ。
//! @param[in]    l     ニアクリッピング面での視錐台左辺の X 座標
//! @param[in]    r     ニアクリッピング面での視錐台右辺の X 座標
//! @param[in]    b     ニアクリッピング面での視錐台下辺の Y 座標
//! @param[in]    t     ニアクリッピング面での視錐台上辺の Y 座標
//! @param[in]    n     ニアクリッピング面までの距離。
//! @param[in]    f     ファークリッピング面までの距離。
//! @param[in]    pivot 画面の回転方向
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44FrustumPivot(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f, PivotDirection pivot)
{
    nw::math::MTX44Frustum( pOut, l, r, b, t, n, f );
    MTX44Pivot( pOut, pivot );

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        画面の回転を掛けた射影行列を作成します。
//!               正射影行列を作成します。
//!
//! @param[out]   pOut  射影行列を格納する行列へのポインタ。
//! @param[in]    l     ニアクリッピング面での視錐台左辺の X 座標
//! @param[in]    r     ニアクリッピング面での視錐台右辺の X 座標
//! @param[in]    b     ニアクリッピング面での視錐台下辺の Y 座標
//! @param[in]    t     ニアクリッピング面での視錐台上辺の Y 座標
//! @param[in]    n     ニアクリッピング面までの距離。
//! @param[in]    f     ファークリッピング面までの距離。
//! @param[in]    pivot 画面の回転方向
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44OrthoPivot(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f, PivotDirection pivot)
{
    nw::math::MTX44Ortho( pOut, l, r, b, t, n, f );
    MTX44Pivot( pOut, pivot );

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        画面の回転を掛けた射影行列を作成します。
//!               射影行列を視野角とアスペクト比から作成します。
//!
//! @param[out]   pOut    射影行列を格納する行列へのポインタ。
//! @param[in]    fovy    縦方向の視野角（Radian）
//! @param[in]    aspect  視野のアスペクト比(幅/高さ)
//! @param[in]    n       ニアクリッピング面までの距離。
//! @param[in]    f       ファークリッピング面までの距離。
//! @param[in]    pivot   画面の回転方向
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44PerspectivePivotRad(MTX44* pOut, f32 fovy, f32 aspect, f32 n, f32 f, PivotDirection pivot)
{
    nw::math::MTX44PerspectiveRad( pOut, fovy, aspect, n, f );
    MTX44Pivot( pOut, pivot );

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        行列の積を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じ行列を指していても構いません。
//! @param[in]    p1    左辺値へのポインタ。
//! @param[in]    p2    右辺値へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44Mult(MTX44* pOut, const MTX44* __restrict p1, const MTX44* __restrict p2)
{
    return NW_MATH_IMPL_NS::MTX44Mult(pOut, p1, p2);
}


//---------------------------------------------------------------------------
//! @brief        行列の配列に左から行列を掛けます。
//!
//! @param[out]   pOut   計算結果を受け取る配列の先頭へのポインタ。
//! @param[in]    p1     左辺値となる行列へのポインタ。
//! @param[in]    pSrc   右辺値となる行列の配列の先頭へのポインタ
//! @param[in]    count  右辺値なる行列の配列の要素数
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44MultArray(MTX44* pOut, const MTX44* __restrict p1, const MTX44* __restrict pSrc, s32 count)
{
    MTX44* pOutBase = pOut;

    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( pSrc );
    NW_ASSERT( count > 1 );

    for ( s32 i = 0 ; i < count ; i++ )
    {
        nw::math::MTX44Mult(pOut, p1, pSrc);

        pSrc++;
        pOut++;
    }

    return pOutBase;
}

namespace {
    inline void SwapF(f32 &a, f32 &b)
    {
        f32 tmp;
        tmp = a;
        a = b;
        b = tmp;
    }
} // namespace (unnamed)


//---------------------------------------------------------------------------
//! @brief        行列の逆行列を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じ行列を指していても構いません。
//! @param[in]    p     元となる行列へのポインタ。
//!
//! @return       逆行列が存在すれば 1 を、存在しなければ 0 を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE u32
MTX44Inverse(MTX44* pOut, const MTX44* p)
{
    MTX44 mTmp;
    f32 (*src)[4];
    f32 (*inv)[4];
    f32   w;

    NW_ASSERT_NOT_NULL( p );
    NW_ASSERT_NOT_NULL( pOut );

    nw::math::MTX44Copy(&mTmp, p);
    nw::math::MTX44Identity(pOut);

    src = mTmp.m;
    inv = pOut->m;

    for (int i = 0; i < 4; ++i)
    {
        f32 max = 0.0f;
        s32 swp = i;

        // ---- partial pivoting -----
        for(int k = i ; k < 4 ; k++ )
        {
            f32 ftmp;
            ftmp = ::std::fabs(src[k][i]);
            if ( ftmp > max )
            {
                max = ftmp;
                swp = k;
            }
        }

        // check singular matrix
        //(or can't solve inverse matrix with this algorithm)
        if ( max == 0.0f )
        {
            return 0;
        }

        // swap row
        if ( swp != i )
        {
            for (int k = 0; k < 4; k++)
            {
                SwapF(src[i][k], src[swp][k]);
                SwapF(inv[i][k], inv[swp][k]);
            }
        }

        // ---- pivoting end ----

        w = 1.0f / src[i][i];
        for (int j = 0; j < 4; ++j)
        {
            src[i][j] *= w;
            inv[i][j] *= w;
        }

        for (int k = 0; k < 4; ++k )
        {
            if ( k == i )
                continue;

            w = src[k][i];
            for (int j = 0; j < 4; ++j)
            {
                src[k][j] -= src[i][j] * w;
                inv[k][j] -= inv[i][j] * w;
            }
        }
    }

    return 1;
}


//---------------------------------------------------------------------------
//! @brief        指定する軸の周りを回転させる回転行列を作成します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。
//! @param[in]    pAxis  回転軸を指定するベクトルへのポインタ。
//! @param[in]    fRad   ラジアン単位での回転量
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
static NW_MATH_INLINE MTX44*
MTX44RotAxisRad_( MTX44* pOut, const VEC3 *pAxis, f32 fRad )
{
    VEC3 vN;
    f32 s, c;             // sinTheta, cosTheta
    f32 t;                // ( 1 - cosTheta )
    f32 x, y, z;          // x, y, z components of normalized axis
    f32 xSq, ySq, zSq;    // x, y, z squared


    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pAxis );

    f32 (*const m)[4] = pOut->m;

    s = ::std::sinf(fRad);
    c = ::std::cosf(fRad);
    t = 1.0f - c;

    VEC3Normalize( &vN, pAxis );

    x = vN.x;
    y = vN.y;
    z = vN.z;

    xSq = x * x;
    ySq = y * y;
    zSq = z * z;

    m[0][0] = ( t * xSq )   + ( c );
    m[0][1] = ( t * x * y ) - ( s * z );
    m[0][2] = ( t * x * z ) + ( s * y );
    m[0][3] = 0.0f;

    m[1][0] = ( t * x * y ) + ( s * z );
    m[1][1] = ( t * ySq )   + ( c );
    m[1][2] = ( t * y * z ) - ( s * x );
    m[1][3] = 0.0f;

    m[2][0] = ( t * x * z ) - ( s * y );
    m[2][1] = ( t * y * z ) + ( s * x );
    m[2][2] = ( t * zSq )   + ( c );
    m[2][3] = 0.0f;

    m[3][0] = 0.0f;
    m[3][1] = 0.0f;
    m[3][2] = 0.0f;
    m[3][3] = 1.0f;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        指定する軸の周りを回転させる回転行列を作成します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。
//! @param[in]    pAxis  回転軸を指定するベクトルへのポインタ。
//! @param[in]    idx    1 円周を 0x100000000 とする単位での回転量
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44RotAxisIdx(MTX44* pOut, const VEC3* pAxis, u32 idx)
{
    // とりあえずは遅くてもいい
    MTX44RotAxisRad_(pOut, pAxis, NW_MATH_IDX_TO_RAD(idx));

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        回転行列を作成します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。
//! @param[in]    idxX  1 円周を 0x100000000 とする単位での X 軸周りの角度
//! @param[in]    idxY  1 円周を 0x100000000 とする単位での Y 軸周りの角度
//! @param[in]    idxZ  1 円周を 0x100000000 とする単位での Z 軸周りの角度
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44RotXYZIdx(MTX44* pOut, u32 idxX, u32 idxY, u32 idxZ)
{
    f32 sinx, cosx;
    f32 siny, cosy;
    f32 sinz, cosz;
    f32 f1, f2;

    SinCosIdx(&sinx, &cosx, idxX);
    SinCosIdx(&siny, &cosy, idxY);
    SinCosIdx(&sinz, &cosz, idxZ);

    pOut->f._20 = -siny;
    pOut->f._00 = cosz * cosy;
    pOut->f._10 = sinz * cosy;
    pOut->f._21 = cosy * sinx;
    pOut->f._22 = cosy * cosx;

    f1 = cosx * sinz;
    f2 = sinx * cosz;

    pOut->f._01 = f2 * siny - f1;
    pOut->f._12 = f1 * siny - f2;

    f1 = sinx * sinz;
    f2 = cosx * cosz;
    pOut->f._02 = f2 * siny + f1;
    pOut->f._11 = f1 * siny + f2;

    pOut->f._03 = 0.f;
    pOut->f._13 = 0.f;
    pOut->f._23 = 0.f;

    pOut->f._30 = 0.0f;
    pOut->f._31 = 0.0f;
    pOut->f._32 = 0.0f;
    pOut->f._33 = 1.0f;

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        スケール変換用の行列を作成します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pM と同じ行列を指していても構いません。
//! @param[in]    pS    それぞれの軸方向のスケール値が格納されたベクトルへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44Scale(MTX44* pOut, const VEC3* pS)
{
    NW_ASSERT_NOT_NULL( pOut  );
    NW_ASSERT_NOT_NULL( pS  );

    f32 (*const m)[4] = pOut->m;

    m[0][0] = pS->x;    m[0][1] = 0.0f;  m[0][2] = 0.0f;  m[0][3] = 0.0f;
    m[1][0] = 0.0f;     m[1][1] = pS->y; m[1][2] = 0.0f;  m[1][3] = 0.0f;
    m[2][0] = 0.0f;     m[2][1] = 0.0f;  m[2][2] = pS->z; m[2][3] = 0.0f;
    m[3][0] = 0.0f;     m[3][1] = 0.0f;  m[3][2] = 0.0f; m[3][3] = 1.0f;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        行列にスケール変換を適用します。スケール行列を右から掛けます。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pM と同じ行列を指していても構いません。
//! @param[in]    pM    元となる行列へのポインタ。
//! @param[in]    pS    それぞれの軸方向のスケール値が格納されたベクトルへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44MultScale(MTX44* pOut, const MTX44* pM, const VEC3* pS)
{
    return NW_MATH_IMPL_NS::MTX44MultScale(pOut, pM, pS);
}


//---------------------------------------------------------------------------
//! @brief        行列にスケール変換を適用します。スケール行列を左から掛けます。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pM と同じ行列を指していても構いません。
//! @param[in]    pS    それぞれの軸方向のスケール値が格納されたベクトルへのポインタ。
//! @param[in]    pM    元となる行列へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44MultScale(MTX44* pOut, const VEC3* pS, const MTX44* pM)
{
    return NW_MATH_IMPL_NS::MTX44MultScale(pOut, pS, pM);
}


//---------------------------------------------------------------------------
//! @brief        平行移動用の行列を作成します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pM と同じ行列を指していても構いません。
//! @param[in]    pT    それぞれの軸方向の移動量が格納されたベクトルへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44Translate(MTX44* pOut, const VEC3* pT)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pT );

    f32 (*const m)[4] = pOut->m;

    m[0][0] = 1.0f;  m[0][1] = 0.0f;  m[0][2] = 0.0f;  m[0][3] = pT->x;
    m[1][0] = 0.0f;  m[1][1] = 1.0f;  m[1][2] = 0.0f;  m[1][3] = pT->y;
    m[2][0] = 0.0f;  m[2][1] = 0.0f;  m[2][2] = 1.0f;  m[2][3] = pT->z;
    m[3][0] = 0.0f;  m[3][1] = 0.0f;  m[3][2] = 0.0f;  m[3][3] = 1.0f;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        行列に平行移動を適用します。移動行列を左から掛けます。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pM と同じ行列を指していても構いません。
//! @param[in]    pT    それぞれの軸方向の移動量が格納されたベクトルへのポインタ。
//! @param[in]    pM    元となる行列へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44MultTranslate(MTX44* pOut, const VEC3* pT, const MTX44* pM)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(pT);
    NW_ASSERT_NOT_NULL(pM);

    const f32 (*const src)[4] = pM->m;
    f32 (*const dst)[4] = pOut->m;

    if ( src != dst )
    {
        dst[0][0] = src[0][0];    dst[0][1] = src[0][1];    dst[0][2] = src[0][2];
        dst[1][0] = src[1][0];    dst[1][1] = src[1][1];    dst[1][2] = src[1][2];
        dst[2][0] = src[2][0];    dst[2][1] = src[2][1];    dst[2][2] = src[2][2];
    }

    dst[0][3] = src[0][3] + pT->x;
    dst[1][3] = src[1][3] + pT->y;
    dst[2][3] = src[2][3] + pT->z;

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        行列に平行移動を適用します。移動行列を右から掛けます。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pM と同じ行列を指していても構いません。
//! @param[in]    pM    元となる行列へのポインタ。
//! @param[in]    pT    それぞれの軸方向の移動量が格納されたベクトルへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44*
MTX44MultTranslate(MTX44* pOut, const MTX44* pM, const VEC3* pT)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(pT);
    NW_ASSERT_NOT_NULL(pM);

    // pOut = pM * pT
    if (pOut != pM)
    {
        (void)MTX44Copy(pOut, pM);
    }

    VEC4 tmp;
    VEC3Transform(&tmp, pM, pT);

    pOut->f._03 = tmp.x;
    pOut->f._13 = tmp.y;
    pOut->f._23 = tmp.z;

    return pOut;
}

//@}

}  // namespace math
}  // namespace nw
