﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_TYPES_H_
#define NW_LYT_TYPES_H_

#include <cstddef>
#include <nw/types.h>

#include <nw/math/math_Types.h>
#include <nw/ut/ut_BinaryFileFormat.h>
#include <nw/ut/ut_LinkList.h>
#include <nw/font/font_DrawerTypes.h>

#if defined(NW_PLATFORM_CAFE)
#include <cafe/gx2.h>
#endif

namespace nw
{
namespace lyt
{

//! ペイン名などのリソースの名前文字列の最大文字数です。
const   int     ResourceNameStrMax  = 24;
//! マテリアルの名前文字列の最大文字数です。
const   int     MaterialNameStrMax  = ResourceNameStrMax + 4;
//! アニメタグの最大文字数です。
const   int     AnimTagNameStrMax   = 64;
//! ユーザーデータの名前文字列の最大文字数です。
const   int     UserDataStrMax      = 8;
//! テクスチャの最大数です。
const   int     TexMapMax           = 3;
//! TEVステージの最大数です。
const   int     TevStageMax         = 6;
//! マテリアルカラーの最大数です。
const   int     MatColorMax         = 2;
//! フォントリソース名の最大(NULLを含む)です。
const   int     FontNameMax         = 128;
//! テクスチャリソース名の最大(NULLを含む)です。
const   int     TexImageNameMax     = 128;
//! シェーダリソース名の最大(NULLを含む)です。
const   int     ShaderNameMax       = 8;
//! レイアウト名の最大文字数です。
const   int     LayoutNameStrMax    = 64;
//! グループ名の最大文字数です。
const   int     GroupNameStrMax     = 32;

//! @brief アーカイブリソースに関する定数です。
enum
{
    //! アーカイブリソースの先頭アドレスの必要アライメント数です。
    //!
    //! アーカイブデータ内に含まれているテクスチャデータを、別の場所にコピーせずに
    //! そのままGPUで扱うことができるようにするためにこのアライメントが必要です。
    //! アーカイブ内のファイルのデータアラインメントは、レイアウトアーカイバが
    //! 自動的に行います。
    //!
    //! この数値は、要求される最も大きいアライメントに合わせて定義しています。
    //!
    ARCHIVE_RESOURCE_ALIGNMENT = 8192,
};

namespace internal
{

//! テクスチャユニットの数です。
const   int     TexUnitMax          = 4;
//! TexEnvユニットの数です。
const   int     TexEnvUnitMax       = 6;

template <typename T>
NW_FORCE_INLINE
T
SetBit(
    T       bits,
    int     pos,
    bool    val
)
{
    const T mask = T(~(1 << pos));

    bits &= mask;
    bits |= (val ? 1: 0) << pos;

    return bits;
}

template <typename T>
NW_FORCE_INLINE
void
SetBit(
    T*      pBits,
    int     pos,
    bool    val
)
{
    const T mask = T(~(1 << pos));

    *pBits &= mask;
    *pBits |= (val ? 1: 0) << pos;
}

using nw::ut::internal::TestBit;

template <typename T>
NW_FORCE_INLINE
T
SetBits(
    T       bits,
    int     pos,
    int     len,
    T       val
)
{
    NW_ASSERT(len <= 32);
    const u32 MaxValue = 0xFFFFFFFFU >> (32 - len);
    NW_ASSERT(val <= MaxValue);

    const T mask = T(~(MaxValue << pos));

    bits &= mask;
    bits |= val << pos;

    return bits;
}

template <typename T>
NW_FORCE_INLINE
void
SetBits(
    T*      pBits,
    int     pos,
    int     len,
    T       val
)
{
    NW_ASSERT(len <= 32);
    const u32 MaxValue = 0xFFFFFFFFU >> (32 - len);
    NW_ASSERT(val <= MaxValue);

    const T mask = T(~(MaxValue << pos));

    *pBits &= mask;
    *pBits |= val << pos;
}

using nw::ut::internal::GetBits;

template<typename T>
NW_FORCE_INLINE
const T*
ConvertOffsToPtr(const void* baseAddress, unsigned int offset)
{
    return
        reinterpret_cast<const T*>(
            static_cast<const u8*>(baseAddress) + offset
        );
}

template<typename T>
NW_FORCE_INLINE
T*
ConvertOffsToPtr(void* baseAddress, unsigned int offset)
{
    return
        reinterpret_cast<T*>(
            static_cast<u8*>(baseAddress) + offset
        );
}

} // namespace nw::lyt::internal

//---------------------------------------------------------------------------
// (削除予定)
// ペインの種類です。
//---------------------------------------------------------------------------

//! @brief 内部用機能のため使用禁止です。
enum PaneType
{
    // その他のペインです。
    PANETYPE_OTHER,
    // ヌルペインです。
    PANETYPE_NULL,
    // ピクチャペインです。
    PANETYPE_PICTURE,
    // ウィンドウペインです。
    PANETYPE_WINDOW,
    // テキストボックスペインです。
    PANETYPE_TEXTBOX,
    // バウンディングペインです。
    PANETYPE_BOUNDING,
    // パーツペインです。
    PANETYPE_PARTS,

    // 列挙子の総数です。
    PANETYPE_MAX
};

//---------------------------------------------------------------------------
//! @brief ペインの表示設定に使用します。
//!
//! @date 2010/04/09 PANEFLAG_MTXCONDITION, PANEFLAG_USERMTX を追加しました。
//! @date 2010/03/26 PANEFLAG_USERALLOCATED を追加しました。
//---------------------------------------------------------------------------
enum PaneFlag
{
    //! ペインを表示するかどうかのフラグです。
    PANEFLAG_VISIBLE,
    //! 子のペインに親ペインのアルファ値が影響するかどうかのフラグです。
    PANEFLAG_INFLUENCEDALPHA,
    //! ペインが位置調整処理の対象となるかどうかのフラグです。
    PANEFLAG_LOCATIONADJUST,
    //! ユーザが作成したペインをあらわすフラグです。
    PANEFLAG_USERALLOCATED,
    //! グローバルマトリックスがDirtyか否かをあらわすフラグです。
    PANEFLAG_ISGLOBALMTXDIRTY,
    //! SRT 行列はユーザによって指定された値です（再計算禁止）。
    PANEFLAG_USERMTX,
    //! グローバル行列はユーザによって指定された値です（再計算禁止）。
    PANEFLAG_USERGLOBALMTX,
    //! ビューアで非表示状態になっているか否かです。
    PANEFLAG_VIEWER_INVISIBLE,

    //! 使用するビット数の合計です。
    PANEFLAG_MAX
};

//---------------------------------------------------------------------------
//! @brief ペインの追加の属性です。
//!
//---------------------------------------------------------------------------
enum PaneFlagEx
{
    //! 部品倍率を無視するか否かのフラグです。
    PANEFLAG_EX_IGNORE_PARTS_MAGNIFY,

    //! 部品倍率をかける際に、部品の境界に合わせて大きくなるか否かのフラグです。
    PANEFLAG_EX_PARTS_MAGNIFY_ADJUST_TO_PARTS_BOUND,

    //! 拡張ユーザーデータアニメーションを使用するかどうかのフラグです。
    PANEFLAG_EX_EXTUSERDATA_ANIMATION_ENABLED,

    //! 使用するビット数の合計です。
    PANEFLAG_EX_MAX,
};

//---------------------------------------------------------------------------
//! @brief 表示の水平方向の基準位置に関する列挙子です。
//!
//---------------------------------------------------------------------------
enum HorizontalPosition
{
    //! 中央を基準にします。
    HORIZONTALPOSITION_CENTER,
    //! 左を基準にします。
    HORIZONTALPOSITION_LEFT,
    //! 右を基準にします。
    HORIZONTALPOSITION_RIGHT,

    //! 列挙子の総数です。
    HORIZONTALPOSITION_MAX
};

//---------------------------------------------------------------------------
//! @brief 表示の垂直方向の基準位置に関する列挙子です。
//!
//---------------------------------------------------------------------------
enum VerticalPosition
{
    //! 中央を基準にします。
    VERTICALPOSITION_CENTER,
    //! 上を基準にします。
    VERTICALPOSITION_TOP,
    //! 下を基準にします。
    VERTICALPOSITION_BOTTOM,

    //! 列挙子の総数です。
    VERTICALPOSITION_MAX
};

//---------------------------------------------------------------------------
//! @brief テクスチャフォーマットの列挙子です。
//!
//---------------------------------------------------------------------------
enum TexFormat
{
    //! L8です。
    TEXFORMAT_L8,
    //! A8です。
    TEXFORMAT_A8,
    //! LA4です。
    TEXFORMAT_LA4,
    //! LA8です。
    TEXFORMAT_LA8,
    //! HILO8です。
    TEXFORMAT_HILO8,
    //! RGB565です。
    TEXFORMAT_RGB565,
    //! RGB8です。
    TEXFORMAT_RGB8,
    //! RGB5A1です。
    TEXFORMAT_RGB5A1,
    //! RGBA4です。
    TEXFORMAT_RGBA4,
    //! RGBA8です。
    TEXFORMAT_RGBA8,
    //! ETC1です。
    TEXFORMAT_ETC1,
    //! ETC1A4です。
    TEXFORMAT_ETC1A4,
    //! BC1です。
    TEXFORMAT_BC1,
    //! BC2です。
    TEXFORMAT_BC2,
    //! BC3です。
    TEXFORMAT_BC3,
    //! ルミナンスのBC4です。
    TEXFORMAT_BC4L,
    //! アルファのBC4です。
    TEXFORMAT_BC4A,
    //! BC5です。
    TEXFORMAT_BC5,
    //! L4です。
    TEXFORMAT_L4,
    //! A4です。
    TEXFORMAT_A4,
    //! sRGBフェッチを行うRGBA8です。
    TEXFORMAT_RGBA8_SRGB,
    //! sRGBフェッチを行うBC1です。
    TEXFORMAT_BC1_SRGB,
    //! sRGBフェッチを行うBC2です。
    TEXFORMAT_BC2_SRGB,
    //! sRGBフェッチを行うBC3です。
    TEXFORMAT_BC3_SRGB,
    //! RGBA10_A2です。
    TEXFORMAT_RGB10_A2,
    //! RGB565_INDIRECTです。RとBがインダイレクトのずらし量に、Gがアルファに割り当てられます。
    TEXFORMAT_RGB565_INDIRECT,

    //! 列挙子の総数です。
    TEXFORMAT_MAX
};

//---------------------------------------------------------------------------
//! @brief テクスチャのラップモードを表す列挙子です。
//!
//---------------------------------------------------------------------------
enum TexWrap
{
    //! クランプします。
    TEXWRAP_CLAMP,
    //! 繰り返します。
    TEXWRAP_REPEAT,
    //! ミラーします。
    TEXWRAP_MIRROR,

    //! 列挙子の総数です。
    TEXWRAP_MAX
};

//---------------------------------------------------------------------------
//! @brief テクスチャフィルターのモードを表す列挙子です。
//!
//---------------------------------------------------------------------------
enum TexFilter
{
    //! NEARフィルターモードです。
    TEXFILTER_NEAR,
    //! LINEARフィルターモードです。
    TEXFILTER_LINEAR,

    //! 列挙子の総数です。
    TEXFILTER_MAX
};

//---------------------------------------------------------------------------
//! @brief 投影テクスチャの表示設定に使用します。
//!
//---------------------------------------------------------------------------
enum TexProjectionFlag
{
    //! 投影テクスチャの自動サイズ調整をするかどうかのフラグです。
    TEXPROJECTIONFLAG_FITTINGLAYOUTSIZE,
    //! ペイン全体に投影するかどうかのフラグです。
    TEXPROJECTIONFLAG_FITTINGPANESIZEENABLED,
    //! ペインの回転スケールを影響させるかどうかのフラグです。
    TEXPROJECTIONFLAG_ADJUSTPROJECTIONSRENABLED,

    //! 使用するビット数の合計です。
    TEXPROJECTIONFLAG_MAX,
};

//---------------------------------------------------------------------------
//! @brief Tevの演算モードを表す列挙子です。
//!
//---------------------------------------------------------------------------
enum TevMode
{
    //! REPLACEモードです。
    TEVMODE_REPLACE,
    //! MODULATEモードです。
    TEVMODE_MODULATE,
    //! ADDモードです。
    TEVMODE_ADD,
    //! ADD_SIGNEDモードです。
    TEVMODE_ADD_SIGNED,
    //! INTERPOLATEモードです。
    TEVMODE_INTERPOLATE,
    //! SUBTRACTモードです。
    TEVMODE_SUBTRACT,
    //! ADD_MULTモードです。
    TEVMODE_ADD_MULT,
    //! MULT_ADDモードです。
    TEVMODE_MULT_ADD,
    //! OVERLAYモードです。
    TEVMODE_OVERLAY,
    //! INDIRECTモードです。
    TEVMODE_INDIRECT,
    //! BLEND_INDIRECTモードです。
    TEVMODE_BLEND_INDIRECT,
    //! EACH_INDIRECTモードです。
    TEVMODE_EACH_INDIRECT,

    //! 列挙子の総数です。
    TEVMODE_MAX
};

//---------------------------------------------------------------------------
//! @brief アルファテストのモードを指定する列挙子です。
//!
//---------------------------------------------------------------------------
enum AlphaTest
{
    //! フラグメントは常に受け入れません。
    ALPHATEST_NEVER,
    //! フラグメント・アルファ＜参照アルファのとき受け入れます。
    ALPHATEST_LESS,
    //! フラグメント・アルファ≦参照アルファのとき受け入れます。
    ALPHATEST_LEQUAL,
    //! フラグメント・アルファ＝参照アルファのとき受け入れます。
    ALPHATEST_EQUAL,
    //! フラグメント・アルファ≠参照アルファのとき受け入れます。
    ALPHATEST_NEQUAL,
    //! フラグメント・アルファ≧参照アルファのとき受け入れます。
    ALPHATEST_GEQUAL,
    //! フラグメント・アルファ＞参照アルファのとき受け入れます。
    ALPHATEST_GREATER,
    //! フラグメントは無条件に受け入れます。
    ALPHATEST_ALWAYS,

    //! 列挙子の総数です。
    ALPHATEST_MAX
};

//---------------------------------------------------------------------------
//! @brief ブレンド処理の係数です。
//!
//---------------------------------------------------------------------------
enum BlendFactor
{
    //! (0.0) です。
    BLENDFACTOR_0,
    //! (1.0) です。
    BLENDFACTOR_1,
    //! (ターゲット・カラー) です。
    BLENDFACTOR_DST_COLOR,
    //! (1.0 - ターゲット・カラー) です。
    BLENDFACTOR_INV_DST_COLOR,
    //! (ソース・アルファ) です。
    BLENDFACTOR_SRC_ALPHA,
    //! (1.0 - ソース・アルファ) です。
    BLENDFACTOR_INV_SRC_ALPHA,
    //! (ターゲット・アルファ) です。
    BLENDFACTOR_DST_ALPHA,
    //! (1.0 - ターゲットアルファ) です。
    BLENDFACTOR_INV_DST_ALPHA,
    //! (ソース・カラー) です。
    BLENDFACTOR_SRC_COLOR,
    //! (1.0 - ソース・カラー) です。
    BLENDFACTOR_INV_SRC_COLOR,

    //! 列挙子の総数です。
    BLENDFACTOR_MAX
};

//---------------------------------------------------------------------------
//! @brief ブレンドの演算方法を指定します。
//!
//---------------------------------------------------------------------------
enum BlendOp
{
    //! ブレンド処理は行いません。
    BLENDOP_DISABLE,
    //! 加算します。
    BLENDOP_ADD,
    //! 減算します。
    BLENDOP_SUBTRACT,
    //! 逆方向に減算します。
    BLENDOP_REVERSE_SUBTRACT,
    //! 小さい値を採用します。
    BLENDOP_SELECT_MIN,
    //! 大きい値を採用します。
    BLENDOP_SELECT_MAX,

    //! 列挙子の総数です。
    BLENDOP_MAX
};

//---------------------------------------------------------------------------
//! @brief 論理演算の演算方法を指定します。
//!
//---------------------------------------------------------------------------
enum LogicOp
{
    //! 論理演算は行いません。
    LOGICOP_DISABLE,
    //! 何もしません。
    LOGICOP_NOOP,
    //! 0x00 です。
    LOGICOP_CLEAR,
    //! 0xff です。
    LOGICOP_SET,
    //! (ソース) です。
    LOGICOP_COPY,
    //! (~ソース) です。
    LOGICOP_INV_COPY,
    //! (~ターゲット) です。
    LOGICOP_INV,
    //! (ソース & ターゲット) です。
    LOGICOP_AND,
    //! ~(ソース & ターゲット) です。
    LOGICOP_NAND,
    //! (ソース | ターゲット です。
    LOGICOP_OR,
    //! ~(ソース | ターゲット) です。
    LOGICOP_NOR,
    //! (ソース ^ ターゲット です。
    LOGICOP_XOR,
    //! ~(ソース ^ ターゲット) です。
    LOGICOP_EQUIV,
    //! (ソース & ~ターゲット) です。
    LOGICOP_REV_AND,
    //! (~ソース & ターゲット) です。
    LOGICOP_INV_AND,
    //! (ソース | ~ターゲット) です。
    LOGICOP_REV_OR,
    //! (~ソース | ターゲット) です。
    LOGICOP_INV_OR,

    //! 列挙子の総数です。
    LOGICOP_MAX
};

//---------------------------------------------------------------------------
//! @brief テクスチャ座標の生成方法を指定します。
//!
//---------------------------------------------------------------------------
enum TexGenType
{
    //! 入力属性に関して 2X4 行列の乗算を行い、S, T テクスチャ座標を生成します。
    TEXGENTYPE_MTX2x4,

    //! 列挙子の総数です。
    TEXGENTYPE_MAX
};

//---------------------------------------------------------------------------
//! @brief テクスチャ座標のソースを指定します。
//!
//---------------------------------------------------------------------------
enum TexGenSrc
{
    //! テクスチャ座標 0 を使います。
    TEXGENSRC_TEX0,
    //! テクスチャ座標 1 を使います。
    TEXGENSRC_TEX1,
    //! テクスチャ座標 2 を使います。
    TEXGENSRC_TEX2,
    //! 平行投影を使います。
    TEXGENSRC_ORTHO_PROJECTION,
    //! ペイン基準の平行投影を使います。
    TEXGENSRC_PANE_BASE_ORTHO_PROJECTION,
    //! 透視投影 を使います。
    TEXGENSRC_PERS_PROJECTION,

    //! 列挙子の総数です。
    TEXGENSRC_MAX
};

//---------------------------------------------------------------------------
//! @brief テクスチャの反転表示に関する列挙子です。
//!
//---------------------------------------------------------------------------
enum TextureFlip
{
    //! そのまま表示します。
    TEXTUREFLIP_NONE,
    //! 左右（水平方向）に反転して表示します。
    TEXTUREFLIP_FLIPH,
    //! 上下（垂直方向）に反転して表示します。
    TEXTUREFLIP_FLIPV,
    //! 90度回転して表示します。
    TEXTUREFLIP_ROTATE90,
    //! 180度回転して表示します。
    TEXTUREFLIP_ROTATE180,
    //! 270度回転して表示します。
    TEXTUREFLIP_ROTATE270,

    //! 列挙子の総数です。
    TEXTUREFLIP_MAX
};

//---------------------------------------------------------------------------
//! @brief 原点位置を指定します。
//!
//---------------------------------------------------------------------------
enum ScreenOriginType
{
    //! レイアウトの左上隅を原点とします。
    SCREENORIGINTYPE_CLASSIC,
    //! レイアウトの中心を原点とします。
    SCREENORIGINTYPE_NORMAL,

    //! 列挙子の総数です。
    SCREENORIGINTYPE_MAX
};

//---------------------------------------------------------------------------
//! @brief テキストボックスフラグです。
//!
//---------------------------------------------------------------------------
enum TextBoxFlag
{
    //! カゲを表示するかどうかのフラグです。
    TEXTBOXFLAG_SHADOW_ENABLED,

    //! 「文字列の長さを強制的に指定する」のチェックが入っているか否かのフラグです。
    TEXTBOXFLAG_FORCE_ASSIGN_TEXT_LENGTH,

    //! フチ付きフォントのフチを非表示にするか否かのフラグです。
    TEXTBOXFLAG_INVISIBLE_BORDER_ENABLED,

    //! フチを二度書きするか否かのフラグです。
    TEXTBOXFLAG_DOUBLE_DRAWN_BORDER_ENABLED,

    //! 文字単位のアニメーションをするかどうかのフラグです。
    TEXTBOXFLAG_PER_CHARACTER_TRANSFORM_ENABLED,

    //! 使用するビット数の合計です。
    TEXTBOXFLAG_MAX,
};

//---------------------------------------------------------------------------
//! @brief テキストボックスの部分上書きフラグです。
//!
//---------------------------------------------------------------------------
enum TextBoxOverrideUsageFlag
{
    //! テキスト及びテキストIDを部分的に上書きするか否かのフラグです。
    TEXTBOXOVERRIDEUSAGEFLAG_TEXT_ENABLED,

    //! 使用するビット数の合計です。
    TEXTBOXOVERRIDEUSAGEFLAG_MAX,
};

//---------------------------------------------------------------------------
//! @brief ペイン基本の部分上書きフラグです。
//!
//---------------------------------------------------------------------------
enum BasicOverrideUsageFlag
{
    //! 表示/非表示を上書きするか否かです。
    BASICOVERRIDEUSAGEFLAG_VISIBLE_ENABLED,

    //! 表示/非表示を上書きする場合の値です。
    BASICOVERRIDEUSAGEFLAG_VISIBLE_VALUE,

    //! ユーザーデータを上書きするか否かです。
    BASICOVERRIDEUSAGEFLAG_BASIC_USERDATA_ENABLED,

    //! 位置を上書きするか否かです。
    BASICOVERRIDEUSAGEFLAG_TRANSLATE_ENABLED,

    //! サイズを上書きするか否かです。
    BASICOVERRIDEUSAGEFLAG_SIZE_ENABLED,

    //! スケールを上書きするか否かです。
    BASICOVERRIDEUSAGEFLAG_SCALE_ENABLED,

    //! 回転を上書きするか否かです。
    BASICOVERRIDEUSAGEFLAG_ROTATE_ENABLED,

    //! 透明度を上書きするか否かです。
    BASICOVERRIDEUSAGEFLAG_ALPHA_ENABLED,

    //! 使用するビット数の合計です。
    BASICOVERRIDEUSAGEFLAG_MAX,
};

//---------------------------------------------------------------------------
//! @brief マテリアルの部分上書きフラグです。
//!
//---------------------------------------------------------------------------
enum MaterialOverrideUsageFlag
{
    //! カラー補完を部分的に上書きするか否かのフラグです。
    MATERIALOVERRIDEUSAGEFLAG_INTERPOLATECOLOR_ENABLED,

    //! 使用するビット数の合計です。
    MATERIALOVERRIDEUSAGEFLAG_MAX,
};

//---------------------------------------------------------------------------
//! @brief 拡張ユーザーデータを上書きする際のオフセットの特別な値です。
//!
//---------------------------------------------------------------------------
enum ExtUserDataOverrideOffset
{
    //! 上書きを行いません。
    EXTUSERDATAOVERRIDEOFFSET_NO_OVERRIDE = 0,

    //! 上書きしますが、データがありません。
    EXTUSERDATAOVERRIDEOFFSET_NO_DATA = 1
};

//---------------------------------------------------------------------------
//! @brief TextBox の行そろえ指定です。
//!
//---------------------------------------------------------------------------
enum TextAlignment
{
    //! 水平方向の位置にあわせます。
    TEXTALIGNMENT_SYNCHRONOUS,
    //! 左にそろえます。
    TEXTALIGNMENT_LEFT,
    //! 中央にそろえます。
    TEXTALIGNMENT_CENTER,
    //! 右にそろえます。
    TEXTALIGNMENT_RIGHT,

    //! 列挙子の総数です。
    TEXTALIGNMENT_MAX
};

//---------------------------------------------------------------------------
//! @brief 頂点属性番号です。
//!
//---------------------------------------------------------------------------
enum VertexAttr
{
    //! 頂点座標です。
    VERTEXATTR_POS,
    //! 頂点カラーです。
    VERTEXATTR_COLOR,
    //! テクスチャ座標 0 です。
    VERTEXATTR_TEXCOORD0,
    //! テクスチャ座標 1 です。
    VERTEXATTR_TEXCOORD1,
    //! テクスチャ座標 2 です。
    VERTEXATTR_TEXCOORD2,

    //! 頂点インデックスです。
    VERTEXATTR_VERTEX_INDEX = 0,

    //! 列挙子の総数です。
    VERTEXATTR_MAX
};

//---------------------------------------------------------------------------
//! @brief 頂点属性に要求されるサイズです。
//!
//---------------------------------------------------------------------------
enum VertexAttrSize
{
    //! 頂点座標に要求されるサイズです。
    VERTEXATTRSIZE_POS = 4,
    //! 頂点カラーに要求されるサイズです。
    VERTEXATTRSIZE_COLOR = 4,
    //! テクスチャ座標に要求されるサイズです。
    VERTEXATTRSIZE_TEXCOORD = 2,
    //! 頂点インデックスに要求されるサイズです。
    VERTEXATTRSIZE_INDEX = 2
};

using nw::font::PaneVertex;
using nw::font::VERTEX_LT;
using nw::font::VERTEX_RT;
using nw::font::VERTEX_LB;
using nw::font::VERTEX_RB;
using nw::font::VERTEX_MAX;

//---------------------------------------------------------------------------
//! @brief ペインの四隅に設定された頂点カラーにアクセスするときに使用します。
//!
//---------------------------------------------------------------------------
enum VertexColor
{
    //! 左上隅の頂点カラーです。
    VERTEXCOLOR_LT,
    //! 右上隅の頂点カラーです。
    VERTEXCOLOR_RT,
    //! 左下隅の頂点カラーです。
    VERTEXCOLOR_LB,
    //! 右下隅の頂点カラーです。
    VERTEXCOLOR_RB,

    //! 列挙子の総数です。
    VERTEXCOLOR_MAX
};

//---------------------------------------------------------------------------
//! @brief ウィンドウフラグです。
//!
//---------------------------------------------------------------------------
enum WindowFlag
{
    //! 一つのマテリアルをウィンドウ全体に使用するかどうかのフラグです。
    WINDOWFLAG_USEONEMATERIALFORALL,

    //! 頂点カラーをウィンドウ全体に使用するかどうかのフラグです。
    WINDOWFLAG_USEVTXCOLALL,

    //! ウィンドウ種類のフラグです（1ビット目）。
    WINDOWFLAG_WINDOWKIND0,
    //! ウィンドウ種類のフラグです（2ビット目）。
    WINDOWFLAG_WINDOWKIND1,

    //! コンテンツ領域を描画しないかのフラグです。
    WINDOWFLAG_NOT_DRAW_CONTENT,

    //! 使用するビット数の合計です。
    WINDOWFLAG_MAX,

    //! WINDOWFLAG_WINDOWKINDのビット数です。
    WINDOWFLAG_WINDOWKIND_LENGTH = 2
};

//---------------------------------------------------------------------------
//! @brief ウィンドウの種類を指定するときに使用します。
//!
//---------------------------------------------------------------------------
enum WindowKind
{
    //! 通常のモード（従来挙動）です。
    WINDOWKIND_AROUND,
    //! 水平モードです。
    WINDOWKIND_HORIZONTAL,
    //! 水平モード、コンテンツ無しです。
    WINDOWKIND_HORIZONTAL_NO_CONTENTS,

    //! 列挙子の総数です。
    WINDOWKIND_MAX
};

//---------------------------------------------------------------------------
//! @brief ウィンドウフレームを指定するときに使用します。
//!
//---------------------------------------------------------------------------
enum WindowFrame
{
    //! 左上フレームです。
    WINDOWFRAME_LT,
    //! 右上フレームです。
    WINDOWFRAME_RT,
    //! 左下フレームです。
    WINDOWFRAME_LB,
    //! 右下フレームです。
    WINDOWFRAME_RB,
    //! 左フレームです。
    WINDOWFRAME_L,
    //! 右フレームです。
    WINDOWFRAME_R,
    //! 上フレームです。
    WINDOWFRAME_T,
    //! 下フレームです。
    WINDOWFRAME_B,

    //! 列挙子の総数です。
    WINDOWFRAME_MAX
};

//---------------------------------------------------------------------------
//! @brief ウィンドウ拡大率の固定少数の定義です。
//!
//---------------------------------------------------------------------------
enum WindowInflationFiexdPoint
{
    WINDOWINFL_FP_FRACTIONAL_BIT = 4,
    WINDOWINFL_FP_INTEGER_BIT = 11,
    WINDOWINFL_FP_SIGN_BIT = 1,

    WINDOWINFL_FP_SCALING_FACTOR = 1 << WINDOWINFL_FP_FRACTIONAL_BIT
};

//---------------------------------------------------------------------------
//! @brief 文字色の設定にアクセスするときに使用します。
//!
//---------------------------------------------------------------------------
enum TextColor
{
    //! 上端の文字色です。
    TEXTCOLOR_TOP,
    //! 下端の文字色です。
    TEXTCOLOR_BOTTOM,

    //! 列挙子の総数です。
    TEXTCOLOR_MAX
};

//---------------------------------------------------------------------------
//! @brief 文字色の設定にアクセスするときに使用します。
//!
//---------------------------------------------------------------------------
enum InterpolateColor
{
    //! 黒カラー補間色です。
    INTERPOLATECOLOR_BLACK,
    //! 白カラー補間色です。
    INTERPOLATECOLOR_WHITE,

    //! 列挙子の総数です。
    INTERPOLATECOLOR_MAX
};

NW_STATIC_ASSERT(INTERPOLATECOLOR_MAX <= MatColorMax);

//---------------------------------------------------------------------------
//! @brief 拡張ユーザデータのデータ種別です。
//!
//---------------------------------------------------------------------------
enum ExtUserDataType
{
    //! 文字列です。
    EXTUSERDATATYPE_STRING,
    //! 整数配列です。
    EXTUSERDATATYPE_INT,
    //! 浮動小数点数配列です。
    EXTUSERDATATYPE_FLOAT,

    //! 列挙子の総数です。
    EXTUSERDATATYPE_MAX
};

//---------------------------------------------------------------------------
//! @brief アニメーションの対象を表します。
//!
//! @sa res::AnimationContent
//!
//---------------------------------------------------------------------------
enum AnimContentType
{
    //! ペインを対象としたアニメーションです。
    ANIMCONTENTTYPE_PANE,
    //! マテリアルを対象としたアニメーションです。
    ANIMCONTENTTYPE_MATERIAL,
    //! 拡張ユーザーデータを対象としたアニメーションです。
    ANIMCONTENTTYPE_EXTUSERDATA,

    //! 列挙子の総数です。
    ANIMCONTENTTYPE_MAX
};

//---------------------------------------------------------------------------
//! @brief アニメーションタグのフラグです。
//!
//---------------------------------------------------------------------------
enum AnimTagFlag
{
    //! ペインの子孫ペインもバインドの対象にします。
    ANIMTAGFLAG_DESCENDINGBIND,

    //! 列挙子の総数です。
    ANIMTAGFLAG_MAX
};

//---------------------------------------------------------------------------
//! @brief ペインアニメーションの対象を指定します。
//!
//---------------------------------------------------------------------------
enum AnimTargetPane
{
    //! 表示位置の X 座標です。
    ANIMTARGET_PANE_TRANSLATEX = 0,
    //! 表示位置の Y 座標です。
    ANIMTARGET_PANE_TRANSLATEY,
    //! 表示位置の Z 座標です。
    ANIMTARGET_PANE_TRANSLATEZ,
    //! X 軸の回転量です。
    ANIMTARGET_PANE_ROTATEX,
    //! Y 軸の回転量です。
    ANIMTARGET_PANE_ROTATEY,
    //! Z 軸の回転量です。
    ANIMTARGET_PANE_ROTATEZ,
    //! X 座標方向の拡大率です。
    ANIMTARGET_PANE_SCALEX,
    //! Y 座標方向の拡大率です。
    ANIMTARGET_PANE_SCALEY,
    //! 幅です。
    ANIMTARGET_PANE_SIZEW,
    //! 高さです。
    ANIMTARGET_PANE_SIZEH,

    //! 列挙子の総数です。
    ANIMTARGET_PANE_MAX
};

//---------------------------------------------------------------------------
//! @brief ビジビリティアニメーションの対象を指定します。
//!
//---------------------------------------------------------------------------
enum AnimTargetVisibility
{
    //! ビジビリティです。
    ANIMTARGET_VISIBILITY_VISIBILITY = 0,

    //! 列挙子の総数です。
    ANIMTARGET_VISIBILITY_MAX
};

//---------------------------------------------------------------------------
//! @brief ペインカラーアニメーションの対象を指定します。
//!
//---------------------------------------------------------------------------
enum AnimTargetPaneColor
{
    //! 左上隅の頂点カラーの R 成分値です。
    ANIMTARGET_VERTEXCOLOR_LT_R = 0,
    //! 左上隅の頂点カラーの G 成分値です。
    ANIMTARGET_VERTEXCOLOR_LT_G,
    //! 左上隅の頂点カラーの B 成分値です。
    ANIMTARGET_VERTEXCOLOR_LT_B,
    //! 左上隅の頂点カラーの A 成分値です。
    ANIMTARGET_VERTEXCOLOR_LT_A,
    //! 右上隅の頂点カラーの R 成分値です。
    ANIMTARGET_VERTEXCOLOR_RT_R,
    //! 右上隅の頂点カラーの G 成分値です。
    ANIMTARGET_VERTEXCOLOR_RT_G,
    //! 右上隅の頂点カラーの B 成分値です。
    ANIMTARGET_VERTEXCOLOR_RT_B,
    //! 右上隅の頂点カラーの A 成分値です。
    ANIMTARGET_VERTEXCOLOR_RT_A,
    //! 左下隅の頂点カラーの R 成分値です。
    ANIMTARGET_VERTEXCOLOR_LB_R,
    //! 左下隅の頂点カラーの G 成分値です。
    ANIMTARGET_VERTEXCOLOR_LB_G,
    //! 左下隅の頂点カラーの B 成分値です。
    ANIMTARGET_VERTEXCOLOR_LB_B,
    //! 左下隅の頂点カラーの A 成分値です。
    ANIMTARGET_VERTEXCOLOR_LB_A,
    //! 右下隅の頂点カラーの R 成分値です。
    ANIMTARGET_VERTEXCOLOR_RB_R,
    //! 右下隅の頂点カラーの G 成分値です。
    ANIMTARGET_VERTEXCOLOR_RB_G,
    //! 右下隅の頂点カラーの B 成分値です。
    ANIMTARGET_VERTEXCOLOR_RB_B,
    //! 右下隅の頂点カラーの A 成分値です。
    ANIMTARGET_VERTEXCOLOR_RB_A,

    //! 列挙子の総数です。
    ANIMTARGET_VERTEXCOLOR_MAX,

    //! ペインのアルファ値です。
    ANIMTARGET_PANE_ALPHA = ANIMTARGET_VERTEXCOLOR_MAX,

    //! 列挙子の総数です。
    ANIMTARGET_PANE_COLOR_MAX
};

//---------------------------------------------------------------------------
//! @brief マテリアルカラーアニメーションの対象を指定します。
//!
//---------------------------------------------------------------------------
enum AnimTargetMatColor
{
    //! コンバイナバッファの R 成分です。
    ANIMTARGET_MATCOLOR_BUFFER_R = 0,
    //! コンバイナバッファの G 成分です。
    ANIMTARGET_MATCOLOR_BUFFER_G,
    //! コンバイナバッファの B 成分です。
    ANIMTARGET_MATCOLOR_BUFFER_B,
    //! コンバイナバッファの A 成分です。
    ANIMTARGET_MATCOLOR_BUFFER_A,
    //! コンスタントレジスタ 0 の R 成分です。
    ANIMTARGET_MATCOLOR_KONST0_R,
    //! コンスタントレジスタ 0 の G 成分です。
    ANIMTARGET_MATCOLOR_KONST0_G,
    //! コンスタントレジスタ 0 の B 成分です。
    ANIMTARGET_MATCOLOR_KONST0_B,
    //! コンスタントレジスタ 0 の A 成分です。
    ANIMTARGET_MATCOLOR_KONST0_A,

    //! 列挙子の総数です。
    ANIMTARGET_MATCOLOR_MAX
};
NW_STATIC_ASSERT(ANIMTARGET_MATCOLOR_MAX / 4 == MatColorMax);

//---------------------------------------------------------------------------
//! @brief テクスチャ SRT アニメーションの対象を指定します。
//!
//---------------------------------------------------------------------------
enum AnimTargetTexSRT
{
    //! S 座標の移動量です。
    ANIMTARGET_TEXSRT_TRANSLATES = 0,
    //! T 座標の移動量です。
    ANIMTARGET_TEXSRT_TRANSLATET,
    //! 回転量です。
    ANIMTARGET_TEXSRT_ROTATE,
    //! S 座標方向の拡大率です。
    ANIMTARGET_TEXSRT_SCALES,
    //! T 座標方向の拡大率です。
    ANIMTARGET_TEXSRT_SCALET,

    //! 列挙子の総数です。
    ANIMTARGET_TEXSRT_MAX
};

//---------------------------------------------------------------------------
//! @brief テクスチャパターンアニメーションの対象を指定します。
//!
//---------------------------------------------------------------------------
enum AnimTargetTexPattern
{
    //! イメージによるアニメーションです。
    ANIMTARGET_TEXPATTURN_IMAGE = 0,

    //! 列挙子の総数です。
    ANIMTARGET_TEXPATTURN_MAX
};

//---------------------------------------------------------------------------
//! @brief インダイレクト SRT アニメーションの対象を指定します。(現状、Transには未対応です。)
//!
//---------------------------------------------------------------------------
enum AnimTargetIndirectSRT
{
    //! 回転量です。
    ANIMTARGET_INDIRECTSRT_ROTATE = 0,
    //! S 座標方向の拡大率です。
    ANIMTARGET_INDIRECTSRT_SCALES,
    //! T 座標方向の拡大率です。
    ANIMTARGET_INDIRECTSRT_SCALET,

    //! 列挙子の総数です。
    ANIMTARGET_INDIRECTSRT_MAX
};

//---------------------------------------------------------------------------
//! @brief フォント影アニメーションの対象を指定します。
//!
//---------------------------------------------------------------------------
enum AnimTargetFontShadow
{
    //! 黒補間カラーの R 成分です。
    ANIMTARGET_FONTSHADOW_BLACK_INTERPORATE_COLOR_R = 0,
    //! 黒補間カラーの G 成分です。
    ANIMTARGET_FONTSHADOW_BLACK_INTERPORATE_COLOR_G,
    //! 黒補間カラーの B 成分です。
    ANIMTARGET_FONTSHADOW_BLACK_INTERPORATE_COLOR_B,
    //! 白補間カラーの R 成分です。
    ANIMTARGET_FONTSHADOW_WHITE_INTERPORATE_COLOR_R,
    //! 白補間カラーの G 成分です。
    ANIMTARGET_FONTSHADOW_WHITE_INTERPORATE_COLOR_G,
    //! 白補間カラーの B 成分です。
    ANIMTARGET_FONTSHADOW_WHITE_INTERPORATE_COLOR_B,
    //! 白補間カラーの A 成分です。
    ANIMTARGET_FONTSHADOW_WHITE_INTERPORATE_COLOR_A,

    //! 列挙子の総数です。
    ANIMTARGET_FONTSHADOW_MAX
};

//---------------------------------------------------------------------------
//! @brief 文字単位アニメーションの対象を指定します。
//!
//---------------------------------------------------------------------------
enum AnimTargetPerCharacterTransform
{
    //! オフセットです。
    ANIMTARGET_PERCHARACTERTRANSFORM_EVAL_TYPE_OFFSET = 0,

    //! 列挙子の総数です。
    ANIMTARGET_PERCHARACTERTRANSFORM_MAX
};

//---------------------------------------------------------------------------
//! @brief 文字単位アニメーションカーブの対象を指定します。
//!
//---------------------------------------------------------------------------
enum AnimTargetPerCharacterTransformCurve
{
    //! 平行移動Xです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_TRANSLATION_X = 0,
    //! 平行移動Yです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_TRANSLATION_Y,
    //! 平行移動Zです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_TRANSLATION_Z,
    //! 回転Xです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_ROTATION_X,
    //! 回転Yです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_ROTATION_Y,
    //! 回転Zです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_ROTATION_Z,
    //! 上端カラーRです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LT_R,
    //! 上端カラーGです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LT_G,
    //! 上端カラーBです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LT_B,
    //! 上端カラーAです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LT_A,
    //! 下端カラーRです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LB_R,
    //! 下端カラーGです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LB_G,
    //! 下端カラーBです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LB_B,
    //! 下端カラーAです。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LB_A,

    //! 列挙子の総数です。
    ANIMTARGET_PERCHARACTERTRANSFORMCURVE_MAX
};

//---------------------------------------------------------------------------
//! @brief 文字単位アニメーションのループ種類を指定します。
//!
//---------------------------------------------------------------------------
enum PerCharacterTransformLoopType
{
    //! 一度のみの再生です。
    PERCHARACTERTRANSFORMLOOPTYPE_ONETIME,
    //! ループ再生です。
    PERCHARACTERTRANSFORMLOOPTYPE_LOOP,

    //! 列挙子の総数です。
    PERCHARACTERTRANSFORMLOOPTYPE_MAX
};

//---------------------------------------------------------------------------
//! @brief 文字単位アニメーションの回転中心を指定します。
//!
//---------------------------------------------------------------------------
enum PerCharacterTransformOriginV
{
    //! 中央です。
    PERCHARACTERTRANSFORMORIGINV_CENTER,
    //! 下端です。
    PERCHARACTERTRANSFORMORIGINV_BOTTOM,

    //! 列挙子の総数です。
    PERCHARACTERTRANSFORMORIGINV_MAX
};

//---------------------------------------------------------------------------
//! @brief アルファコンペアアニメーションの対象を指定します。
//!
//---------------------------------------------------------------------------
enum AnimTargetAlphaCompare
{
    //! アルファコンペアの参照値です。
    ANIMTARGET_ALPHACOMPARE_REF = 0,

    //! 列挙子の総数です。
    ANIMTARGET_ALPHACOMPARE_MAX
};

//---------------------------------------------------------------------------
//! @brief アニメーションの計算形式です。
//!
//---------------------------------------------------------------------------
enum AnimCurve
{
    //! コンスタント形式です。
    ANIMCURVE_CONSTANT,
    //! ステップ形式です。
    ANIMCURVE_STEP,
    //! エルミート補間形式です。
    ANIMCURVE_HERMITE,

    //! 列挙子の総数です。
    ANIMCURVE_MAX
};

//---------------------------------------------------------------------------
//! @brief アニメーションの計算を制御するオプションです。
//!
//! @sa Layout::Animate
//! @sa Pane::Animate
//!
//---------------------------------------------------------------------------
enum AnimateOpt
{
    ANIMATEOPT_NOANIMATEINVISIBLE   = 0x0001 //!< 非表示のペインの計算を省略します。
};

//---------------------------------------------------------------------------
//! @brief シェーダのバリエーションを識別するための定義です。
//!
//---------------------------------------------------------------------------
enum ShaderVariation
{
    STANDARD_SHADER_VARIATION,
    WITHOUT_VERTEX_COLOR_SHADER_VARIATION,
    SHADER_VARIATION_QUANTITY
};

//@}

//---------------------------------------------------------------------------
//! @brief ウィンドウペインのフレーム（枠）のサイズを持つ構造体です。
//!
//---------------------------------------------------------------------------
struct WindowFrameSize
{
    f32          l; //!< 左辺の枠の幅です。
    f32          r; //!< 右辺の枠の幅です。
    f32          t; //!< 上辺の枠の高さです。
    f32          b; //!< 下辺の枠の高さです。
};

//---------------------------------------------------------------------------
//! @brief 幅と高さを持つ構造体です。
//!
//---------------------------------------------------------------------------
struct Size
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 幅と高さを 0 に初期化します。
    //!
    Size()
        : width(0), height(0)
    {}

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 指定の幅と高さで初期化します。
    //!
    //! @param[in] aWidth   幅です。
    //! @param[in] aHeight  高さです。
    //!
    Size(f32 aWidth, f32 aHeight)
        : width(aWidth), height(aHeight)
    {}

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 設定をコピーします。
    //!
    //! @param[in] other    コピー元です。
    //!
    Size(const Size& other)
        : width(other.width), height(other.height)
    {}

    //@}

    //! 幅です。
    f32 width;

    //! 高さです。
    f32 height;
};

//---------------------------------------------------------------------------
//! @brief 二つの Size が等しいかどうかを判定します。
//!
//! @param[in] a    比較する Size 構造体のリファレンスです。
//! @param[in] b    比較する Size 構造体のリファレンスです。
//!
//! @return
//! 幅と高さが等しければ true を、等しくなければ false を返します。
//!
//---------------------------------------------------------------------------
inline bool
operator == (const Size& a, const Size& b)
{
    return a.width == b.width && a.height == b.height;
}

//---------------------------------------------------------------------------
//! @brief テクスチャの幅と高さを持つ構造体です。
//!
//---------------------------------------------------------------------------
struct TexSize
{
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 幅と高さを 0 で初期化します。
    //!
    TexSize()
        : width(0), height(0)
    {}

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 指定の幅と高さで初期化します。
    //!
    //! @param[in] aWidth   幅です。
    //! @param[in] aHeight  高さです。
    //!
    TexSize(u16 aWidth, u16 aHeight)
        : width(aWidth), height(aHeight)
    {}

    //! @brief コピーコンストラクタです。
    //!
    //! @param[in] other    コピー元です。
    //!
    TexSize(const TexSize& other)
        : width(other.width), height(other.height)
    {}

    //@}

    //! 幅です。
    u16 width;

    //! 高さです。
    u16 height;
};

//---------------------------------------------------------------------------
//! @brief テクスチャ情報を保持するクラスです。
//!
//! @sa ResourceAccessor
//!
//---------------------------------------------------------------------------
class TextureInfo
{
    NW_DISALLOW_COPY_AND_ASSIGN(TextureInfo);

public:
    //! 無効なテクスチャを表します。
    static const u32 INVALID = 0;

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 無効なテクスチャオブジェクトへのハンドル (0) とサイズ (0, 0) で初期化します。
    //!
    TextureInfo()
        : m_TexObject(INVALID)
    {
    }

    //! @brief コンストラクタです。
    //!
    //! @details
    //! テクスチャイメージの一部を利用する場合に使用します。
    //!
    //! テクスチャイメージはデバイスメモリに置かれなければなりません。
    //!
    //! texObject は通常描画で使用されます。
    //!
    //! @param[in] texObject    テクスチャオブジェクトのハンドルです。
    //! @param[in] size         利用エリアのサイズ (幅, 高さ) です。
    //! @param[in] format       テクスチャのフォーマットです。
    //!
    //! @date 2010/04/23 format 引数を追加しました。
    TextureInfo(
        u32 texObject,
        const TexSize& size,
        TexFormat format)
    {
        this->Set(texObject, size, format);
    }

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief テクスチャの情報を設定します。
    //!
    //! @details
    //! すべての設定をコピーします。
    //!
    //! @param[in] src  コピー元です。
    //!
    void Set(const TextureInfo& src)
    {
        this->Set(
             src.m_TexObject,
             src.m_Size,
             static_cast<TexFormat>(src.m_Format));
    }

    //! @brief テクスチャの情報を設定します。
    //!
    //! @details
    //! 指定の値を設定します。
    //!
    //! テクスチャイメージの一部を利用する場合に使用します。
    //!
    //! テクスチャイメージはデバイスメモリに置かれなければなりません。
    //!
    //! texObject は通常描画で使用されます。
    //!
    //!
    //! @param[in] texObject    テクスチャオブジェクトのハンドルです。
    //! @param[in] size         利用エリアのサイズ (幅, 高さ) です。
    //! @param[in] format       テクスチャのフォーマットです。
    //!
    //! @date 2010/04/23 format 引数を追加しました。
    void Set(
        u32 texObject,
        const TexSize& size,
        TexFormat format
        )
    {
        m_TexObject = texObject;
        m_Size = size;
        m_Format = static_cast<u8>(format);
    }

    //! @brief テクスチャオブジェクトを取得します。
    //!
    //! @return テクスチャオブジェクトへのハンドルを返します。
    //!
    u32 GetTextureObject() const
    {
        return m_TexObject;
    }

    //! @brief テクスチャの利用領域のサイズ (幅, 高さ) を取得します。
    //!
    //! @return テクスチャの利用領域のサイズ (幅, 高さ) を返します。
    //!
    const TexSize& GetSize() const
    {
        return m_Size;
    }

    //! @brief テクスチャフォーマットを取得します。
    //!
    //! @return テクスチャフォーマットを返します。
    //!
    TexFormat GetFormat() const
    {
        return TexFormat(m_Format);
    }

    //! @brief テクスチャが有効か調べます。
    //!
    //! @return テクスチャが有効な場合には true を返します。
    //!
    bool IsValid() const
    {
        return m_TexObject != INVALID;
    }

#if defined(NW_PLATFORM_CAFE)
    //! @brief GX2 のテクスチャを取得します。
    //!
    //! @return テクスチャを返します。
    //!
    GX2Texture& GetTexture() { return m_Texture; }

    //! @brief GX2 のテクスチャを取得します。
    //!
    //! @return テクスチャを返します。
    //!
    const GX2Texture& GetTexture() const { return m_Texture; }
#endif
    //@}

private:
    // value が 2 のべき数であれば true を返す。
    static bool IsPowerOfTwo(u32 value)
    {
        return (value & ~(value - 1)) == value;
    }

    u32 m_TexObject;
    TexSize m_Size;
    u8 m_Format;
#if defined(NW_PLATFORM_CAFE)
    GX2Texture m_Texture;
#endif
};

#if defined(NW_PLATFORM_CAFE)
    struct ShaderConnection;
#endif

//---------------------------------------------------------------------------
//! @brief シェーダ情報を保持するクラスです。
//!
//! @sa ResourceAccessor
//!
//---------------------------------------------------------------------------
class ArchiveShaderInfo
{
    NW_DISALLOW_COPY_AND_ASSIGN(ArchiveShaderInfo);

public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    ArchiveShaderInfo()
    {
#if defined(NW_PLATFORM_CAFE)
        for (int i = 0; i < SHADER_VARIATION_QUANTITY; i++)
        {
            m_Connections[i] = NULL;
        }
#endif
    }

    //@}

#if defined(NW_PLATFORM_CAFE)
    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief シェーダコネクションを設定します。
    //!
    //! @param[in] id           IDです。
    //! @param[in] vertexShader 頂点シェーダです。
    //! @param[in] pixelShader  ピクセルシェーダです。
    //! @param[in] fetchShader  フェッチシェーダです。
    //!
    void SetConnection(u32 id, ShaderConnection* connection)
    {
        m_Connections[id] = connection;
    }

    //! @brief シェーダコネクションを取得します。
    //!
    //! @param[in] id           IDです。
    //!
    //! @return シェーダコネクションを返します。
    //!
    ShaderConnection* GetConnection(u32 id) const { return m_Connections[id]; }
#endif
    //@}

private:
#if defined(NW_PLATFORM_CAFE)
    ShaderConnection* m_Connections[SHADER_VARIATION_QUANTITY];
#endif
};

//! @brief ペインの四隅のテクスチャ座標を保持する配列の定義です。
typedef nw::math::VEC2 TexCoordQuad[VERTEX_MAX];

//! 登録したフォントを表します。
typedef const void* FontKey;

//! 登録したテクスチャを表します。
typedef const void* TextureKey;

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_TYPES_H_
