﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_MATERIALHELPER_H_
#define NW_LYT_MATERIALHELPER_H_

#include <nw/types.h>
#include <nw/lyt/lyt_Resources.h>
#include <nw/lyt/lyt_GraphicsResource.h>
#include <nw/lyt/lyt_Material.h>

#include <nw/gfnd/gfnd_ShaderHelper.h>

namespace nw
{
namespace lyt
{

//----------------------------------------
NW_INLINE f32
ColorU8ToFloat(u8 value)
{
    return 1.0f / 255.0f * value;
}

//----------------------------------------
NW_INLINE void
InitTexSRT(res::TexSRT* texSRTs, u32 num)
{
    for (u32 i = 0; i < num; ++i)
    {
        texSRTs[i].translate = res::Vec2(0.f, 0.f);
        texSRTs[i].rotate    = 0.f;
        texSRTs[i].scale     = res::Vec2(1.f, 1.f);
    }
}

//----------------------------------------
NW_INLINE void
LoadTexCoordMatrix(DrawInfo& drawInfo, int texIdx, f32 (*v)[3])
{
    NW_ASSERT_MINMAXLT(texIdx, 0, TexMapMax);
    namespace shhelp = nw::gfnd::shader_helper;

    // 2 x 4 行列をロード。
    gfnd::ShaderUniformId xzId = drawInfo.GetUniformId(UNIFORM_uTexMtx0_xz + texIdx);
    gfnd::ShaderUniformId ywId = drawInfo.GetUniformId(UNIFORM_uTexMtx0_yw + texIdx);
    shhelp::SetVertexUniformReg(xzId, v[0][0], v[0][1], 0.0f, v[0][2]);
    shhelp::SetVertexUniformReg(ywId, v[1][0], v[1][1], 0.0f, v[1][2]);
}

//----------------------------------------
#if defined(NW_PLATFORM_CAFE)
typedef GX2BlendCombine BlendCombineType;
typedef GX2BlendFunction BlendFunctionType;
typedef GX2LogicOp LogicOpType;

NW_INLINE BlendCombineType
ToBlendCombine(int src)
{
    NW_ASSERT_MINMAXLT(src, 0, BLENDOP_MAX);
    static const BlendCombineType combines[BLENDOP_MAX] =
    {
        GX2_BLEND_COMBINE_LAST, // DISABLE
        GX2_BLEND_COMBINE_ADD,
        GX2_BLEND_COMBINE_SRC_MINUS_DST,
        GX2_BLEND_COMBINE_DST_MINUS_SRC,
        GX2_BLEND_COMBINE_MIN,
        GX2_BLEND_COMBINE_MAX,
    };

    return combines[src];
}

NW_INLINE BlendFunctionType
ToBlendFunction(int src)
{
    NW_ASSERT_MINMAXLT(src, 0, BLENDFACTOR_MAX);
    static const BlendFunctionType functions[BLENDFACTOR_MAX] =
    {
        GX2_BLEND_ZERO,
        GX2_BLEND_ONE,
        GX2_BLEND_DST_COLOR ,
        GX2_BLEND_ONE_MINUS_DST_COLOR,
        GX2_BLEND_SRC_ALPHA,
        GX2_BLEND_ONE_MINUS_SRC_ALPHA,
        GX2_BLEND_DST_ALPHA,
        GX2_BLEND_ONE_MINUS_DST_ALPHA,
        GX2_BLEND_SRC_COLOR ,
        GX2_BLEND_ONE_MINUS_SRC_COLOR,
    };

    return functions[src];
}

NW_INLINE LogicOpType
ToLogicOp(int src)
{
    NW_ASSERT_MINMAXLT(src, 0, LOGICOP_MAX);
    static const LogicOpType operations[LOGICOP_MAX] =
    {
        GX2_LOGIC_OP_NONE,
        GX2_LOGIC_OP_NOOP,
        GX2_LOGIC_OP_CLEAR,
        GX2_LOGIC_OP_SET,
        GX2_LOGIC_OP_COPY,
        GX2_LOGIC_OP_INVCOPY,
        GX2_LOGIC_OP_INV,
        GX2_LOGIC_OP_AND,
        GX2_LOGIC_OP_NAND,
        GX2_LOGIC_OP_OR,
        GX2_LOGIC_OP_NOR,
        GX2_LOGIC_OP_XOR,
        GX2_LOGIC_OP_EQUIV,
        GX2_LOGIC_OP_REVAND,
        GX2_LOGIC_OP_INVAND,
        GX2_LOGIC_OP_REVOR,
        GX2_LOGIC_OP_INVOR,
    };

    return operations[src];
}

#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
typedef GLint BlendCombineType;
typedef GLint BlendFunctionType;
typedef GLint LogicOpType;

NW_INLINE BlendCombineType
ToBlendCombine(int src)
{
    NW_ASSERT_MINMAXLT(src, 0, BLENDOP_MAX);
    static const BlendCombineType combines[BLENDOP_MAX] =
    {
        0, // DISABLE
        GL_FUNC_ADD,
        GL_FUNC_SUBTRACT,
        GL_FUNC_REVERSE_SUBTRACT,
        GL_MIN,
        GL_MAX,
    };

    return combines[src];
}

NW_INLINE BlendFunctionType
ToBlendFunction(int src)
{
    NW_ASSERT_MINMAXLT(src, 0, BLENDFACTOR_MAX);
    static const BlendFunctionType functions[BLENDFACTOR_MAX] =
    {
        GL_ZERO,
        GL_ONE,
        GL_DST_COLOR,
        GL_ONE_MINUS_DST_COLOR,
        GL_SRC_ALPHA,
        GL_ONE_MINUS_SRC_ALPHA,
        GL_DST_ALPHA,
        GL_ONE_MINUS_DST_ALPHA,
        GL_SRC_COLOR,
        GL_ONE_MINUS_SRC_COLOR,
    };

    return functions[src];
}

NW_INLINE LogicOpType
ToLogicOp(int src)
{
    NW_ASSERT_MINMAXLT(src, 0, LOGICOP_MAX);
    static const LogicOpType operations[LOGICOP_MAX] =
    {
        0 , // DISABLE
        GL_NOOP,
        GL_CLEAR,
        GL_SET,
        GL_COPY,
        GL_COPY_INVERTED,
        GL_INVERT,
        GL_AND,
        GL_NAND,
        GL_OR,
        GL_NOR,
        GL_XOR,
        GL_EQUIV,
        GL_AND_REVERSE,
        GL_AND_INVERTED,
        GL_OR_REVERSE,
        GL_OR_INVERTED,
    };

    return operations[src];
}

#endif

} // namespace lyt
} // namespace nw

#endif // NW_LYT_MATERIALHELPER_H_
