﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_MATERIAL_H_
#define NW_LYT_MATERIAL_H_

#include <nw/lyt/lyt_TexMap.h>
#include <nw/lyt/lyt_Resources.h>
#include <nw/lyt/lyt_GraphicsResource.h>
#include <nw/gfnd/gfnd_DisplayList.h>

#include <nw/ut/ut_Color.h>
#include <nw/ut/ut_String.h>
#include <nw/math/math_Types.h>

#if defined(NW_PLATFORM_CAFE)
#include <cafe/gx2.h>
#endif

namespace nw
{
namespace lyt
{
namespace res
{

struct Material;

} // namespace nw::lyt::res

namespace internal
{

// Material Member Numbers
struct MatMemNums
{
    u16 texMap: 2;
    u16 texSRT: 2;
    u16 texCoordGen: 2;
    u16 tevStage: 3;
    u16 alpComp: 1;
    u16 blendMode: 2;
    u16 indirectParameter: 1;
    u16 projectionTexGen: 2;
    u16 fontShadowParameter: 1;
};

//----------------------------------------
NW_INLINE u32
CalcOffsetTexSRTAry(const MatMemNums& bitNums)
{
    return sizeof(TexMap) * bitNums.texMap;
}

//----------------------------------------
NW_INLINE u32
CalcOffsetTexCoordGenAry(const MatMemNums& bitNums)
{
    return CalcOffsetTexSRTAry(bitNums) + sizeof(res::TexSRT) * bitNums.texSRT;
}

//----------------------------------------
NW_INLINE u32
CalcOffsetGetAlphaCompare(const MatMemNums& bitNums)
{
    return CalcOffsetTexCoordGenAry(bitNums) + sizeof(res::TexCoordGen) * bitNums.texCoordGen;
}

//----------------------------------------
NW_INLINE u32
CalcOffsetBlendMode(const MatMemNums& bitNums)
{
    return CalcOffsetGetAlphaCompare(bitNums) + sizeof(res::AlphaCompare) * bitNums.alpComp;
}

//----------------------------------------
NW_INLINE u32
CalcOffsetIndirectParameter(const MatMemNums& bitNums)
{
    return CalcOffsetBlendMode(bitNums) + sizeof(res::BlendMode) * bitNums.blendMode;
}

//----------------------------------------
NW_INLINE u32
CalcOffsetTevStageAry(const MatMemNums& bitNums)
{
    return CalcOffsetIndirectParameter(bitNums) + sizeof(res::IndirectParameter) * bitNums.indirectParameter;
}

//----------------------------------------
NW_INLINE u32
CalcOffsetProjectionTexGen(const MatMemNums& bitNums)
{
    return CalcOffsetTevStageAry(bitNums) + sizeof(res::TevStage) * bitNums.tevStage;
}

//----------------------------------------
NW_INLINE u32
CalcOffsetFontShadowParameter(const MatMemNums& bitNums)
{
    return CalcOffsetProjectionTexGen(bitNums) + sizeof(res::ProjectionTexGenParamaters) * bitNums.projectionTexGen;
}

} // namespace nw::lyt::internal

class AnimTransform;
class AnimResource;
class DrawInfo;
class Layout;
class TexMap;
struct BuildResSet;

//---------------------------------------------------------------------------
//! @brief マテリアルの情報クラスです。
//!
//---------------------------------------------------------------------------
class Material
{
public:
    //! @brief マテリアルが保持する情報のフラグです。
    enum
    {
        //! ユーザーがマテリアルを構築したかを示します。
        FLAG_USER_ALLOCATED,
        //! テクスチャのみを設定します。
        FLAG_TEXTURE_ONLY,
        //! 閾値によるアルファ補間が有効であるかを示します。
        FLAG_THRESHOLDING_ALPHA_INTERPOLATION,
        //! 列挙子の総数です。
        FLAG_MAX
    };

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! デフォルト設定の状態で初期化します。
    //!
    Material();

    //! @brief コンストラクタです。
    //!
    //! @details
    //! データブロックを反映した状態で初期化します。
    //!
    //! @param[in] pBaseRes     マテリアルデータブロックへのポインタです。
    //! @param[in] pOverrideRes 上書き用のマテリアルデータブロックへのポインタです。
    //! @param[in] buildArgSet  BuildArgSet への参照です。
    //!
    Material(
        const res::Material* pBaseRes,
        const res::Material* pOverrideRes,
        const BuildArgSet& buildArgSet);

    //! @brief コピーコンストラクタです。
    //!
    //! @details
    //! 引数で指定したマテリアルをコピーしたマテリアルを作成します。
    //!
    //! マテリアルの元々のレイアウトデータの状態ではなく、そのときの状態でコピー
    //! しますのでご注意ください。つまり、アニメーションを再生した後はそのままの
    //! 状態でコピーします。
    //!
    //! また、アニメーションについては、何もバインドされていない状態になります。
    //!
    //! @param[in] material コピー元のマテリアルです。
    //!
    explicit Material(const Material& material);

    //! @brief デストラクタです。
    //!
    virtual ~Material();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief マテリアルの設定を保持する領域を確保します。
    //!
    //! @param[in] texMapNum 格納可能なテクスチャの数です。
    //! @param[in] texSRTNum 格納可能なテクスチャ SRT 変換行列の数です。
    //! @param[in] texCoordGenNum 格納可能なテクスチャ座標生成の数です。
    //! @param[in] tevStageNum 格納可能な TEV ステージの数です。
    //! @param[in] allocAlpComp アルファコンペアの設定領域を確保するには true を指定します。
    //! @param[in] blendModeNum ブレンドモードの数です。
    //! @param[in] allocIndirectParameter インダイレクトのパラメータの設定領域を確保するには true を指定します。
    //! @param[in] projectionTexGenNum 格納可能な投影テクスチャ座標生成の数です。
    //! @param[in] fontShadowParameter フォント影の補間色の設定領域を確保するには true を指定します。
    //!
    //! @return 領域が確保できなかった場合には false が返ります。
    //!
    bool ReserveMem(
        u8 texMapNum,
        u8 texSRTNum,
        u8 texCoordGenNum,
        u8 tevStageNum = 0,
        bool allocAlpComp = false,
        u8 blendModeNum = 0,
        bool allocIndirectParameter = false,
        u8 projectionTexGenNum = 0,
        bool fontShadowParameter = false);

    //! @brief マテリアルに格納することのできるテクスチャの上限数を取得します。
    //!
    //! @return TexMap を格納できる上限数を返します。
    //!
    //! @sa ReserveMem
    //!
    u8 GetTexMapCap() const
    {
        return u8(m_MemCap.texMap);
    }

    //! @brief マテリアルに格納することのできるテクスチャ SRT 変換行列の上限数を取得します。
    //!
    //! @return テクスチャ SRT 変換行列を格納できる上限数を返します。
    //!
    //! @sa ReserveMem
    //!
    u8 GetTexSRTCap() const
    {
        return u8(m_MemCap.texSRT);
    }

    //! @brief マテリアルに格納することのできるテクスチャ座標生成の上限数を取得します。
    //!
    //! @return テクスチャ座標生成を格納できる上限数を返します。
    //!
    //! @sa ReserveMem
    //!
    u8 GetTexCoordGenCap() const
    {
        return u8(m_MemCap.texCoordGen);
    }

    //! @brief マテリアルに格納することのできる TEV ステージの上限数を取得します。
    //!
    //! @return TEV ステージを格納できる上限数を返します。
    //!
    //! @sa ReserveMem
    //!
    u8 GetTevStageCap() const
    {
        return u8(m_MemCap.tevStage);
    }

    //! @brief マテリアルがアルファコンペアの設定を格納することができるかどうかを取得します。
    //!
    //! @return アルファコンペアの設定を格納できる場合は true を返します。
    //!
    //! @sa ReserveMem
    //!
    bool IsAlphaCompareCap() const
    {
        return m_MemCap.alpComp != 0;
    }

    //! @brief マテリアルがブレンドモード設定を格納することができるかどうかを取得します。
    //!
    //! @return ブレンドモード設定を格納できる場合は true を返します。
    //!
    //! @sa ReserveMem
    //!
    bool IsBlendModeCap() const
    {
        return m_MemCap.blendMode != 0;
    }

    //! @brief マテリアルがアルファのブレンドモード設定を格納することができるかどうかを取得します。
    //!
    //! @return アルファのブレンドモード設定を格納できる場合は true を返します。
    //!
    //! @sa ReserveMem
    //!
    bool IsBlendModeAlphaCap() const
    {
        return m_MemCap.blendMode == 2;
    }

    //! @brief マテリアルがインダイレクトのパラメータを格納することができるかどうかを取得します。
    //!
    //! @return インダイレクトのパラメータを格納できる場合は true を返します。
    //!
    //! @sa ReserveMem
    //!
    bool IsIndirectParameterCap() const
    {
        return m_MemCap.indirectParameter == 1;
    }

    //! @brief マテリアルがフォント影のパラメータを格納できるかどうかを取得します。
    //!
    //! @return フォント影の補間カラーを格納できる場合は true を返します。
    //!
    bool IsFontShadowParameterCap() const
    {
        return m_MemCap.fontShadowParameter == 1;
    }

    //! @brief マテリアルの名前を取得します。
    //!
    //! @return マテリアルの名前を返します。
    //!
    //! @sa SetName
    //!
    const char* GetName() const
    {
        return m_Name;
    }

    //! @brief マテリアルの名前を設定します。
    //!
    //! @details
    //! セットできるマテリアル名文字列の最大長は lyt::MaterialNameStrMax です。
    //!
    //! NW4F1.12.0以前ではMaterialクラスは内部に名前のための文字列バッファを持っていましたが、
    //! このサイズが大きく、キャッシュ効率に悪影響を与えていたため、削除しました。NW4F1.12.0
    //! 以降では、このメソッドで与えたポインタを保持しますので、スタックなどで一時的に確保した
    //! ポインタを与えないようにしてください。
    //!
    //! @param[in] name 設定する名前です。
    //!
    //! @sa GetName
    //!
    void SetName(const char* name);

    //! @brief テクスチャのみ使うかを取得します。
    //!
    //! @return テクスチャのみ使う場合は true を返します。
    //!
    //! @sa SetTextureOnly
    //!
    bool GetTextureOnly() const
    {
        return internal::TestBit(m_Flag, FLAG_TEXTURE_ONLY);
    }

    //! @brief テクスチャのみ使うか設定します。
    //!
    //! @details
    //! テクスチャ以外の設定は以前のマテリアルの設定をそのまま使います。
    //!
    //! @param[in] value    テクスチャのみ使う場合は true を指定します。
    //!
    //! @sa GetTextureOnly
    //!
    void SetTextureOnly(bool value)
    {
        m_Flag = internal::SetBit(m_Flag, FLAG_TEXTURE_ONLY, value);
    }

    //! @brief 閾値によるアルファ補間が有効であるかを取得します。
    //!
    //! @return 閾値によるアルファ補間が有効であれば true を返します。
    //!
    //! @sa SetThresholdingAlphaInterpolation
    //!
    bool GetThresholdingAlphaInterpolation() const
    {
        return internal::TestBit(m_Flag, FLAG_THRESHOLDING_ALPHA_INTERPOLATION);
    }

    //! @brief 閾値によるアルファ補間が有効であるかを設定します。
    //!
    //! @param[in] value 閾値によるアルファ補間を有効にする場合は true を返します。
    //!
    //! @sa GetThresholdingAlphaInterpolation
    //!
    void SetThresholdingAlphaInterpolation(bool value)
    {
        m_Flag = internal::SetBit(m_Flag, FLAG_THRESHOLDING_ALPHA_INTERPOLATION, value);
    }

    //! @brief シェーダIDを取得します。
    //!
    //! @return シェーダID
    //!
    ShaderId GetShaderId() const
    {
        return static_cast<ShaderId>(m_ShaderId);
    }

    //! @brief シェーダIDを設定します。
    //!
    //! @details
    //! シェーダIDは、リソースからMaterialクラスを作ったときやコピーしたときは、
    //! 自動的に適切な値に設定されますので、変更しないようにしてください。
    //!
    //! デフォルト設定の状態でMaterialクラスを作ったときのみこのメソッドを
    //! 使用してください。
    //!
    //! @param[in] id   シェーダID
    //!
    void SetShaderId(ShaderId id)
    {
        m_ShaderId = static_cast<u8>(id);
    }
    //@}

    //----------------------------------------
    //! @name テクスチャ
    //@{

    //! @brief テクスチャの数を取得します。
    //!
    //! @return テクスチャの数を返します。
    //!
    //! @sa SetTexMapNum
    //!
    u8 GetTexMapNum() const
    {
        return u8(m_MemNum.texMap);
    }

    //! @brief テクスチャの数を設定します。
    //!
    //! @param[in] num  テクスチャの数です。
    //!
    //! @sa GetTexMapNum
    //!
    void SetTexMapNum(u8 num);

    //! @brief テクスチャ座標生成の数を取得します。
    //!
    //! @return テクスチャ座標生成の数を返します。
    //!
    //! @sa SetTexCoordGenNum
    //!
    u8 GetTexCoordGenNum() const
    {
        return u8(m_MemNum.texCoordGen);
    }

    //! @brief テクスチャ座標生成の数を設定します。
    //!
    //! @param[in] num  テクスチャ座標生成の数です。
    //!
    //! @sa GetTexCoordGenNum
    //!
    void SetTexCoordGenNum(u8 num);

    //! @brief テクスチャSRTの数を取得します。
    //!
    //! @return テクスチャSRTの数を返します。
    //!
    //! @sa SetSRTNum
    //!
    u8 GetTexSRTNum() const
    {
        return u8(m_MemNum.texSRT);
    }

    //! @brief テクスチャSRTの数を設定します。
    //!
    //! @param[in] num  テクスチャSRTの数です。
    //!
    //! @sa GetTexSRTNum
    //!
    void SetTexSRTNum(u8 num);

    //! @brief テクスチャデータを取得します。
    //!
    //! @details
    //! idx に ReserveMem() または SetTexMapNum() で設定したテクスチャの数よりも
    //! 大きな値を渡した場合はアサートが発生します。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return オブジェクトへの参照を返します。
    //!
    //! @sa SetTexMap
    //! @sa ReserveMem
    //! @sa SetTexMapNum
    //!
    const TexMap& GetTexMap(u32 idx) const
    {
        NW_ASSERTMSG(idx < m_MemNum.texMap, "out of bounds: idx[%u] < m_MemNum.texMap[%u] for material[%s]", idx, m_MemNum.texMap, GetName());
        return GetTexMapAry()[idx];
    }

    //! @brief テクスチャデータを追加します。
    //!
    //! @details
    //! テクスチャデータの数がReserveMem()で設定したテクスチャ数よりも多くなった
    //! 場合はアサートが発生します。
    //!
    //! @param[in] textureInfo  TextureInfo オブジェクトへの参照です。
    //!
    //! @sa GetTexMap
    //! @sa ReserveMem
    //! @sa SetTexMapNum
    //!
    void AppendTexMap(const TextureInfo& textureInfo)
    {
        NW_ASSERTMSG(m_MemNum.texMap < GetTexMapCap(), "out of bounds: m_MemNum.texMap[%u] < GetTexMapCap()[%u] for material[%s]", m_MemNum.texMap, GetTexMapCap(), GetName());
        new (&GetTexMapAry()[m_MemNum.texMap]) TexMap(&textureInfo);
        m_MemNum.texMap += 1;
    }

    //! @brief テクスチャデータを設定します。
    //!
    //! @details
    //! idx に ReserveMem() または SetTexMapNum() で設定したテクスチャ数よりも
    //! 大きな値を渡した場合はアサートが発生します。
    //!
    //! @param[in] idx      インデックスです。
    //! @param[in] value    TextureInfo オブジェクトへの参照です。
    //!
    //! @sa GetTexMap
    //! @sa ReserveMem
    //! @sa SetTexMapNum
    //!
    void SetTexMap(u32 idx, const TextureInfo* value)
    {
        NW_ASSERTMSG(idx < m_MemNum.texMap, "out of bounds: idx[%u] < m_MemNum.texMap[%u] for material[%s]", idx, m_MemNum.texMap, GetName());
        GetTexMapAry()[idx].Set(value);
    }

    //! @brief テクスチャのラップモードを設定します。
    //!
    //! @details
    //! idx に ReserveMem() または SetTexMapNum() で設定したテクスチャ数よりも
    //! 大きな値を渡した場合はアサートが発生します。
    //!
    //! @param[in] idx      インデックスです。
    //! @param[in] wrapS    テクスチャの S 方向のラップモードです。
    //! @param[in] wrapT    テクスチャの T 方向のラップモードです。
    //!
    //! @sa SetTexMap
    //! @sa ReserveMem
    //! @sa SetTexMapNum
    //!
    void SetTexMapWrapMode(u32 idx, TexWrap wrapS, TexWrap wrapT)
    {
        NW_ASSERTMSG(idx < m_MemNum.texMap, "out of bounds: idx[%u] < m_MemNum.texMap[%u] for material[%s]", idx, m_MemNum.texMap, GetName());
        GetTexMapAry()[idx].SetWrapMode(wrapS, wrapT);
    }

    //! @brief テクスチャのフィルタモードを設定します。
    //!
    //! @details
    //! idx に ReserveMem() または SetTexMapNum() で設定したテクスチャ数よりも
    //! 大きな値を渡した場合はアサートが発生します。
    //!
    //! @param[in] idx      インデックスです。
    //! @param[in] minFlt   テクスチャが縮小されるときに適用されるフィルタモードです。
    //! @param[in] magFlt   テクスチャが拡大されるときに適用されるフィルタモードです。
    //!
    //! @sa SetTexMap
    //! @sa ReserveMem
    //! @sa SetTexMapNum
    //!
    void SetTexMapFilter(u32 idx, TexFilter minFlt, TexFilter magFlt)
    {
        NW_ASSERTMSG(idx < m_MemNum.texMap, "out of bounds: idx[%u] < m_MemNum.texMap[%u] for material[%s]", idx, m_MemNum.texMap, GetName());
        GetTexMapAry()[idx].SetFilter(minFlt, magFlt);
    }

    //! @brief テクスチャ SRT 行列を取得します。
    //!
    //! @details
    //! idx に ReserveMem() または SetTexSRTNum() で設定したテクスチャ SRT 行列の数よりも
    //! 大きな値を渡した場合はアサートが発生します。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return オブジェクトへの参照を返します。
    //!
    //! @sa SetTexSRT
    //! @sa ReserveMem
    //! @sa SetTexSRTNum
    //!
    const res::TexSRT& GetTexSRT(u32 idx) const
    {
        NW_ASSERTMSG(idx < m_MemNum.texSRT, "out of bounds: idx[%u] < m_MemNum.texSRT[%u] for material[%s]", idx, m_MemNum.texSRT, GetName());
        return GetTexSRTAry()[idx];
    }

    //! @brief テクスチャ SRT 行列を設定します。
    //!
    //! @details
    //! idx に ReserveMem() または SetTexSRTNum() で設定したテクスチャ SRT 行列の数よりも
    //! 大きな値を渡した場合はアサートが発生します。
    //!
    //! @param[in] idx      インデックスです。
    //! @param[in] value    設定するテクスチャ SRT 行列です。
    //!
    //! @sa GetTexSRT
    //! @sa ReserveMem
    //! @sa SetTexSRTNum
    //!
    void SetTexSRT(u32 idx, const res::TexSRT& value)
    {
        NW_ASSERTMSG(idx < m_MemNum.texSRT, "out of bounds: idx[%u] < m_MemNum.texSRT[%u] for material[%s]", idx, m_MemNum.texSRT, GetName());
        GetTexSRTAry()[idx] = value;
    }

    //! @brief テクスチャ座標生成を取得します。
    //!
    //! @details
    //! idx に ReserveMem() または SetTexCoordGenNum() で設定した
    //! テクスチャ座標生成の数よりも大きな値を渡した場合は
    //! アサートが発生します。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return オブジェクトへの参照を返します。
    //!
    //! @sa SetTexCoordGen
    //! @sa ReserveMem
    //! @sa GetTexCoordGenNum
    //! @sa SetTexCoordGenNum
    //!
    const res::TexCoordGen& GetTexCoordGen(u32 idx) const
    {
        NW_ASSERTMSG(idx < m_MemNum.texCoordGen, "out of bounds: idx[%u] < m_MemNum.texCoordGen[%u] for material[%s]", idx, m_MemNum.texCoordGen, GetName());
        return GetTexCoordGenAry()[idx];
    }

    //! @brief テクスチャ座標生成を設定します。
    //!
    //! @details
    //! idx に ReserveMem() または SetTexCoordGenNum() で設定したテクスチャ座標生成の
    //! 格納可能数よりも大きな値を渡した場合はアサートが発生します。
    //!
    //! @param[in] idx      インデックスです。
    //! @param[in] value    設定するテクスチャ座標生成です。
    //!
    //! @sa GetTexCoordGen
    //! @sa ReserveMem
    //!
    void SetTexCoordGen(u32 idx, const res::TexCoordGen& value)
    {
        NW_ASSERTMSG(idx < m_MemNum.texCoordGen, "out of bounds: idx[%u] < m_MemNum.texCoordGen[%u] for material[%s]", idx, m_MemNum.texCoordGen, GetName());
        GetTexCoordGenAry()[idx] = value;
    }

    //! @brief 投影テクスチャ生成の数を取得します。
    //!
    //! @return 投影テクスチャ生成の数を返します。
    //!
    //! @sa SetProjectionTexGenNum
    //!
    u8 GetProjectionTexGenNum() const
    {
        return u8(m_MemNum.projectionTexGen);
    }

    //! @brief 投影テクスチャ生成の数を設定します。
    //!
    //! @param[in] num  投影テクスチャ生成の数です。
    //!
    //! @sa GetProjectionTexGenNum
    //!
    void SetProjectionTexGenNum(u8 num);

    //! @brief 成分を指定してテクスチャ SRT のパラメータを取得します。
    //!
    //! @details
    //! テクスチャSRTの表示位置、回転、拡大率の成分を指定して取得します。
    //!
    //! eleIdx の指定には lyt::AnimTargetTexSRT の値を使用してください。
    //!
    //! @param[in] idx      TexSRT のインデックスです。
    //! @param[in] eleIdx   成分のインデックスです。
    //!
    //! @return テクスチャSRTの指定された成分を返します。
    //!
    //! @sa AnimTargetTexSRT
    //! @sa SetTexSRTElement
    //!
    f32 GetTexSRTElement(u32 idx, u32 eleIdx) const
    {
        NW_ASSERTMSG(idx < m_MemNum.texSRT, "out of bounds: idx[%u] < m_MemNum.texSRT[%u] for material[%s]", idx, m_MemNum.texSRT, GetName());

        const ut::ResF32 *const srtAry = &GetTexSRTAry()[idx].translate.x;
        return srtAry[eleIdx];
    }

    //! @brief 成分を指定してテクスチャ SRT のパラメータを設定します。
    //!
    //! @details
    //! テクスチャの表示位置、回転、拡大率の成分を指定して設定します。
    //!
    //! eleIdx の指定には lyt::AnimTargetTexSRT を使用してください。
    //!
    //! idx に ReserveMem() または SetTexSRTNum() で設定したテクスチャ SRT 行列の
    //! 格納可能数よりも大きな値を渡した場合はアサートが発生します。
    //!
    //! @param[in] idx      TexSRT のインデックスです。
    //! @param[in] eleIdx   成分のインデックスです。
    //! @param[in] value    成分の値です。
    //!
    //! @sa AnimTargetTexSRT
    //! @sa GetTexSRTElement
    //! @sa ReserveMem
    //! @sa SetTexSRTNum
    //!
    void SetTexSRTElement(u32 idx, u32 eleIdx, f32 value)
    {
        NW_ASSERTMSG(idx < m_MemNum.texSRT, "out of bounds: idx[%u] < m_MemNum.texSRT[%u] for material[%s]", idx, m_MemNum.texSRT, GetName());

        ut::ResF32 *const srtAry = &GetTexSRTAry()[idx].translate.x;
        srtAry[eleIdx] = value;
    }

    //@}

    //----------------------------------------
    //! @name TEV
    //@{

    //! @brief TEV ステージ数を取得します。
    //!
    //! @return TEV ステージ数を返します。
    //!
    //! @sa SetTevStageNum
    //!
    u8 GetTevStageNum() const
    {
        return u8(m_MemNum.tevStage);
    }

    //! @brief TEV ステージ数を設定します。
    //!
    //! @param[in] num  TEV ステージの数です。
    //!
    //! @sa GetTevStageNum
    //!
    void SetTevStageNum(u8 num);

    //! @brief カラーを取得します。
    //!
    //! @details
    //! idx には lyt::InterpolateColor の値を指定します。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return オブジェクトへの参照を返します。
    //!
    //! @sa SetColor
    //! @sa lyt::InterpolateColor
    //!
    const ut::Color8& GetColor(u32 idx) const
    {
        NW_ASSERTMSG(idx < MatColorMax, "out of bounds: idx[%u] < MatColorMax[%u] for material[%s]", idx, MatColorMax, GetName());
        return m_Colors[idx];
    }

    //! @brief カラーを設定します。
    //!
    //! @details
    //! idx には lyt::InterpolateColor の値を指定します。
    //!
    //! @param[in] idx      インデックスです。
    //! @param[in] value    カラーの値です。
    //!
    //! @sa GetColor
    //! @sa lyt::InterpolateColor
    //!
    void SetColor(u32 idx, ut::Color8 value)
    {
        NW_ASSERTMSG(idx < MatColorMax, "out of bounds: idx[%u] < MatColorMax[%u] for material[%s]", idx, MatColorMax, GetName());
        m_Colors[idx] = value;
    }

    //! @brief カラーの種類を指定して TEV のカラーの成分を取得します。
    //!
    //! @details
    //! colorType の指定には lyt::AnimTargetMatColor を使用してください。
    //!
    //! @param[in] colorType    カラーの種類です。
    //!
    //! @return カラー成分の値を返します。
    //!
    //! @sa SetColorElement
    //! @sa lyt::AnimTargetMatColor
    //!
    u8 GetColorElement(u32 colorType) const;

    //! @brief カラーの種類を指定して TEV のカラーの成分を設定します。
    //!
    //! @details
    //! colorType の指定には lyt::AnimTargetMatColor を使用してください。
    //!
    //! @param[in] colorType    カラーの種類です。
    //! @param[in] value        カラー値です。
    //!
    //! @sa GetColorElement
    //! @sa lyt::AnimTargetMatColor
    //!
    void SetColorElement(u32 colorType, u8 value);

    //! @brief 成分を指定してインダイレクト SRT のパラメータを設定します。
    //!
    //! @details
    //! インダイレクトの回転、拡大率の成分を指定して設定します。(現状、Transには未対応です。)
    //!
    //! eleIdx の指定には lyt::AnimTargetTexSRT を使用してください。
    //!
    //! @param[in] eleIdx   成分のインデックスです。
    //! @param[in] value    成分の値です。
    //!
    //! @sa AnimTargetTexSRT
    //! @sa ReserveMem
    //!
    void SetIndirectSRTElement(u32 eleIdx, f32 value)
    {
        ut::ResF32 *const srtAry = &GetIndirectParameterPtr()->rotate;
        srtAry[eleIdx] = value;
    }

    //! @brief TEV ステージを取得します。
    //!
    //! @details
    //! idx に ReserveMem() 関数または SetTevStageNum() 関数で設定した
    //! TEVステージ数よりも大きな値を渡した場合にはアサートが発生します。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return TevStage オブジェクトへの参照を返します。
    //!
    //! @sa SetTevStage
    //! @sa ReserveMem
    //! @sa GetTevStageNum
    //! @sa SetTevStageNum
    //!
    const res::TevStage& GetTevStage(u32 idx) const
    {
        NW_ASSERTMSG(idx < m_MemNum.tevStage, "out of bounds: idx[%u] < m_MemNum.tevStage[%u] for material[%s]", idx, m_MemNum.tevStage, GetName());
        return GetTevStageAry()[idx];
    }

    //! @brief TEV ステージを設定します。
    //!
    //! @param[in] idx      インデックスです。
    //! @param[in] value    TEV ステージ設定です。
    //!
    //! @details
    //! idx に ReserveMem または SetTevStageNum で設定したTEVステージ数よりも
    //! 大きな値を渡した場合はアサートが発生します。
    //!
    //! @sa ReserveMem
    //! @sa GetTevStage
    //!
    void SetTevStage(u32 idx, const res::TevStage& value)
    {
        NW_ASSERTMSG(idx < m_MemNum.tevStage, "out of bounds: idx[%u] < m_MemNum.tevStage[%u] for material[%s]", idx, m_MemNum.tevStage, GetName());
        GetTevStageAry()[idx] = value;
    }

    //@}

    //----------------------------------------
    //! @name レンダーステート
    //@{

    //! @brief アルファコンペア設定を取得します。
    //!
    //! @return アルファコンペア設定を返します。
    //!
    //! @sa SetAlphaCompare
    //!
    const res::AlphaCompare& GetAlphaCompare() const
    {
        NW_ASSERTMSG(IsAlphaCompareCap(), "IsAlphaCompareCap() must be true for material[%s]", GetName());
        return *GetAlphaComparePtr();
    }

    //! @brief アルファコンペアの設定を行います。
    //!
    //! @param[in] value    アルファコンペア設定です。
    //!
    //! @sa GetAlphaCompare
    //!
    void SetAlphaCompare(const res::AlphaCompare& value)
    {
        NW_ASSERTMSG(IsAlphaCompareCap(), "IsAlphaCompareCap() must be true for material[%s]", GetName());
        *GetAlphaComparePtr() = value;
    }

    //! @brief ブレンドモード設定を取得します。
    //!
    //! @return ブレンドモード設定を返します。
    //!
    //! @sa SetBlendMode
    //!
    const res::BlendMode& GetBlendMode() const
    {
        NW_ASSERTMSG(IsBlendModeCap(), "IsBlendModeCap() must be true for material[%s]", GetName());
        return *GetBlendModePtr();
    }

    //! @brief ブレンドモードの設定を行います。
    //!
    //! @param[in] value    ブレンドモード設定です。
    //!
    //! @sa GetBlendMode
    //!
    void SetBlendMode(res::BlendMode value)
    {
        NW_ASSERTMSG(IsBlendModeCap(), "IsBlendModeCap() must be true for material[%s]", GetName());
        *GetBlendModePtr() = value;
    }

    //! @brief アルファのブレンドモード設定を取得します。
    //!
    //! @return アルファのブレンドモード設定を返します。
    //!
    //! @sa SetBlendModeAlpha
    //!
    const res::BlendMode& GetBlendModeAlpha() const
    {
        NW_ASSERTMSG(IsBlendModeAlphaCap(), "IsBlendModeAlphaCap() must be true for material[%s]", GetName());
        return *GetBlendModeAlphaPtr();
    }

    //! @brief ブレンドモードの設定を行います。
    //!
    //! @param[in] value    ブレンドモード設定です。
    //!
    //! @sa GetBlendMode
    //!
    void SetBlendModeAlpha(res::BlendMode value)
    {
        NW_ASSERTMSG(IsBlendModeAlphaCap(), "IsBlendModeAlphaCap() must be true for material[%s]", GetName());
        *GetBlendModeAlphaPtr() = value;
    }

    //! @brief インダイレクトのパラメータを返します。
    //!
    //! @return インダイレクトのパラメータを返します。
    //!
    const res::IndirectParameter& GetIndirectParameter() const
    {
        NW_ASSERTMSG(IsIndirectParameterCap(), "IsIndirectParameterCap() must be true for material[%s]", GetName());
        return *GetIndirectParameterPtr();
    }

    //! @brief インダイレクトのパラメータの設定を行います。
    //!
    //! @param[in] value    インダイレクトのパラメータ設定です。
    //!
    //! @sa GetBlendMode
    //!
    void SetIndirectParameter(const res::IndirectParameter& value)
    {
        NW_ASSERTMSG(IsIndirectParameterCap(), "IsIndirectParameterCap() must be true for material[%s]", GetName());
        *GetIndirectParameterPtr() = value;
    }

    //! @brief 投影テクスチャ生成のパラメータを設定します。
    //!
    //! @details
    //! インデックスで指定したテクスチャ座標生成は、平行投影かペイン相対の平行投影である必要があります。
    //!
    //! @param[in] idx      設定したいテクスチャ座標生成のインデックスです。
    //! @param[in] value    投影テクスチャ生成のパラメータ設定です。
    //!
    void SetProjectionTexGenParamaters(u32 idx, const res::ProjectionTexGenParamaters& value)
    {
        GetProjectionTexGenAry()[GetProjectionTexGenParamatersIdxFromTexCoordGenIdx(idx)] = value;
    }

    //! @brief 投影テクスチャ生成のパラメータを取得します。
    //!
    //! @details
    //! インデックスで指定したテクスチャ座標生成は、平行投影かペイン相対の平行投影である必要があります。
    //!
    //! @param[in] idx     取得したいテクスチャ座標生成のインデックスです。
    //! @return 投影テクスチャ生成のパラメータを返します。
    //!
    const res::ProjectionTexGenParamaters& GetProjectionTexGenParamaters(u32 idx) const
    {
        return GetProjectionTexGenAry()[GetProjectionTexGenParamatersIdxFromTexCoordGenIdx(idx)];
    }
    //@}

    //----------------------------------------
    //! @name カラー
    //@{

    //! @brief フォント影のパラメータの数を設定します。
    //!
    //! @param[in] num フォント影のパラメータの数です。
    //!
    void SetFontShadowParameterNum(u8 num);

    //! @brief フォント影のパラメータを返します。
    //!
    //! @return フォント影のパラメータを返します。
    //!
    const res::FontShadowParameter& GetFontShadowParameter() const
    {
        NW_ASSERTMSG(IsFontShadowParameterCap(), "IsFontShadowParameterCap() must be true for material[%s]", GetName());
        return *GetFontShadowParameterPtr();
    }

    //! @brief フォント影のパラメータの設定を行います。
    //!
    //! @param[in] value    フォント影のパラメータです。
    //!
    //! @sa GetBlendMode
    //!
    void SetFontShadowParameter(const res::FontShadowParameter& value)
    {
        NW_ASSERTMSG(IsFontShadowParameterCap(), "IsFontShadowParameterCap() must be true for material[%s]", GetName());
        *GetFontShadowParameterPtr() = value;
    }

    //! @brief 成分を指定してフォント影のパラメータを設定します。
    //!
    //! eleIdx の指定には lyt::AnimTargetFontShadow を使用してください。
    //!
    //! @param[in] eleIdx   成分のインデックスです。
    //! @param[in] value    成分の値です。
    //!
    //! @sa AnimTargetFontShadow
    //! @sa ReserveMem
    //!
    void SetFontShadowParameterElement(u32 eleIdx, u8 value)
    {
        ut::ResU8 *const elements = GetFontShadowParameterPtr()->blackInterporateColor;
        elements[eleIdx] = value;
    }

    //@}

    //----------------------------------------
    //! @name アニメーション
    //@{

    //! @brief アニメーションを関連付けます。
    //!
    //! @param[in,out] pAnimTrans   アニメーションへのポインタです。
    //!
    //! @sa UnbindAnimation
    //!
    virtual void BindAnimation(AnimTransform* pAnimTrans);

    //! @brief アニメーションの関連付けを解除します。
    //!
    //! @param[in,out] pAnimTrans   アニメーションへのポインタです。
    //!
    //! @sa BindAnimation
    //!
    virtual void UnbindAnimation(AnimTransform* pAnimTrans);

    //----------------------------------------
    //! @name その他
    //@{

    //! @brief ユーザーがマテリアルを構築したかどうかを判定します。
    //!
    //! @return マテリアルをユーザーが構築した場合は true を返します。
    //!
    bool IsUserAllocated() const
    {
        return internal::TestBit(m_Flag, FLAG_USER_ALLOCATED);
    }

    //! @brief ユーザーが構築したマテリアルであることを宣言します。
    //!
    //! @details
    //! ユーザが構築したと宣言されたマテリアルはライブラリによって
    //! 解放されません。
    //!
    void SetUserAllocated()
    {
        internal::SetBit(&m_Flag, FLAG_USER_ALLOCATED, true);
    }

    //@}

    // マテリアルの設置をグラフィックスシステムに送信します。
    // ペインの描画中に呼び出されます。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in,out] drawInfo 描画情報です。
    //! @param[in] alpha                マテリアルカラーに対するアルファ値です。
    //! @param[in] shaderVariation      シェーダの種類
    //! @param[in] bInitFrameTransform  フレーム変換にデフォルト値を設定する場合は true を指定します。
    //! @param[in] paneGlbMtx           ペインの平行移動量です。
    //! @param[in] paneSize             ペインのサイズです。
    //!
    virtual void SetupGraphics(
        DrawInfo& drawInfo,
        u8 alpha,
        ShaderVariation shaderVariation,
        bool bInitFrameTransform = true,
        const math::MTX34& paneGlbMtx = math::MTX34::Identity(),
        const nw::lyt::Size* paneSize = NULL);

    void SetupSubmaterialOf_TextureMatrix(DrawInfo& drawInfo, const math::MTX34& paneGlbMtx = math::MTX34::Identity(), const nw::lyt::Size* paneSize = NULL);
    void SetupSubmaterialOf_Texture(const DrawInfo& drawInfo);
    void SetupSubmaterialOf_Tev(DrawInfo& drawInfo);
    void SetupSubmaterialOf_Blender();

protected:
    //! @brief テクスチャ画像設定の配列を取得します。
    //!
    //! @return テクスチャ画像設定の配列を返します。
    //!
    const TexMap* GetTexMapAry() const;

    //! @brief テクスチャ画像設定の配列を取得します。
    //!
    //! @return テクスチャ画像設定の配列を返します。
    //!
    TexMap* GetTexMapAry();

    //! @brief テクスチャ座標変換設定の配列を取得します。
    //!
    //! @return テクスチャ座標変換設定の配列を返します。
    //!
    const res::TexSRT* GetTexSRTAry() const;

    //! @brief テクスチャ座標変換設定の配列を取得します。
    //!
    //! @return テクスチャ座標変換設定の配列を返します。
    //!
    res::TexSRT* GetTexSRTAry();

    //! @brief テクスチャ座標生成設定の配列を取得します。
    //!
    //! @return テクスチャ座標生成設定の配列を返します。
    //!
    const res::TexCoordGen* GetTexCoordGenAry() const;

    //! @brief テクスチャ座標生成設定の配列を取得します。
    //!
    //! @return テクスチャ座標生成設定の配列を返します。
    //!
    res::TexCoordGen* GetTexCoordGenAry();

    //! @brief アルファコンペア設定のアドレスを取得します。
    //!
    //! @return アルファコンペア設定のアドレスを返します。
    //!
    const res::AlphaCompare* GetAlphaComparePtr() const;

    //! @brief アルファコンペア設定のアドレスを取得します。
    //!
    //! @return アルファコンペア設定のアドレスを返します。
    //!
    res::AlphaCompare* GetAlphaComparePtr();

    //! @brief ブレンドモード設定のアドレスを取得します。
    //!
    //! @return ブレンドモード設定のアドレスを返します。
    //!
    const res::BlendMode* GetBlendModePtr() const;

    //! @brief ブレンドモード設定のアドレスを取得します。
    //!
    //! @return ブレンドモード設定のアドレスを返します。
    //!
    res::BlendMode* GetBlendModePtr();

    //! @brief アルファのブレンドモード設定のアドレスを取得します。
    //!
    //! @return アルファのブレンドモード設定のアドレスを返します。
    //!
    const res::BlendMode* GetBlendModeAlphaPtr() const;

    //! @brief アルファのブレンドモード設定のアドレスを取得します。
    //!
    //! @return アルファのブレンドモード設定のアドレスを返します。
    //!
    res::BlendMode* GetBlendModeAlphaPtr();

    //! @brief インダイレクトのパラメータのアドレスを取得します。
    //!
    //! @return インダイレクトのパラメータのアドレスを返します。
    //!
    const res::IndirectParameter* GetIndirectParameterPtr() const;

    //! @brief インダイレクトのパラメータのアドレスを取得します。
    //!
    //! @return インダイレクトのパラメータのアドレスを返します。
    //!
    res::IndirectParameter* GetIndirectParameterPtr();

    //! @brief 投影テクスチャ生成の配列を取得します。
    //!
    //! @return 投影テクスチャ生成の配列を返します。
    //!
    const res::ProjectionTexGenParamaters* GetProjectionTexGenAry() const;

    //! @brief 投影テクスチャ生成の配列を取得します。
    //!
    //! @return 投影テクスチャ生成の配列を返します。
    //!
    res::ProjectionTexGenParamaters* GetProjectionTexGenAry();

    //! @brief TEV設定のアドレスを取得します。
    //!
    //! @return TEV設定のアドレスを返します。
    //!
    const res::TevStage* GetTevStageAry() const;

    //! @brief TEV設定の配列を取得します。
    //!
    //! @return TEV設定の配列を返します。
    //!
    res::TevStage* GetTevStageAry();

    //! @brief フォント影のパラメータを取得します。
    //!
    //! @return フォント影のパラメータを返します。
    //!
    const res::FontShadowParameter* GetFontShadowParameterPtr() const;

    //! @brief フォント影のパラメータを取得します。
    //!
    //! @return フォント影のパラメータを返します。
    //!
    res::FontShadowParameter* GetFontShadowParameterPtr();

    //! @brief テクスチャ座標生成のインデックスから対応する投影テクスチャ生成のパラメータのインデックスを取得します。
    //!
    //! @param[in] texCoordGenIdx   投影テクスチャ生成のパラメータのインデックスを取得したいテクスチャ座標生成のインデックスです。
    //!
    //! @return 投影テクスチャ生成のパラメータのインデックスを返します。
    //!
    u32 GetProjectionTexGenParamatersIdxFromTexCoordGenIdx(u32 texCoordGenIdx) const;

    //! @brief テクスチャの座標変換行列を求めます。
    //!
    //! @param[out] pTexMtx 計算結果の格納先です。
    //! @param[in] texSRT   テクスチャのSRT情報です。
    //! @param[in] texMap   テクスチャ情報です。
    //!
    static void CalcTextureMtx(math::MTX23* pTexMtx, const res::TexSRT& texSRT, const TexMap& texMap);

    //! @brief インダイレクトテクスチャの座標変換行列を求めます。
    //!
    //! @param[out] pIndirectMtx 計算結果の格納先です。
    //! @param[in]  rotate テクスチャの回転情報です。
    //! @param[in]  scale テクスチャのスケール情報です。
    //!
    static void CalcIndirectMtx(math::MTX23* pIndirectMtx, const f32 rotate, const res::Vec2& scale);

private:
    void Init();
    void InitMatMemNums(internal::MatMemNums* ptr);
    void GetShaderName(char* shader_name_buf);

    // コピー演算子を禁止します。
    const Material& operator=( const Material& );

    ut::Color8 m_Colors[MatColorMax];
    internal::MatMemNums m_MemCap;
    internal::MatMemNums m_MemNum;
    void* m_pMem;
    const ArchiveShaderInfo* m_ShaderInfo;
    u8 m_ShaderId;
    u8 m_Flag;
    const char* m_Name;

};

//----------------------------------------
NW_INLINE const TexMap*
Material::GetTexMapAry() const
{
    return internal::ConvertOffsToPtr<TexMap>(m_pMem, 0);
}

//----------------------------------------
NW_INLINE TexMap*
Material::GetTexMapAry()
{
    return internal::ConvertOffsToPtr<TexMap>(m_pMem, 0);
}

//----------------------------------------
NW_INLINE const res::TexSRT*
Material::GetTexSRTAry() const
{
    return internal::ConvertOffsToPtr<res::TexSRT>(m_pMem, internal::CalcOffsetTexSRTAry(m_MemCap));
}

//----------------------------------------
NW_INLINE res::TexSRT*
Material::GetTexSRTAry()
{
    return internal::ConvertOffsToPtr<res::TexSRT>(m_pMem, internal::CalcOffsetTexSRTAry(m_MemCap));
}

//----------------------------------------
NW_INLINE const res::TexCoordGen*
Material::GetTexCoordGenAry() const
{
    return internal::ConvertOffsToPtr<res::TexCoordGen>(m_pMem, internal::CalcOffsetTexCoordGenAry(m_MemCap));
}

//----------------------------------------
NW_INLINE res::TexCoordGen*
Material::GetTexCoordGenAry()
{
    return internal::ConvertOffsToPtr<res::TexCoordGen>(m_pMem, internal::CalcOffsetTexCoordGenAry(m_MemCap));
}

//----------------------------------------
NW_INLINE const res::AlphaCompare*
Material::GetAlphaComparePtr() const
{
    return internal::ConvertOffsToPtr<res::AlphaCompare>(m_pMem, internal::CalcOffsetGetAlphaCompare(m_MemCap));
}

//----------------------------------------
NW_INLINE res::AlphaCompare*
Material::GetAlphaComparePtr()
{
    return internal::ConvertOffsToPtr<res::AlphaCompare>(m_pMem, internal::CalcOffsetGetAlphaCompare(m_MemCap));
}

//----------------------------------------
NW_INLINE const res::BlendMode*
Material::GetBlendModePtr() const
{
    return internal::ConvertOffsToPtr<res::BlendMode>(m_pMem, internal::CalcOffsetBlendMode(m_MemCap));
}

//----------------------------------------
NW_INLINE res::BlendMode*
Material::GetBlendModePtr()
{
    return internal::ConvertOffsToPtr<res::BlendMode>(m_pMem, internal::CalcOffsetBlendMode(m_MemCap));
}

//----------------------------------------
NW_INLINE const res::BlendMode*
Material::GetBlendModeAlphaPtr() const
{
    return internal::ConvertOffsToPtr<res::BlendMode>(m_pMem, internal::CalcOffsetBlendMode(m_MemCap) + sizeof(res::BlendMode));
}

//----------------------------------------
NW_INLINE res::BlendMode*
Material::GetBlendModeAlphaPtr()
{
    return internal::ConvertOffsToPtr<res::BlendMode>(m_pMem, internal::CalcOffsetBlendMode(m_MemCap) + sizeof(res::BlendMode));
}

//----------------------------------------
NW_INLINE const res::IndirectParameter*
Material::GetIndirectParameterPtr() const
{
    return internal::ConvertOffsToPtr<res::IndirectParameter>(m_pMem, internal::CalcOffsetIndirectParameter(m_MemCap));
}

//----------------------------------------
NW_INLINE res::IndirectParameter*
Material::GetIndirectParameterPtr()
{
    return internal::ConvertOffsToPtr<res::IndirectParameter>(m_pMem, internal::CalcOffsetIndirectParameter(m_MemCap));
}

//----------------------------------------
NW_INLINE const res::ProjectionTexGenParamaters*
Material::GetProjectionTexGenAry() const
{
    return internal::ConvertOffsToPtr<res::ProjectionTexGenParamaters>(m_pMem, internal::CalcOffsetProjectionTexGen(m_MemCap));
}

//----------------------------------------
NW_INLINE res::ProjectionTexGenParamaters*
Material::GetProjectionTexGenAry()
{
    return internal::ConvertOffsToPtr<res::ProjectionTexGenParamaters>(m_pMem, internal::CalcOffsetProjectionTexGen(m_MemCap));
}

//----------------------------------------
NW_INLINE const res::TevStage*
Material::GetTevStageAry() const
{
    return internal::ConvertOffsToPtr<res::TevStage>(m_pMem, internal::CalcOffsetTevStageAry(m_MemCap));
}

//----------------------------------------
NW_INLINE res::TevStage*
Material::GetTevStageAry()
{
    return internal::ConvertOffsToPtr<res::TevStage>(m_pMem, internal::CalcOffsetTevStageAry(m_MemCap));
}

//----------------------------------------
NW_INLINE const res::FontShadowParameter*
Material::GetFontShadowParameterPtr() const
{
    return internal::ConvertOffsToPtr<res::FontShadowParameter>(m_pMem, internal::CalcOffsetFontShadowParameter(m_MemCap));
}

//----------------------------------------
NW_INLINE res::FontShadowParameter*
Material::GetFontShadowParameterPtr()
{
    return internal::ConvertOffsToPtr<res::FontShadowParameter>(m_pMem, internal::CalcOffsetFontShadowParameter(m_MemCap));
}

//----------------------------------------
NW_INLINE void
Material::SetName(const char* name)
{
    m_Name = name;
}

//----------------------------------------
NW_INLINE void
Material::SetTexMapNum(u8 num)
{
    NW_ASSERTMSG(num <= m_MemCap.texMap, "out of bounds: num[%u] <= m_MemCap.texMap[%u] for material[%s]", num, m_MemCap.texMap, GetName());
    m_MemNum.texMap = num;
}

//----------------------------------------
NW_INLINE void
Material::SetTexSRTNum(u8 num)
{
    NW_ASSERTMSG(num <= m_MemCap.texSRT, "out of bounds: num[%u] <= m_MemCap.texSRT[%u] for material[%s]", num, m_MemCap.texSRT, GetName());
    m_MemNum.texSRT = num;
}

//----------------------------------------
NW_INLINE void
Material::SetTexCoordGenNum(u8 num)
{
    NW_ASSERTMSG(num <= m_MemCap.texCoordGen, "out of bounds: num[%u] <= m_MemCap.texCoordGen[%u] for material[%s]", num, m_MemCap.texCoordGen, GetName());
    m_MemNum.texCoordGen = num;
}

//----------------------------------------
NW_INLINE void
Material::SetTevStageNum(u8 num)
{
    NW_ASSERTMSG(num <= m_MemCap.tevStage, "out of bounds: num[%u] <= m_MemCap.tevStage[%u] for material[%s]", num, m_MemCap.tevStage, GetName());
    m_MemNum.tevStage = num;
}

//----------------------------------------
NW_INLINE void
Material::SetProjectionTexGenNum(u8 num)
{
    NW_ASSERTMSG(num <= m_MemCap.projectionTexGen, "out of bounds: num[%u] <= m_MemCap.projectionTexGen[%u] for material[%s]", num, m_MemCap.projectionTexGen, GetName());
    m_MemNum.projectionTexGen = num;
}

//----------------------------------------
NW_INLINE void
Material::SetFontShadowParameterNum(u8 num)
{
    NW_ASSERTMSG(num <= m_MemCap.fontShadowParameter, "out of bounds: num[%u] <= m_MemCap.fontShadowParameter[%u] for material[%s]", num, m_MemCap.fontShadowParameter, GetName());
    m_MemNum.fontShadowParameter = num;
}

//----------------------------------------
NW_INLINE u8
Material::GetColorElement(u32 colorType) const
{
    if (colorType < ANIMTARGET_MATCOLOR_MAX)
    {
        const u32 idx = (colorType - ANIMTARGET_MATCOLOR_BUFFER_R) / 4;
        switch ((colorType - ANIMTARGET_MATCOLOR_BUFFER_R) % 4)
        {
        case 0: return m_Colors[idx].r;
        case 1: return m_Colors[idx].g;
        case 2: return m_Colors[idx].b;
        case 3: return m_Colors[idx].a;
        }
    }

    return ut::Color8::ELEMENT_MAX;
}

//----------------------------------------
NW_INLINE void
Material::SetColorElement(u32 colorType, u8 value)
{
    if (colorType < ANIMTARGET_MATCOLOR_MAX)
    {
       const u32 idx = (colorType - ANIMTARGET_MATCOLOR_BUFFER_R) / 4;
        switch ((colorType - ANIMTARGET_MATCOLOR_BUFFER_R) % 4)
        {
        case 0: m_Colors[idx].r = value; break;
        case 1: m_Colors[idx].g = value; break;
        case 2: m_Colors[idx].b = value; break;
        case 3: m_Colors[idx].a = value; break;
        }
    }
}

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_MATERIAL_H_
