﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_GROUP_H_
#define NW_LYT_GROUP_H_

#include <cstddef>
#include <nw/ut/ut_LinkList.h>

#include <nw/lyt/lyt_Pane.h>
#include <nw/lyt/lyt_Types.h>

namespace nw
{
namespace lyt
{

namespace res
{

struct Group;

}   // namespace res

//---------------------------------------------------------------------------
//! @brief グループに属するペインへのリンクです。
//!
//---------------------------------------------------------------------------
struct PaneLink
{
    // リンク情報です。

    //! @brief 内部用機能のため使用禁止です。
    ut::LinkListNode link;

    //! グループに属するペインへのポインタです。
    Pane* target;
};

//! @brief ペインへのリンクの一覧を保持します。
//!
typedef ut::LinkList<PaneLink, offsetof(PaneLink, link)> PaneLinkList;

//---------------------------------------------------------------------------
//! @brief ペインをグループで管理するためのクラスです。
//!
//---------------------------------------------------------------------------
class Group
{
public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! ペインのない状態で初期化します。
    //!
    Group();

    //! @brief コンストラクタです。
    //!
    //! @details
    //! ペインのない状態で、グループの名前が指定された状態で初期化します。
    //!
    //! NW4F1.12.0以前ではGroupクラスは内部に名前のための文字列バッファを持っていましたが、
    //! これによりグループ名に制限ができていたことと、消費メモリが多かったため、削除しました。
    //! NW4F1.12.0以降では、引数で与えた文字列のポインタを保持しますので、スタックなどで
    //! 一時的に確保した文字列のポインタポインタを与えないようにしてください。
    //!
    //! 文字列の長さはnw::lyt::GroupNameStrMax以下としてください。
    //!
    //! @param[in] name グループの名前です。NULLではない文字列を指定してください。
    //!
    explicit Group(const char* name);

    //! @brief コンストラクタです。
    //!
    //! @details
    //! グループリソースをもとにペインを追加した状態で初期化します。
    //!
    //! @param[in] pResGroup    グループリソースへのポインタです。
    //! @param[in] pRootPane    ルートペインへのポインタです。
    //!
    Group(const res::Group* pResGroup, Pane* pRootPane);

    //! @brief コピーコンストラクタです。
    //!
    //! @details
    //! 引数pRootPaneにnullptrを与えると、コピー元と同じペインを登録します。
    //! 引数pRootPaneにペインのポインタを与えると、そのペイン以下からコピー元と
    //! 同じ名前のペインを再検索します。これは、レイアウトをコピーする際に、
    //! コピー先の新しくコピーされたペインを参照するようにするために使用する機能です。
    //!
    //! @param[in] group        グループリソースへの参照です。
    //! @param[in] pRootPane    ルートペインへのポインタです。
    //!
    Group(const Group& group, Pane* pRootPane);

    //! @brief デストラクタです。
    //!
    virtual ~Group();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief グループの名前を取得します。
    //!
    //! @return グループの名前を返します。
    //!
    const char* GetName() const
    {
        return m_Name;
    }

    //! @brief グループに所属するペインの一覧を取得します。
    //!
    //! @return ペインの一覧を返します。
    //!
    PaneLinkList& GetPaneList()
    {
        return m_PaneLinkList;
    }

    //@}

    //----------------------------------------
    //! @name 追加
    //@{

    //! @brief ペインを追加します。
    //!
    //! @param[in] pPane    ペインへのポインタです。
    //!
    void AppendPane(Pane* pPane);

    //@}

    //----------------------------------------
    //! @name その他
    //@{

    //! @brief ユーザーがグループを構築したかどうかを判定します。
    //!
    //! @return グループをユーザーが構築した場合は true を返します。
    //!
    bool IsUserAllocated() const
    {
        return m_UserAllocated != 0;
    }

    //! @brief ユーザーが構築したグループであることを宣言します。
    //!
    //! @details
    //! ユーザが構築したと宣言されたグループはライブラリによって解放されません。
    //!
    void SetUserAllocated()
    {
        m_UserAllocated = 1;
    }

    //@}

    // リンク情報です。

    //! @brief 内部用機能のため使用禁止です。
    ut::LinkListNode m_Link_;

private:
    PaneLinkList m_PaneLinkList;
    const char* m_Name;
    u8 m_UserAllocated; // ユーザにより構築された場合は真
    u8 m_Padding[3];

private:
    // コピー演算子を禁止します。
    const Group& operator=( const Group& );
};

//! @brief グループの一覧を保持します。
//!
typedef ut::LinkList<Group, offsetof(Group, m_Link_)> GroupList;

//---------------------------------------------------------------------------
//! @brief グループを管理するためのクラスです。
//!
//! @details
//! Layout::GetGroupContainer() で取得します。
//!
//---------------------------------------------------------------------------
class GroupContainer
{
    NW_DISALLOW_COPY_AND_ASSIGN(GroupContainer);

public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    GroupContainer()
    {}

    //! @brief デストラクタです。
    //!
    ~GroupContainer();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief グループを追加します。
    //!
    //! @param[in] pGroup   グループへのポインタです。
    //!
    void AppendGroup(Group* pGroup);

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief グループの一覧を取得します。
    //!
    //! @return
    //! グループの一覧を返します。
    //!
    GroupList& GetGroupList()
    {
        return m_GroupList;
    }

    //@}

    //----------------------------------------
    //! @name 検索
    //@{

    //! @brief 名前でグループを検索します。
    //!
    //! @param[in] findName 検索するグループ名です。
    //!
    //! @return 見つかった場合はグループへのポインタを、
    //! 見つからなかった場合は NULL を返します。
    //!
    Group* FindGroupByName(const char* findName);

    //@}

private:
    GroupList m_GroupList;

};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_GROUP_H_

