﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_FONTCONTAINER_H_
#define NW_LYT_FONTCONTAINER_H_

#include <nw/types.h>

#include <nw/ut/ut_LinkList.h>
#include <nw/lyt/lyt_ResourceAccessor.h>
#include <cstddef>

namespace nw
{
namespace lyt
{

//---------------------------------------------------------------------------
// フォントオブジェクトを保持するクラスです。
//
// sa FontContainer
//
//---------------------------------------------------------------------------

//! @brief 内部用機能のため使用禁止です。
class FontRefLink
{
    NW_DISALLOW_COPY_AND_ASSIGN(FontRefLink);

public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    // コンストラクタです。

    //! @brief 内部用機能のため使用禁止です。
    FontRefLink();

    // デストラクタです。
    // 所有の指定されたフォントオブジェクトを破棄します。
    // 2010/01/26 所有の指定されたフォントオブジェクトは破棄する仕様に変更しました。

    //! @brief 内部用機能のため使用禁止です。
    ~FontRefLink();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    // 情報を設定します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] name     フォントリソースの名前です。
    //! @param[in] pFont    フォントオブジェクトへのポインタです。
    //! @param[in] own      コンテナがフォントオブジェクトを所有するか指定します。
    //!                     true を指定した場合にはコンテナがフォントオブジェクトを破棄します。
    //!
    void Set(
        const char* name,
        font::Font* pFont,
        bool own
    );

    // 関連付けられているフォント名を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return 関連付けられているフォント名を返します。
    //!
    const char* GetFontName() const
    {
        return m_FontName;
    }

    // 設定されているフォントオブジェクトを取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return フォントオブジェクトへのポインタを返します。
    //!
    font::Font* GetFont() const
    {
        return m_pFont;
    }

    //@}

    // リストのリンク情報です。

    //! @brief 内部用機能のため使用禁止です。
    ut::LinkListNode m_Link;

private:
    char m_FontName[FontNameMax];
    font::Font* m_pFont;
    bool m_Own;
};

//---------------------------------------------------------------------------
//! @brief フォントを保持するクラスです。
//!
//! @details
//! ResourceAccessor の実装でフォントを保持するのに利用されます。
//!
//! @sa ResourceAccessor
//!
//---------------------------------------------------------------------------
class FontContainer : public ut::LinkList<FontRefLink, offsetof(FontRefLink, m_Link)>
{
    NW_DISALLOW_COPY_AND_ASSIGN(FontContainer);

public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 所有しているフォントオブジェクトも破棄します。
    //!
    FontContainer() {}

    //! @brief デストラクタです。
    //!
    //! @details
    //! 所有しているフォントオブジェクトも破棄します。
    //!
    ~FontContainer();

    // @}

    //----------------------------------------
    //! @name 登録／破棄
    //@{

    //! @brief 登録を抹消します。
    //!
    //! @details
    //! 所有しているフォントオブジェクトを破棄します。
    //!
    void Finalize();

    //! @brief フォントを登録します。
    //!
    //! @param[in] name     フォントリソースの名前です。
    //! @param[in] pFont    フォントオブジェクトです。
    //! @param[in] own      フォントオブジェクトを所有するか指定します。
    //!                     true を指定した場合にはコンテナがフォントオブジェクトを破棄します。
    //!
    //! @return 登録キーを返します。キーは登録の抹消に使用します。
    //!
    FontKey RegistFont(const char* name, font::Font* pFont, bool own);

    //! @brief フォントの登録を抹消します。
    //!
    //! @param[in] key  登録時に取得した登録キーを指定します。
    //!
    void UnregistFont(FontKey key);

    //@}

    //----------------------------------------
    //! @name 検索
    //@{

    //! @brief フォントを検索します。
    //!
    //! @param[in] name 検索するフォントの名前です。
    //!
    //! @return 見つかったフォント、または NULL を返します。
    //!
    nw::font::Font* FindFontByName(
        const char* name
    );
};

}   // namespace lyt
}   // namespace nw

/* NW_LYT_FONTCONTAINER_H_ */
#endif
