﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_ARCRESOURCEACCESSOR_H_
#define NW_LYT_ARCRESOURCEACCESSOR_H_

#include <nw/types.h>

#include <nw/ut/ut_LinkList.h>
#include <nw/lyt/lyt_FontContainer.h>
#include <nw/lyt/lyt_ResourceAccessor.h>
#include <nw/lyt/lyt_TextureContainer.h>
#include <nw/lyt/lyt_ShaderContainer.h>
#include <cstddef>
#include <nw/lyt/lyt_ArcExtractor.h>

namespace nw
{
namespace lyt
{

typedef ArcExtractor ArchiveHandle;

//---------------------------------------------------------------------------
//! @brief アーカイブリソースにアクセスするためのクラスです。
//!
//---------------------------------------------------------------------------
class ArcResourceAccessor : public ResourceAccessor
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(ResourceAccessor);

    //! パス名の最大長です。
    static const int ROOTPATH_MAX = 64;

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    ArcResourceAccessor();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief アーカイブデータを登録します。
    //!
    //! @details
    //! archiveStartで与えるアーカイブデータは、 nw::lyt::ARCHIVE_RESOURCE_ALIGNMENT バイトで
    //! アライメントしてください。
    //!
    //! これは、アーカイブデータ内に含まれているテクスチャデータを、別の場所にコピーせずに
    //! そのままGPUで扱うことができるようにするためです。アーカイブ内のファイルのデータ
    //! アラインメントは、レイアウトアーカイバが自動的に行います。
    //!
    //! nw::lyt::ARCHIVE_RESOURCE_ALIGNMENT は、要求される最も大きいアライメントに合わせて
    //! 定義しています。
    //!
    //! @param[in] archiveStart             アーカイブデータへのポインタです。
    //! @param[in] resourceRootDirectory    リソース検索のルートとなるディレクトリ名です。
    //!
    //! @return 登録に成功したら true、失敗したら falseを返します。
    //!
    bool Attach(
        void* archiveStart,
        const char* resourceRootDirectory
    );

    //! @brief 登録したアーカイブデータの登録を解除します。
    //!
    //! @details
    //! レイアウトとリソースは1対1ではなく、複数のレイアウトから同じリソースを共有することができるため、
    //! レイアウトの解放時であってもリソースは自動で解放されません。
    //! 本関数を呼ぶことで、リソースを解放することができます。
    //!
    //! @return 登録されていたアーカイブデータへのポインタを返します。
    //!
    void* Detach();

    virtual void* GetResource(
        ResType resType,
        const char* name,
        u32* pSize = 0
    );

    //! @brief アクセサにアーカイブが設定されているかどうかを判定します。
    //!
    //! @return すでにアーカイブが設定されている場合は true を返します。
    //!
    bool IsAttached() const
    {
        return m_ArcBuf != 0;
    }

    //! @brief アーカイブの先頭アドレスを取得します。
    //!
    //! @return
    //! 設定されているアーカイブの先頭アドレスを返します。
    //! アーカイブが設定されていない場合は NULL を返します。
    //!
    const void* GetArchiveDataStart() const
    {
        return m_ArcBuf;
    }

    //! @brief フォントオブジェクトを登録します。
    //!
    //! @details
    //! フォントオブジェクトをリソースアクセサに設定し、
    //! GetFont() の検索対象に含まれるようにします。
    //!
    //! @param[in] name     フォントリソースの名前です。
    //! @param[in] pFont    フォントオブジェクトへのポインタです。
    //!
    //! @return 登録キーを返します。キーは登録の抹消に使用します。
    //!
    //! @sa UnregistFont
    //!
    FontKey RegistFont(const char* name, font::Font* pFont);

    //! @brief フォントオブジェクトの登録を解除します。
    //!
    //! @param[in] key  フォントの登録時に取得した登録キーを指定します。
    //!
    //! @sa RegistFont
    //!
    void UnregistFont(FontKey key);

    //! @brief テクスチャを登録します。
    //!
    //! @details
    //! テクスチャをリソースアクセサに設定し、
    //! GetTexture() の検索対象に含まれるようにします。
    //!
    //! テクスチャを登録するには、返り値の TextureInfo* に対して、
    //! テクスチャの設定を行ってください。
    //!
    //! @param[in] name テクスチャの名前です。
    //!
    //! @return テクスチャ情報です。メモリ確保できない場合には NULL が返ります。
    //!
    //! @sa UnregistTexture
    //!
    TextureInfo* RegistTexture(const char* name);

    //! @brief テクスチャの登録を解除します。
    //!
    //! @param[in] info テクスチャ情報です。
    //!
    //! @sa RegistTexture
    //!
    void UnregistTexture(TextureInfo* info);

    //! @brief シェーダオブジェクトを登録します。
    //!
    //! @details
    //! シェーダオブジェクトをリソースアクセサに設定し、
    //! GetShader() の検索対象に含まれるようにします。
    //!
    //! @param[in] name     シェーダリソースの名前です。
    //!
    //! @return 登録キーを返します。キーは登録の抹消に使用します。
    //!
    //! @sa UnregistShader
    //!
    ArchiveShaderInfo* RegistShader(const char* name);

    //! @brief シェーダオブジェクトの登録を解除します。
    //!
    //! @param[in] pShader  シェーダの登録時に取得した登録キーを指定します。
    //!
    //! @sa RegistShader
    //!
    void UnregistShader(ArchiveShaderInfo* pShader);

    //! @brief テクスチャ情報を取得します。
    //!
    //! @details
    //! name で指定したテクスチャリソース名と関連付けられた
    //! テクスチャ情報を取得します。
    //!
    //! テクスチャを所有していない場合には LoadTexture() でロードを試みます。
    //!
    //! @param[in] name テクスチャリソースの名前です。
    //!
    //! @return テクスチャ情報を返します。
    //! テクスチャを所有していない場合にはロードを試みます。
    //! ロードできなかった場合にはテクスチャオブジェクトには無効な値
    //! (TextureInfo::INVALID) が設定されます。
    //!
    //! @sa LoadTexture
    //!
    virtual const TextureInfo* GetTexture(const char* name);

    //! @brief フォントを取得します。
    //!
    //! @details
    //! name で指定したフォントリソース名と関連付けられた
    //! フォントオブジェクトを取得します。
    //!
    //! 指定されたフォントが登録されていなかった場合には
    //! LoadFont() により生成したフォントを登録します。
    //!
    //! @param[in] name フォントリソースの名前です。
    //!
    //! @return フォントへのポインタを返します。
    //!
    //! @sa LoadFont
    //!
    virtual font::Font* GetFont(const char* name);

    //! @brief シェーダ情報を取得します。
    //!
    //! @details
    //! name で指定したシェーダリソース名と関連付けられた
    //! シェーダ情報を取得します。
    //!
    //! シェーダを所有していない場合には LoadShader() でロードを試みます。
    //!
    //! @param[in] name シェーダリソースの名前です。
    //!
    //! @return シェーダ情報を返します。
    //! シェーダを所有していない場合にはロードを試みます。
    //! ロードできなかった場合にはシェーダオブジェクトには無効な値
    //! が設定されます。
    //!
    //! @sa LoadShader
    //!
    virtual const ArchiveShaderInfo* GetShader(const char* name);

    //@}

private:
    ArchiveHandle m_ArcHandle;
    void* m_ArcBuf;
    FontContainer m_FontList;
    TextureContainer m_TextureList;
    ShaderContainer m_ShaderList;
    char m_ResRootDir[ROOTPATH_MAX];
};

class MultiArcResourceAccessor;

//---------------------------------------------------------------------------
//! @brief アーカイブの情報を保持するためのクラスです。
//!
//---------------------------------------------------------------------------
class ArcResourceLink
{
public:
    //! パス名の最大長です。
    static const int ROOTPATH_MAX = ArcResourceAccessor::ROOTPATH_MAX;

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    ArcResourceLink()
    {
    }

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief アーカイブを設定し、初期化します。
    //!
    //! @details
    //! アーカイブのフォーマットが正しくなければ失敗します。
    //!
    //! archiveStartで与えるアーカイブデータは、 nw::lyt::ARCHIVE_RESOURCE_ALIGNMENT バイトで
    //! アライメントしてください。
    //!
    //! 詳しくは、 nw::lyt::ArcResourceAccessor::Attach メソッドの説明をご参照ください。
    //!
    //! @param[in] archiveStart             アーカイブの先頭アドレスです。
    //! @param[in] resourceRootDirectory    リソースのルートディレクトリ名です。
    //!
    //! @return アーカイブの設定が成功した場合は true を、失敗した場合は false を返します。
    //!
    //! @sa ArcResourceAccessor::Attach
    //!
    bool Set(
        void* archiveStart,
        const char* resourceRootDirectory
    );

    //! @brief アーカイブの先頭アドレスを取得します。
    //!
    //! @return 設定されているアーカイブの先頭アドレスを返します。
    //!
    const void* GetArchiveDataStart() const;

    //! @brief リソースのルートディレクトリ名を取得します。
    //!
    //! @return リソースのルートディレクトリ名を返します。
    //!
    const char* GetResRootDir() const
    {
        return m_ResRootDir;
    }

    //! @brief アーカイブハンドルを取得します。
    //!
    //! @return アーカイブハンドル
    //!
    ArchiveHandle* GetArcHandle()
    {
        return &m_ArcHandle;
    }

    //@}

    // リストのリンク情報です。

    //! @brief 内部用機能のため使用禁止です。
    ut::LinkListNode m_Link;

private:
    ArchiveHandle m_ArcHandle;
    char m_ResRootDir[ROOTPATH_MAX];
};

namespace internal
{

typedef ut::LinkList<ArcResourceLink, offsetof(ArcResourceLink, m_Link)> ArcResourceList;

}   // namespace internal

//---------------------------------------------------------------------------
//! @brief 複数のアーカイブリソースにアクセスするためのクラスです。
//!
//---------------------------------------------------------------------------
class MultiArcResourceAccessor : public ResourceAccessor
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(ResourceAccessor);

    //! パス名の最大長です。
    static const int ROOTPATH_MAX = ArcResourceAccessor::ROOTPATH_MAX;

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    MultiArcResourceAccessor();

    //! @brief デストラクタです。
    //!
    virtual ~MultiArcResourceAccessor();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief アーカイブ情報を登録します。
    //!
    //! @details
    //! ArcResourceLink を登録します。
    //!
    //! @param[in] pLink    ArcResourceLinkオブジェクトへのポインタです。
    //!
    void Attach(ArcResourceLink* pLink);

    //! @brief 指定したアーカイブデータの登録を解除します。
    //!
    //! @param[in] archiveStart アーカイブデータへのポインタ。
    //!
    //! @return 登録が解除できた場合、ArcResourceLink オブジェクトへのポインタを
    //! 返します。解除できなかった場合、NULLを返します。
    //!
    ArcResourceLink* Detach(const void* archiveStart);

    //! @brief 指定したアーカイブ情報の登録を解除します。
    //!
    //! @details
    //! ArcResourceLink の登録を解除します。
    //!
    //! @param[in] pLink    ArcResourceLink オブジェクトへのポインタです。
    //!
    void Detach(ArcResourceLink* pLink);

    //! @brief 登録されているすべてのアーカイブ情報を登録解除します。
    //!
    //! @sa Detach
    //!
    void DetachAll();

    virtual void* GetResource(
        ResType resType,
        const char* name,
        u32* pSize = 0
    );

    //! @brief フォントオブジェクトを登録します。
    //!
    //! @details
    //! フォントオブジェクトをリソースアクセサに設定し、
    //! GetFont() の検索対象に含まれるようにします。
    //!
    //! @param[in] name     フォントリソースの名前です。
    //! @param[in] pFont    フォントオブジェクトへのポインタです。
    //!
    //! @return 登録キーを返します。キーは登録の抹消に使用します。
    //!
    //! @sa UnregistFont
    //!
    FontKey RegistFont(const char* name, font::Font* pFont);

    //! @brief フォントオブジェクトの登録を解除します。
    //!
    //! @param[in] key  フォントの登録時に取得した登録キーを指定します。
    //!
    //! @sa RegistFont
    //!
    void UnregistFont(FontKey key);

    //! @brief テクスチャを登録します。
    //!
    //! @details
    //! テクスチャをリソースアクセサに設定し、
    //! GetTexture() の検索対象に含まれるようにします。
    //!
    //! テクスチャを登録するには、返り値の TextureInfo* に対して、
    //! テクスチャの設定を行ってください。
    //!
    //! @param[in] name テクスチャの名前です。
    //!
    //! @return テクスチャ情報です。メモリ確保できない場合には NULL が返ります。
    //!
    //! @sa UnregistTexture
    //!
    TextureInfo* RegistTexture(const char* name);

    //! @brief テクスチャの登録を解除します。
    //!
    //! @param[in] info  テクスチャ情報です。
    //!
    //! @sa RegistTexture
    //!
    void UnregistTexture(TextureInfo* info);

    //! @brief シェーダを登録します。
    //!
    //! @details
    //! シェーダをリソースアクセサに設定し、
    //! GetShader() の検索対象に含まれるようにします。
    //!
    //! @param[in] name シェーダの名前です。
    //!
    //! @return シェーダ情報です。メモリ確保できない場合には NULL が返ります。
    //!
    //! @sa UnregistShader
    //!
    ArchiveShaderInfo* RegistShader(const char* name);

    //! @brief シェーダの登録を解除します。
    //!
    //! @param[in] pShader  シェーダ情報です。
    //!
    //! @sa RegistShader
    //!
    void UnregistShader(ArchiveShaderInfo* pShader);

    //! @brief テクスチャ情報を取得します。
    //!
    //! @details
    //! name で指定したテクスチャリソース名と関連付けられた
    //! テクスチャ情報を取得します。
    //!
    //! テクスチャを所有していない場合には LoadTexture() でロードを試みます。
    //!
    //! @param[in] name テクスチャリソースの名前です。
    //!
    //! @return テクスチャ情報を返します。
    //! テクスチャを所有していない場合にはロードを試みます。
    //! ロードできなかった場合にはテクスチャオブジェクトには無効な値
    //! (TextureInfo::INVALID) が設定されます。
    //!
    //! @sa LoadTexture
    //!
    virtual const TextureInfo* GetTexture(const char* name);

    //! @brief フォントを取得します。
    //!
    //! @details
    //! name で指定したフォントリソース名と関連付けられた
    //! フォントオブジェクトを取得します。
    //!
    //! 指定されたフォントが登録されていなかった場合には
    //! LoadFont() により生成したフォントを登録します。
    //!
    //! @param[in] name フォントリソースの名前です。
    //!
    //! @return フォントへのポインタを返します。
    //!
    //! @sa LoadFont
    //!
    virtual font::Font* GetFont(const char* name);

    //! @brief シェーダ情報を取得します。
    //!
    //! @details
    //! name で指定したシェーダリソース名と関連付けられた
    //! シェーダ情報を取得します。
    //!
    //! シェーダを所有していない場合には LoadShader() でロードを試みます。
    //!
    //! @param[in] name シェーダリソースの名前です。
    //!
    //! @return シェーダ情報を返します。
    //! シェーダを所有していない場合にはロードを試みます。
    //! ロードできなかった場合にはシェーダオブジェクトには無効な値
    //! が設定されます。
    //!
    //! @sa LoadShader
    //!
    virtual const ArchiveShaderInfo* GetShader(const char* name);

    //@}

protected:
    //! @brief アーカイバのリストを取得します。
    //!
    //! @details
    //! アーカイバに含まれているファイルを独自に検索したい場合などに
    //! 使用してください。
    //!
    //! @return アーカイバのリスト
    //!
    internal::ArcResourceList* GetArcList()
    {
        return &m_ArcList;
    }

private:
    internal::ArcResourceList m_ArcList;
    FontContainer m_FontList;
    TextureContainer m_TextureList;
    ShaderContainer m_ShaderList;
};

}   // namespace lyt
}   // namespace nw

/* NW_LYT_ARCRESOURCEACCESSOR_H_ */
#endif
