﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_ARC_EXTRACTOR_H_
#define NW_LYT_ARC_EXTRACTOR_H_

#include <nw/types.h>
#include <nw/ut/ut_Array.h>

namespace nw
{
namespace lyt
{

//! Endian 種別です。
enum EndianTypes
{
    BIG_ENDIAN = 0, //!< ビッグエンディアンです。
    LITTLE_ENDIAN = 1, //!< リトルエンディアンです。

#if defined(NW_PLATFORM_WIN32) || defined(NW_PLATFORM_CTR) || defined(NW_USE_NINTENDO_SDK)
    HOST_ENDIAN = LITTLE_ENDIAN
#else
    HOST_ENDIAN = BIG_ENDIAN
#endif
};

//------------------------------------------------------------------------------
//! アーカイブ中の要素です。
//!
struct ArcEntry
{
    static const int MAX_NAME_LENGTH = 256;

    //! コンストラクタ
    ArcEntry()
    {
        name[0] = '\0';
    }

    char name[MAX_NAME_LENGTH]; //!< 要素のパス
};

//------------------------------------------------------------------------------
//! アーカイブ内ファイルの情報です。
//!
class ArcFileInfo
{
    friend class ArcExtractor;
public:
    explicit ArcFileInfo() : mStartOffset( 0 ), mLength( 0 ) {};

    //! ファイルイメージブロックから数えた、ファイル先頭へのオフセットを取得します。
    //!
    //! @return オフセットです。
    //!
    u32 GetStartOffset() const { return mStartOffset; }

    //! ファイルの長さを取得します。
    //!
    //! @return ファイルの長さです。
    //!
    u32 GetLength() const { return mLength; }

    //! ファイルイメージブロックから数えた、ファイル先頭へのオフセットを取得します。
    //! この関数は廃止予定です。GetStartOffset() をご使用ください。
    //!
    //! @return オフセットです。
    //!
    NW_DEPRECATED_FUNCTION(u32 getStartOffset() const) { return mStartOffset; }

    //! ファイルの長さを取得します。
    //! この関数は廃止予定です。GetLength() をご使用ください。
    //!
    //! @return 長さです。
    //!
    NW_DEPRECATED_FUNCTION(u32 getLength() const) { return mLength; }

private:
    u32 mStartOffset; //!< ファイルイメージ先頭の、ファイルイメージブロックからのオフセットです。
    u32 mLength; //!< ファイルの長さです。
};

//------------------------------------------------------------------------------
/**
 *  レイアウトアーカイバで生成したファイルを読み込むクラス.
 *
 *  レイアウトアーカイバで生成するアーカイブを操作します.
 */
class ArcExtractor
{
public:
    //! コンストラクタです。
    //!
    //! @param[in] archive  アーカイブです。
    //!
    explicit ArcExtractor(const void* archive);

    //! コンストラクタです。
    ArcExtractor();

    //! デストラクタです。
    ~ArcExtractor();

    /**
     *  アーカイブのバイナリデータを解釈して、アーカイブ操作を行えるようにします.
     *
     *  @param[in]  archive アーカイブのバイナリデータ.
     *  @return             trueなら読み込み完了. falseなら失敗.
     */
    bool PrepareArchive( const void* archive );

    /**
     *  アーカイブ内のファイルの先頭ポインタをパスから取得します.
     *
     *  @param[in]  file_path   読み込むファイルのパス.
     *  @param[out] file_info   ファイルの長さなどの情報を受け取るポインタ. NULLなら無視する.
     *  @return                 ファイルの先頭ポインタ. NULLのとき失敗.
     */
    NW_INLINE void* GetFile( const char* file_path, ArcFileInfo* file_info = NULL );

    /**
     *  アーカイブ内のファイルをエントリIDから取得します.
     *
     *  エントリーIDはあらかじめConvertPathToEntryIDなどで取得する必要があります.
     *  @param[in]  entry_id    読み込むファイルのエントリーID.
     *  @param[out] file_info   ファイルの長さなどの情報を受け取るポインタ. NULLなら無視する.
     *  @return                 ファイルの先頭ポインタ. NULLのとき失敗.
     */
    void* GetFileFast( s32 entry_id, ArcFileInfo* file_info = NULL );

    /**
     *  アーカイブ内のパスをエントリIDに変換します.
     *
     *  ここで取得したエントリIDは、GetFileFastで利用することができます.
     *  @param[in]  file_path   読み込むファイルのパス.
     *  @return                 ファイルのエントリーID. -1のとき失敗.
     */
    s32 ConvertPathToEntryID( const char* file_path );

    /**
     *  エントリを読み取ります.
     *
     *  - ファイル名がアーカイブに含まれている場合のみ、ファイル名をentryに代入します.
     *  - ファイル名がアーカイブに含まれていないとき、ファイル名の変わりにhashを文字列化したものを代入します.(コンバート時に-nオプションを指定したとき)
     *  @param[in,out] handle  読み込みを開始するエントリID。最初から読み取るときは0に初期化してください。次のReadEntryに渡すことで連続して読み出すことができます。
     *  @param[out] entry      読み込んだエントリ情報を入れる場所.
     *  @param[in] num         読み込む個数.
     *  @return                読み込みが成功したら実際に読み込んだエントリ情報の数.
     */
    u32 ReadEntry( u32* handle, ArcEntry entry[], u32 num );

    //! アーカイブの先頭アドレスを取得します。
    const void* GetArchiveDataStart() const { return mArchiveBlockHeader; }

protected:
    /**
     *  ファイル情報を設定します.
     *
     *  継承先のクラスで利用するメソッドです.
     * @param[in] file_info     ファイル情報です。
     * @param[in] start_offset  開始のオフセットです。
     * @param[in] length        長さです。
     */
    NW_INLINE static void SetArcFileInfo( ArcFileInfo* file_info, u32 start_offset, u32 length );

public:
    //! @name 定数です.
    //! @{
    static const u32 cArchiveVersion = 0x0100;          //!< 想定するレイアウトアーカイバのコンバータ＆ランタイムバージョン.
    static const u32 cArchiveEntryMax = 0x3fff;         //!< エントリの最大数
    static const u32 cTotalDataSizeMax = 0xffffffff;    //!< データブロックに格納できる最大長さ.
    static const u32 cFileNameTableAlign = 4;           //!< FileNameTableのアライメント.
    //! @}

    //! アーカイブブロックヘッダ
    struct ArchiveBlockHeader
    {
        char    signature[ 4 ];     //!< "SARC".
        u16     header_size;        //!< このブロックヘッダの大きさ.
        u16     byte_order;         //!< バイトオーダーマーク.
        u32     file_size;          //!< このファイル全体の大きさ(HeaderBlockを含む).
        u32     data_block_offset;  //!< ファイル先頭からデータブロックまでのオフセット.
        u16     version;            //!< 上位8ビットと下位8ビットに分けてバージョンを保存.
        u16     reserved;           //!< パディング.
    };

    //! ファイルアロケーションテーブルヘッダ
    struct FATBlockHeader
    {
        char    signature[ 4 ];     //!< "SFAT".
        u16     header_size;        //!< このブロックヘッダの大きさ.
        u16     file_num;           //!< ファイル数.
        u32     hash_key;           //!< ハッシュのキー.
    };

    //! ファイルアロケーションテーブルエントリ
    struct FATEntry
    {
        u32 hash;                   //!< ハッシュ.
        u32 name_offset;            //!< 上位8bitに同じハッシュ内のID(1から開始)、残りの24bitに名前の位置÷FNTのアライメント(4).
        u32 data_start_offset;      //!< データ開始位置.
        u32 data_end_offset;        //!< データ終了位置.
    };

    //! ファイルネームテーブルヘッダ
    struct FNTBlockHeader
    {
        char    signature[ 4 ];     //!< "SFNT".
        u16     header_size;        //!< このブロックヘッダの大きさ.
        u16     reserved;           //!< パディング.
    };

private:
    const ArchiveBlockHeader*     mArchiveBlockHeader;    //!< ArchiveBlockHeader(NULLのとき未prepare).
    const FATBlockHeader*         mFATBlockHeader;        //!< FATBlockHeader.
    const char*                   mFNTBlock;              //!< FNTBlockの先頭アドレス.
    ut::Array<FATEntry>           mFATEntrys;             //!< ファイルアロケーションテーブルのエントリ配列
    const u8*                     mDataBlock;             //!< データブロックのポインタ
    EndianTypes                   mEndianType;
};

//------------------------------------------------------------------------------
NW_INLINE void* ArcExtractor::GetFile( const char* file_path, ArcFileInfo* file_info )
{
    s32 id = ConvertPathToEntryID( file_path );
    if ( id < 0 )
    {
        return NULL;
    }
    return GetFileFast( id, file_info );
}

//------------------------------------------------------------------------------
NW_INLINE void ArcExtractor::SetArcFileInfo( ArcFileInfo* file_info, u32 start_offset, u32 length )
{
    NW_ASSERT_NOT_NULL( file_info );
    file_info->mStartOffset = start_offset;
    file_info->mLength      = length;
}


} // namespace lyt
} // namespace nw

#endif // NW_LYT_ARC_EXTRACTOR_H_
