﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_GFND_GRAPHICS_CONTEXT_H_
#define NW_GFND_GRAPHICS_CONTEXT_H_

#include <nw/gfnd/gfnd_Graphics.h>

namespace nw
{
namespace gfnd
{

//---------------------------------------------------------------------------
//! @brief 各種ステートの設定を一括で行うクラスです。
//---------------------------------------------------------------------------
class GraphicsContext
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ GraphicsContext();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    /* dtor */ virtual ~GraphicsContext() {}


    //---------------------------------------------------------------------------
    //! @brief        この Context オブジェクトが持つ情報をグラフィックスエンジンに適用します。
    //---------------------------------------------------------------------------
    void Apply() const;


    //---------------------------------------------------------------------------
    //! @brief        深度テスト／書き込みを有効にするかどうか設定します。
    //!
    //! @param[in]    testEnable  深度テストを有効にする場合、 true を指定します。
    //! @param[in]    writeEnable 深度書き込みを有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetDepthEnable( bool testEnable, bool writeEnable )
    {
        SetDepthTestEnable( testEnable );
        SetDepthWriteEnable( writeEnable );
    }

    //---------------------------------------------------------------------------
    //! @brief        深度テストを有効にするかどうか設定します。
    //!
    //! @param[in]    testEnable 有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetDepthTestEnable( bool testEnable )
    {
        m_DepthTestEnable = testEnable;
    }

    //---------------------------------------------------------------------------
    //! @brief        深度書き込みを有効にするかどうか設定します。
    //!
    //! @param[in]    writeEnable 有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetDepthWriteEnable( bool writeEnable )
    {
        m_DepthWriteEnable = writeEnable;
    }

    //---------------------------------------------------------------------------
    //! @brief        深度テストの比較関数を設定します。
    //!
    //! @param[in]    func     比較関数です。
    //---------------------------------------------------------------------------
    void SetDepthFunc( Graphics::DepthFunc func )
    {
        m_DepthFunc = func;
    }

    //---------------------------------------------------------------------------
    //! @brief        カリングの向きを設定します。
    //!
    //! @param[in]    mode     設定するカリングの向きです。
    //---------------------------------------------------------------------------
    void SetCullingMode( Graphics::CullingMode mode )
    {
        m_CullingMode = mode;
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理を有効にするかどうか設定します。
    //!
    //! @param[in]    enable 有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetBlendEnable( bool enable )
    {
        m_BlendEnable = enable;
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理のオペレータに掛ける係数を設定します。
    //!
    //! @param[in]    srcFactor 入力側の色に掛ける係数です。
    //! @param[in]    dstFactor 書き込み先のフレームバッファの色に掛ける係数です。
    //---------------------------------------------------------------------------
    void SetBlendFactor( Graphics::BlendFactor srcFactor, Graphics::BlendFactor dstFactor )
    {
        SetBlendFactorSrc( srcFactor );
        SetBlendFactorDst( dstFactor );
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理のオペレータに掛ける係数を RGB とアルファで別々に設定します。
    //!
    //! @param[in]    srcFactorRGB 入力側の色の RGB 成分に掛ける係数です。
    //! @param[in]    dstFactorRGB 書き込み先のフレームバッファの色の RGB 成分に掛ける係数です。
    //! @param[in]    srcFactorA   入力側の色のアルファ成分に掛ける係数です。
    //! @param[in]    dstFactorA   書き込み先のフレームバッファの色のアルファ成分に掛ける係数です。
    //---------------------------------------------------------------------------
    void SetBlendFactorSeparate(
        Graphics::BlendFactor srcFactorRGB,
        Graphics::BlendFactor dstFactorRGB,
        Graphics::BlendFactor srcFactorA,
        Graphics::BlendFactor dstFactorA
    )
    {
        SetBlendFactorSrcRGB( srcFactorRGB );
        SetBlendFactorDstRGB( dstFactorRGB );
        SetBlendFactorSrcAlpha( srcFactorA );
        SetBlendFactorDstAlpha( dstFactorA );
    }

    //---------------------------------------------------------------------------
    //! @brief        入力側のブレンド係数を設定します。
    //!
    //! @param[in]    factor    入力側のブレンド係数です。
    //---------------------------------------------------------------------------
    void SetBlendFactorSrc( Graphics::BlendFactor factor )
    {
        SetBlendFactorSrcRGB( factor );
        SetBlendFactorSrcAlpha( factor );
    }

    //---------------------------------------------------------------------------
    //! @brief        フレームバッファ側のブレンド係数を設定します。
    //!
    //! @param[in]    factor    フレームバッファ側のブレンド係数です。
    //---------------------------------------------------------------------------
    void SetBlendFactorDst( Graphics::BlendFactor factor )
    {
        SetBlendFactorDstRGB( factor );
        SetBlendFactorDstAlpha( factor );
    }

    //---------------------------------------------------------------------------
    //! @brief        入力側の RGB 成分のブレンド係数を設定します。
    //!
    //! @param[in]    factor    入力側の RGB 成分のブレンド係数です。
    //---------------------------------------------------------------------------
    void SetBlendFactorSrcRGB( Graphics::BlendFactor factor )
    {
        m_BlendFactorSrcRGB = factor;
    }

    //---------------------------------------------------------------------------
    //! @brief        入力側のアルファ成分のブレンド係数を設定します。
    //!
    //! @param[in]    factor    入力側のアルファ成分のブレンド係数です。
    //---------------------------------------------------------------------------
    void SetBlendFactorSrcAlpha( Graphics::BlendFactor factor )
    {
        m_BlendFactorSrcA = factor;
    }

    //---------------------------------------------------------------------------
    //! @brief        出力側の RGB 成分のブレンド係数を設定します。
    //!
    //! @param[in]    factor    出力側の RGB 成分のブレンド係数です。
    //---------------------------------------------------------------------------
    void SetBlendFactorDstRGB( Graphics::BlendFactor factor )
    {
        m_BlendFactorDstRGB = factor;
    }

    //---------------------------------------------------------------------------
    //! @brief        出力側のアルファ成分のブレンド係数を設定します。
    //!
    //! @param[in]    factor    出力側のアルファ成分のブレンド係数です。
    //---------------------------------------------------------------------------
    void SetBlendFactorDstAlpha( Graphics::BlendFactor factor )
    {
        m_BlendFactorDstA = factor;
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理の式を設定します。
    //!
    //! @param[in]    equation ブレンド処理の式です。
    //---------------------------------------------------------------------------
    void SetBlendEquation( Graphics::BlendEquation equation )
    {
        SetBlendEquationRGB( equation );
        SetBlendEquationAlpha( equation );
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理の式を RGB とアルファで別々に設定します。
    //!
    //! @param[in]    equationRGB RGB 成分の計算に用いる式です。
    //! @param[in]    equationA   アルファ成分の計算に用いる式です。
    //---------------------------------------------------------------------------
    void SetBlendEquationSeparate( Graphics::BlendEquation equationRGB, Graphics::BlendEquation equationA )
    {
        SetBlendEquationRGB( equationRGB );
        SetBlendEquationAlpha( equationA );
    }

    //---------------------------------------------------------------------------
    //! @brief        RGB 成分のブレンド式を設定します。
    //!
    //! @param[in]    equation RGB 成分の計算に用いる式です。
    //---------------------------------------------------------------------------
    void SetBlendEquationRGB( Graphics::BlendEquation equation )
    {
        m_BlendEquationRGB = equation;
    }

    //---------------------------------------------------------------------------
    //! @brief        アルファ成分のブレンド式を設定します。
    //!
    //! @param[in]    equation アルファ成分の計算に用いる式です。
    //---------------------------------------------------------------------------
    void SetBlendEquationAlpha( Graphics::BlendEquation equation )
    {
        m_BlendEquationA = equation;
    }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド処理の色定数 (ConstantColor) を設定します。
    //!
    //! @param[in]    color    設定する色定数です。
    //---------------------------------------------------------------------------
    void SetBlendConstantColor( const nw::ut::Color4u8& color )
    {
        m_BlendConstantColor = color;
    }

    //---------------------------------------------------------------------------
    //! @brief        アルファテストの有効／無効を切り替えます。
    //!
    //! @param[in]    enable   有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetAlphaTestEnable( bool enable )
    {
        m_AlphaTestEnable = enable;
    }

    //---------------------------------------------------------------------------
    //! @brief        アルファテストの計算式と参照値を設定します。
    //!
    //! @param[in]    func     アルファテストの計算式を指定します。
    //! @param[in]    ref      アルファテストの参照値を指定します。
    //---------------------------------------------------------------------------
    void SetAlphaTestFunc( Graphics::AlphaFunc func, f32 ref )
    {
        m_AlphaTestFunc = func;
        m_AlphaTestRef = ref;
    }

    //---------------------------------------------------------------------------
    //! @brief        カラーマスクを設定します。
    //!
    //! @param[in]    r        R 成分へのマスクです。
    //! @param[in]    g        G 成分へのマスクです。
    //! @param[in]    b        B 成分へのマスクです。
    //! @param[in]    a        アルファ成分へのマスクです。
    //---------------------------------------------------------------------------
    void SetColorMask( bool r, bool g, bool b, bool a )
    {
        m_ColorMaskR = r;
        m_ColorMaskG = g;
        m_ColorMaskB = b;
        m_ColorMaskA = a;
    }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストの有効／無効を切り替えます。
    //!
    //! @param[in]    enable   有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetStencilTestEnable( bool enable )
    {
        m_StencilTestEnable = enable;
    }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストの計算式と参照値とマスク値を設定します。
    //!
    //! @param[in]    func     ステンシルテストの計算式を指定します。
    //! @param[in]    ref      ステンシルテストの参照値を指定します。
    //! @param[in]    mask     ステンシルテストのマスク値を指定します。
    //---------------------------------------------------------------------------
    void SetStencilTestFunc( Graphics::StencilFunc func, s32 ref, u32 mask )
    {
        m_StencilTestFunc = func;
        m_StencilTestRef = ref;
        m_StencilTestMask = mask;
    }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストの値の操作を設定します。
    //!
    //! @param[in]    fail     ステンシルテストに失敗したときの操作を指定します。
    //! @param[in]    zFail    ステンシルテストに成功し、デプステストに失敗したときの操作を指定します。
    //! @param[in]    zPass    デプステストに成功、またはデプステストが行われていないときの操作を指定します。
    //---------------------------------------------------------------------------
    void SetStencilTestOp( Graphics::StencilOp fail, Graphics::StencilOp zFail, Graphics::StencilOp zPass )
    {
        m_StencilOpFail = fail;
        m_StencilOpZFail = zFail;
        m_StencilOpZPass = zPass;
    }

    //---------------------------------------------------------------------------
    //! @brief        ポリゴンモードを設定します。
    //!
    //! @param[in]    front    表側のポリゴンモードです。
    //! @param[in]    back     裏側のポリゴンモードです。
    //---------------------------------------------------------------------------
    void SetPolygonMode( Graphics::PolygonMode front, Graphics::PolygonMode back )
    {
        m_PolygonModeFront = front;
        m_PolygonModeBack = back;
    }

    //---------------------------------------------------------------------------
    //! @brief        ポリゴンオフセットの有効／無効を設定します。
    //!
    //! @param[in]    fillFrontEnable 表側のポリゴンオフセットが有効かを指定します。
    //! @param[in]    fillBackEnable  裏側のポリゴンオフセットが有効かを指定します。
    //! @param[in]    pointLineEnable ポイント・ラインのポリゴンオフセットが有効かを指定します。
    //---------------------------------------------------------------------------
    void SetPolygonOffsetEnable( bool fillFrontEnable, bool fillBackEnable, bool pointLineEnable )
    {
        m_PolygonOffsetFrontEnable = fillFrontEnable;
        m_PolygonOffsetBackEnable = fillBackEnable;
        m_PolygonOffsetPointLineEnable = pointLineEnable;
    }


    //---------------------------------------------------------------------------
    //! @brief        深度テストの設定を取得します。
    //!
    //! @return       有効の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool GetDepthTestEnable() const { return m_DepthTestEnable; }

    //---------------------------------------------------------------------------
    //! @brief        深度書き込みの設定を取得します。
    //!
    //! @return       有効の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool GetDepthWriteEnable() const { return m_DepthWriteEnable; }

    //---------------------------------------------------------------------------
    //! @brief        深度テストの比較関数の設定を取得します。
    //!
    //! @return       比較関数の設定を返します。
    //---------------------------------------------------------------------------
    Graphics::DepthFunc GetDepthFunc() const { return m_DepthFunc; }

    //---------------------------------------------------------------------------
    //! @brief        カリング方向の設定を取得します。
    //!
    //! @return       カリング方向の設定を返します。
    //---------------------------------------------------------------------------
    Graphics::CullingMode GetCullingMode() const { return m_CullingMode; }

    //---------------------------------------------------------------------------
    //! @brief        ブレンドの設定を取得します。
    //!
    //! @return       有効の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool GetBlendEnable() const { return m_BlendEnable; }

    //---------------------------------------------------------------------------
    //! @brief        入力側の RGB 成分のブレンド係数を取得します。
    //!
    //! @return       ブレンド係数の設定を返します。
    //---------------------------------------------------------------------------
    Graphics::BlendFactor GetBlendFactorSrcRGB() const { return m_BlendFactorSrcRGB; }

    //---------------------------------------------------------------------------
    //! @brief        入力側のアルファ成分のブレンド係数を取得します。
    //!
    //! @return       ブレンド係数の設定を返します。
    //---------------------------------------------------------------------------
    Graphics::BlendFactor GetBlendFactorSrcAlpha() const { return m_BlendFactorSrcA; }

    //---------------------------------------------------------------------------
    //! @brief        フレームバッファ側の RGB 成分のブレンド係数を取得します。
    //!
    //! @return       ブレンド係数の設定を返します。
    //---------------------------------------------------------------------------
    Graphics::BlendFactor GetBlendFactorDstRGB() const { return m_BlendFactorDstRGB; }

    //---------------------------------------------------------------------------
    //! @brief        フレームバッファ側のアルファ成分のブレンド係数を取得します。
    //!
    //! @return       ブレンド係数の設定を返します。
    //---------------------------------------------------------------------------
    Graphics::BlendFactor GetBlendFactorDstAlpha() const { return m_BlendFactorDstA; }

    //---------------------------------------------------------------------------
    //! @brief        RGB 成分のブレンド式を取得します。
    //!
    //! @return       ブレンド式の設定を返します。
    //---------------------------------------------------------------------------
    Graphics::BlendEquation GetBlendEquationRGB() const { return m_BlendEquationRGB; }

    //---------------------------------------------------------------------------
    //! @brief        アルファ成分のブレンド式を取得します。
    //!
    //! @return       ブレンド式の設定を返します。
    //---------------------------------------------------------------------------
    Graphics::BlendEquation GetBlendEquationAlpha() const { return m_BlendEquationA; }

    //---------------------------------------------------------------------------
    //! @brief        ブレンド時のコンスタントカラーを取得します。
    //!
    //! @return       コンスタントカラーの設定を返します。
    //---------------------------------------------------------------------------
    const nw::ut::Color4u8& GetBlendConstantColor() const { return m_BlendConstantColor; }

    //---------------------------------------------------------------------------
    //! @brief        アルファテストの設定を取得します。
    //!
    //! @return       有効の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool GetAlphaTestEnable() const { return m_AlphaTestEnable; }

    //---------------------------------------------------------------------------
    //! @brief        アルファテストの比較関数を取得します。
    //!
    //! @return       比較関数の設定を返します。
    //---------------------------------------------------------------------------
    Graphics::AlphaFunc GetAlphaTestFunc() const { return m_AlphaTestFunc; }

    //---------------------------------------------------------------------------
    //! @brief        アルファテストの参照値を取得します。
    //!
    //! @return       参照値の設定を返します。
    //---------------------------------------------------------------------------
    f32 GetAlphaTestRef() const { return m_AlphaTestRef; }

    //---------------------------------------------------------------------------
    //! @brief        カラーマスクの R 成分の値を取得します。
    //!
    //! @return       カラーマスクの R 成分の設定を返します。
    //---------------------------------------------------------------------------
    bool GetColorMaskR() const { return m_ColorMaskR; }

    //---------------------------------------------------------------------------
    //! @brief        カラーマスクの G 成分の値を取得します。
    //!
    //! @return       カラーマスクの G 成分の設定を返します。
    //---------------------------------------------------------------------------
    bool GetColorMaskG() const { return m_ColorMaskG; }

    //---------------------------------------------------------------------------
    //! @brief        カラーマスクの B 成分の値を取得します。
    //!
    //! @return       カラーマスクの B 成分の設定を返します。
    //---------------------------------------------------------------------------
    bool GetColorMaskB() const { return m_ColorMaskB; }

    //---------------------------------------------------------------------------
    //! @brief        カラーマスクのアルファ成分の値を取得します。
    //!
    //! @return       カラーマスクのアルファ成分の設定を返します。
    //---------------------------------------------------------------------------
    bool GetColorMaskA() const { return m_ColorMaskA; }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストの設定を取得します。
    //!
    //! @return       有効の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool GetStencilTestEnable() const { return m_StencilTestEnable; }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストの比較関数を取得します。
    //!
    //! @return       比較関数の設定を返します。
    //---------------------------------------------------------------------------
    Graphics::StencilFunc GetStencilTestFunc() const { return m_StencilTestFunc; }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストの参照値を取得します。
    //!
    //! @return       参照値の設定を返します。
    //---------------------------------------------------------------------------
    s32 GetStencilTestRef() const { return m_StencilTestRef; }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストのマスクを取得します。
    //!
    //! @return       マスクの設定を返します。
    //---------------------------------------------------------------------------
    u32 GetStencilTestMask() const { return m_StencilTestMask; }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストに失敗したときの操作設定を取得します。
    //!
    //! @return       操作設定を返します。
    //---------------------------------------------------------------------------
    Graphics::StencilOp GetStencilTestOpFail() const { return m_StencilOpFail; }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストに成功し、 Z テストに失敗したときの操作設定を取得します。
    //!
    //! @return       操作設定を返します。
    //---------------------------------------------------------------------------
    Graphics::StencilOp GetStencilTestOpZFail() const { return m_StencilOpZFail; }

    //---------------------------------------------------------------------------
    //! @brief        ステンシルテストと Z テストに通過したときの操作設定を取得します。
    //!
    //! @return       操作設定を返します。
    //---------------------------------------------------------------------------
    Graphics::StencilOp GetStencilTestOpZPass() const { return m_StencilOpZPass; }

    //---------------------------------------------------------------------------
    //! @brief        表側のポリゴンモードを取得します。
    //!
    //! @return       ポリゴンモードを返します。
    //---------------------------------------------------------------------------
    Graphics::PolygonMode GetPolygonModeFront() const { return m_PolygonModeFront; };

    //---------------------------------------------------------------------------
    //! @brief        裏側のポリゴンモードを取得します。
    //!
    //! @return       ポリゴンモードを返します。
    //---------------------------------------------------------------------------
    Graphics::PolygonMode GetPolygonModeBack() const { return m_PolygonModeBack; };

    //---------------------------------------------------------------------------
    //! @brief        表側のポリゴンオフセットが有効かを取得します。
    //!
    //! @return       有効の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool GetPolygonOffsetFrontEnable() const { return m_PolygonOffsetFrontEnable; };

    //---------------------------------------------------------------------------
    //! @brief        裏側のポリゴンオフセットが有効かを取得します。
    //!
    //! @return       有効の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool GetPolygonOffsetBackEnable() const { return m_PolygonOffsetBackEnable; };

    //---------------------------------------------------------------------------
    //! @brief        裏側のポリゴンオフセットが有効かを取得します。
    //!
    //! @return       有効の場合、 true を返します。
    //---------------------------------------------------------------------------

    //---------------------------------------------------------------------------
    //! @brief        ポイント・ラインのポリゴンオフセットが有効かを取得します。
    //!
    //! @return       有効の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool GetPolygonOffsetPointLineEnable() const { return m_PolygonOffsetPointLineEnable; };


private:
    bool                    m_DepthTestEnable;      //!< デプステストのON/OFF。
    bool                    m_DepthWriteEnable;     //!< デプス書き込みのON/OFF。
    Graphics::DepthFunc     m_DepthFunc;            //!< デプステスト関数。
    Graphics::CullingMode   m_CullingMode;          //!< カリングモードの設定。
    bool                    m_BlendEnable;          //!< ブレンドのON/OFF。
    Graphics::BlendFactor   m_BlendFactorSrcRGB;    //!< 入力側ブレンド係数RGB。
    Graphics::BlendFactor   m_BlendFactorSrcA;      //!< 入力側ブレンド係数A。
    Graphics::BlendFactor   m_BlendFactorDstRGB;    //!< フレームバッファ側ブレンド係数RGB。
    Graphics::BlendFactor   m_BlendFactorDstA;      //!< フレームバッファ側ブレンド係数A。
    Graphics::BlendEquation m_BlendEquationRGB;     //!< RGB成分ブレンド式。
    Graphics::BlendEquation m_BlendEquationA;       //!< アルファ成分ブレンド式。
    nw::ut::Color4u8        m_BlendConstantColor;   //!< コンスタントカラー。
    bool                    m_AlphaTestEnable;      //!< アルファテストON/OFF。
    Graphics::AlphaFunc     m_AlphaTestFunc;        //!< アルファテスト関数。
    f32                     m_AlphaTestRef;         //!< アルファテスト参照値。
    bool                    m_ColorMaskR;           //!< カラーマスクR。
    bool                    m_ColorMaskG;           //!< カラーマスクG。
    bool                    m_ColorMaskB;           //!< カラーマスクB。
    bool                    m_ColorMaskA;           //!< カラーマスクA。
    bool                    m_StencilTestEnable;    //!< ステンシルテストON/OFF。
    Graphics::StencilFunc   m_StencilTestFunc;      //!< ステンシルテスト関数。
    s32                     m_StencilTestRef;       //!< ステンシルテスト参照値。
    u32                     m_StencilTestMask;      //!< ステンシルテストマスク値。
    Graphics::StencilOp     m_StencilOpFail;        //!< ステンシル更新(Fail時)。
    Graphics::StencilOp     m_StencilOpZFail;       //!< ステンシル更新(ZFail時)。
    Graphics::StencilOp     m_StencilOpZPass;       //!< ステンシル更新(ZPass時)。
    Graphics::PolygonMode   m_PolygonModeFront;     //!< ポリゴンモード(表)。
    Graphics::PolygonMode   m_PolygonModeBack;      //!< ポリゴンモード(裏)。
    bool                    m_PolygonOffsetFrontEnable; //!< ポリゴンオフセットが有効か否か(表)。
    bool                    m_PolygonOffsetBackEnable;  //!< ポリゴンオフセットが有効か否か(裏)。
    bool                    m_PolygonOffsetPointLineEnable;  //!< ラインのオフセットが有効か否か。
};

} // namespace nw::gfnd
} // namespace nw

#endif // NW_GFND_GRAPHICS_CONTEXT_H_
