﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESSKELETON_H_
#define NW_G3D_RES_RESSKELETON_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/math/g3d_Vector3.h>
#include <nw/g3d/math/g3d_Matrix34.h>
#include <nw/g3d/math/g3d_Quaternion.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/res/g3d_ResUserData.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d {namespace res {

class ResFile;

//--------------------------------------------------------------------------------------------------

//! @brief ボーンの構造体です。
struct ResBoneData
{
    BinString ofsName;
    u16 index; //!< ボーン配列中のインデックス
    u16 parentIndex;
    s16 smoothMtxIndex;
    s16 rigidMtxIndex;
    u16 billboardIndex; // 階層ビルボードの場合に影響を受けるボーンのインデックス
    u16 numUserData;

    bit32 flag;
    Vec3 scale;
    union
    {
        Quat quat;
        Vec3 euler;
    } rotate;
    Vec3 translate;

    Offset ofsUserDataDic;
};



//! @brief ボーンのリソースです。
class ResBone : private ResBoneData
{
    NW_G3D_RES_COMMON(ResBone);

public:
    //! @brief 無効なボーンインデックスを示す値です。
    static const int InvalidBoneIndex = 0xFFFF;

    //! @brief ボーンに関するフラグです。
    enum Flag
    {
        //! @brief ビジビリティの初期値です。
        VISIBILITY                  = 0x1 << 0,

        // RotateMode
        // スケルトンと同一。不正呼び出しをチェックするためにボーンにも持つ。
        //! @briefprivate
        ROT_SHIFT                   = 12,

        //! @brief Quaternion 方式で回転を計算します。
        ROT_QUAT                    = 0x0 << ROT_SHIFT,

        //! @brief Euler 角方式で回転を計算します。
        ROT_EULER_XYZ               = 0x1 << ROT_SHIFT,

        // reserved
        //! @briefprivate
        ROT_MASK                    = 0x7 << ROT_SHIFT,

        // ビルボード
        //! @briefprivate
        BILLBOARD_SHIFT             = 16,

        //! @brief ビルボード計算をしません。
        BILLBOARD_NONE              = 0x0 << BILLBOARD_SHIFT,

        //! @brief 階層ビルボードの子として計算を行います。
        BILLBOARD_CHILD             = 0x1 << BILLBOARD_SHIFT,

        //! @brief World 方式でビルボード計算を行います。
        BILLBOARD_WORLD_VIEWVECTOR  = 0x2 << BILLBOARD_SHIFT,

        //! @brief WorldViewpoint 方式でビルボード計算を行います。
        BILLBOARD_WORLD_VIEWPOINT   = 0x3 << BILLBOARD_SHIFT,

        //! @brief Screen 方式でビルボード計算を行います。
        BILLBOARD_SCREEN_VIEWVECTOR = 0x4 << BILLBOARD_SHIFT,

        //! @brief ScreenViewpoint 方式でビルボード計算を行います。
        BILLBOARD_SCREEN_VIEWPOINT  = 0x5 << BILLBOARD_SHIFT,

        //! @brief YAxis 方式でビルボード計算を行います。
        BILLBOARD_YAXIS_VIEWVECTOR  = 0x6 << BILLBOARD_SHIFT,

        //! @brief YAxisViewpoint 方式でビルボード計算を行います。
        BILLBOARD_YAXIS_VIEWPOINT   = 0x7 << BILLBOARD_SHIFT,

        //! @briefprivate
        BILLBOARD_MAX               = BILLBOARD_YAXIS_VIEWPOINT,

        //! @briefprivate
        BILLBOARD_MASK              = 0x7 << BILLBOARD_SHIFT,

        // ボーンのトランスフォーム
        //! @brief MayaSSC で計算します。
        SEGMENT_SCALE_COMPENSATE    = 0x1 << 23,

        //! @brief Scale が等方です。
        SCALE_UNIFORM               = 0x1 << 24,

        //! @brief Scale の体積が１です。
        SCALE_VOLUME_ONE            = 0x1 << 25,

        //! @brief Rotate が 0 です。
        ROTATE_ZERO                 = 0x1 << 26,

        //! @brief Translate が 0 です。
        TRANSLATE_ZERO              = 0x1 << 27,

        //! @brief Scale が１です。
        SCALE_ONE                   = SCALE_VOLUME_ONE | SCALE_UNIFORM,

        //! @brief Rotate, Translate が 0 です。
        ROTTRANS_ZERO               = ROTATE_ZERO | TRANSLATE_ZERO,

        //! @brief トランスフォームが単位行列です。
        IDENTITY                    = SCALE_ONE | ROTATE_ZERO | TRANSLATE_ZERO,

        //! @briefprivate
        TRANSFORM_MASK              = SEGMENT_SCALE_COMPENSATE | IDENTITY,

        // ボーンの累積トランスフォーム
        //! @briefprivate
        HIERARCHY_SHIFT             = 4,

        //! @brief 累積 Scale が等方です。
        HI_SCALE_UNIFORM            = SCALE_UNIFORM     << HIERARCHY_SHIFT,

        //! @brief 累積 Scale の体積が１です。
        HI_SCALE_VOLUME_ONE         = SCALE_VOLUME_ONE  << HIERARCHY_SHIFT,

        //! @brief 累積 Rotate が 0 です。
        HI_ROTATE_ZERO              = ROTATE_ZERO       << HIERARCHY_SHIFT,

        //! @brief 累積 Translate が 0 です。
        HI_TRANSLATE_ZERO           = TRANSLATE_ZERO    << HIERARCHY_SHIFT,

        //! @brief 累積 Scale が１です。
        HI_SCALE_ONE                = SCALE_ONE         << HIERARCHY_SHIFT,

        //! @brief 累積 Rotate, Translate が 0 です。
        HI_ROTTRANS_ZERO            = ROTTRANS_ZERO     << HIERARCHY_SHIFT,

        //! @brief 累積トランスフォームが単位行列です。
        HI_IDENTITY                 = IDENTITY          << HIERARCHY_SHIFT
    };

    enum
    {
        //! @brief 階層ビルボードの影響を受けるボーンがありません。
        BB_INDEX_NONE               = 0xFFFF
    };

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief スケルトン内でのインデクスを取得します。
    int GetIndex() const { return ref().index; }

    NW_G3D_RES_FIELD_STRING_DECL(Name);

    //! @brief スムーススキニング用の行列パレットにおけるインデクスを取得します。
    int GetSmoothMtxIndex() const { return ref().smoothMtxIndex; }

    //! @brief リジッドスキニング用の行列パレットにおけるインデクスを取得します。
    int GetRigidMtxIndex() const { return ref().rigidMtxIndex; }

    //! @brief 親ボーンのインデクスを取得します。
    int GetParentIndex() const { return ref().parentIndex; }

    //! @brief 回転の計算方式を取得します。
    bit32 GetRotateMode() const { return ref().flag & ROT_MASK; }

    //! @brief ビルボードの計算方式を取得します。
    bit32 GetBillboardMode() const { return ref().flag & BILLBOARD_MASK; }

    //! @brief スケールを取得します。
    Vec3& GetScale() { return ref().scale; }

    //! @brief スケールを取得します。
    const Vec3& GetScale() const { return ref().scale; }

    //! @brief 移動値を取得します。
    Vec3& GetTranslate() { return ref().translate; }

    //! @brief 移動値を取得します。
    const Vec3& GetTranslate() const { return ref().translate; }

    //! @brief Euler で回転を取得します。
    Vec3& GetRotateEuler()
    {
        NW_G3D_ASSERT(ROT_EULER_XYZ == GetRotateMode());
        return ref().rotate.euler;
    }

    //! @brief Euler で回転を取得します。
    const Vec3& GetRotateEuler() const
    {
        NW_G3D_ASSERT(ROT_EULER_XYZ == GetRotateMode());
        return ref().rotate.euler;
    }

    //! @brief Quaternion で回転を取得します。
    Quat& GetRotateQuat()
    {
        NW_G3D_ASSERT(ROT_QUAT == GetRotateMode());
        return ref().rotate.quat;
    }

    //! @brief Quaternion で回転を取得します。
    const Quat& GetRotateQuat() const
    {
        NW_G3D_ASSERT(ROT_QUAT == GetRotateMode());
        return ref().rotate.quat;
    }

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief スケルトンの構造体です。
struct ResSkeletonData
{
    BinaryBlockHeader blockHeader;

    bit32 flag;
    u16 numBone;
    u16 numSmoothMtx;
    u16 numRigidMtx;
    u16 reserved;

    Offset ofsBoneDic;
    Offset ofsBoneArray;
    Offset ofsMtxToBoneTable;
    Offset ofsInvModelMatrixArray; //!< 頂点からボーンローカル座標系への変換行列

    BinPtr pUserPtr;
};

//! @brief スケルトンのリソースです。
class ResSkeleton : private ResSkeletonData
{
    NW_G3D_RES_COMMON(ResSkeleton);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'S', 'K', 'L') };

    //! @brief スケルトンに関するフラグです。
    enum Flag
    {
        // ScaleMode
        //! @briefprivate
        SCALE_SHIFT     = 8,

        //! @brief スケール計算をしません。
        SCALE_NONE      = 0x0 << SCALE_SHIFT,

        //! @brief 標準的な方法でスケールの計算をします。
        SCALE_STD       = 0x1 << SCALE_SHIFT,

        //! @brief Maya 方式でスケールの計算をします。
        SCALE_MAYA      = 0x2 << SCALE_SHIFT,

        //! @brief Softimage 方式でスケールの計算をします。
        SCALE_SOFTIMAGE = 0x3 << SCALE_SHIFT,

        //! @briefprivate
        SCALE_MASK      = 0x3 << SCALE_SHIFT,

        // RotateMode
        //! @briefprivate
        ROT_SHIFT       = ResBone::ROT_SHIFT,

        //! @brief Quaternion 方式で回転を計算します。
        ROT_QUAT        = ResBone::ROT_QUAT,

        //! @brief Euler 角方式で回転を計算します。
        ROT_EULER_XYZ   = ResBone::ROT_EULER_XYZ,

        // reserved
        //! @briefprivate
        ROT_MASK        = ResBone::ROT_MASK,

        //! @briefprivate
        TRANSFORM_MASK  = SCALE_MASK | ROT_MASK
    };

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief スムーススキニングのマトリクス数を取得します。
    int GetSmoothMtxCount() const { return ref().numSmoothMtx; }

    //! @brief リジッドスキニングのマトリクス数を取得します。
    int GetRigidMtxCount() const { return ref().numRigidMtx; }

    //! @brief マトリクス数を取得します。
    //!
    //! GetSmoothMtxCount() と GetRigidMtxCount() を合わせた数になります。
    //!
    int GetMtxCount() const { return GetSmoothMtxCount() + GetRigidMtxCount(); }

    //! @brief スケールの計算方式を取得します。
    bit32 GetScaleMode() const { return ref().flag & SCALE_MASK; }

    //! @brief 回転の計算方式を取得します。
    bit32 GetRotateMode() const { return ref().flag & ROT_MASK; }

    //! @brief 指定したボーン配下のボーンツリーで最後のボーンの次のインデックスを取得します。
    //!
    //! ボーンは深さ優先順で並んでいるため、EndIndex より小さいインデックスのボーンを辿ることで
    //! 指定したボーン配下のボーンツリーを辿ることができます。
    //!
    int GetBranchEndIndex(int boneIndex) const;

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { ref().pUserPtr.set_ptr(pUserPtr); }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return ref().pUserPtr.to_ptr<T>(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return ref().pUserPtr.to_ptr<T>(); }

    //! @briefprivate ビルボードの設定を更新します。
    //!
    //! ボーンの階層構造を辿って階層ビルボードの設定を行います。
    //! インスタンスの初期化に影響するため動的に変更はできません。
    //!
    void UpdateBillboardMode();

    //@}

    //----------------------------------------
    //! @name ボーン
    //@{

    NW_G3D_RES_FIELD_CLASS_NAMED_ARRAY_DECL_DETAIL(ResBone, Bone, "")

    //@}
};

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS

#endif // NW_G3D_RES_RESSKELETON_H_
