﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESSHAPEANIM_H_
#define NW_G3D_RES_RESSHAPEANIM_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/res/g3d_ResAnimCurve.h>
#include <nw/g3d/res/g3d_ResModel.h>
#include <nw/g3d/res/g3d_Binding.h>

namespace nw { namespace g3d { namespace res {

class BindCallback;
class ResFile;

//--------------------------------------------------------------------------------------------------

//! @brief 頂点シェイプアニメーションの構造体です。
struct ResVertexShapeAnimData
{
    u16 numCurve;
    u16 numKeyShapeAnim;
    s32 beginCurve; // ShapeAnim 内での先頭 Curve のインデックス
    s32 beginKeyShapeAnim; // ShapeAnim 内での KeyShape のインデックス

    BinString ofsName;

    Offset ofsKeyShapeAnimInfoArray; // 先頭はベースシェイプの情報が挿入されている
    Offset ofsCurveArray;
    Offset ofsBaseValueArray; // ベースシェイプの値は持たない
};

//! @brief 頂点シェイプアニメーションのリソースです。
class ResVertexShapeAnim : private ResVertexShapeAnimData
{
    NW_G3D_RES_COMMON(ResVertexShapeAnim);

public:
    //! @brief キーシェイプアニメーションの要素情報です。
    struct KeyShapeAnimInfo
    {
        s8 curveIndex; // VertexAnim 内での Curve のインデックス
        s8 subbindIndex; // Shape 内での KeyShape のインデックス
        u16 reserved;
        BinString ofsName;
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{
    //! @brief 事前に ResShape と関連付けを行います。
    //!
    //! リソース同士を PreBind によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    BindResult PreBind(const ResShape* target);

    //! @brief 事前に ResShape と関連付けが成功するか、チェックのみを行います。
    BindResult BindCheck(const ResShape* target) const;

    //! @brief 初期化を行います。
    void Init(float* pResult) const;

    //! @brief Init() 関数の別名関数です。
    void Initialize(float* pResult) const
    {
        Init(pResult);
    }

    //@}

    //----------------------------------------
    //! @name 評価
    //@{

    //! @brief 頂点属性アニメーションを評価します。
    void Eval(float* pResult, float frame, const s8* pSubBindIndex) const;

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(float* pResult, float frame, const s8* pSubBindIndex) const
    {
        Eval(pResult, frame, pSubBindIndex);
    }

    //! @brief 頂点属性アニメーションを評価します。
    void Eval(float* pResult, float frame, const s8* pSubBindIndex,
        AnimFrameCache* pFrameCache) const;

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(float* pResult, float frame, const s8* pSubBindIndex,
        AnimFrameCache* pFrameCache) const
    {
        Eval(pResult, frame, pSubBindIndex, pFrameCache);
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResAnimCurve, Curve, GetName())

    //! @brief アニメーション数を取得します。
    int GetKeyShapeAnimCount() const { return ref().numKeyShapeAnim; }

    //@}

};

//--------------------------------------------------------------------------------------------------

//! @brief シェイプアニメーションの構造体です。
struct ResShapeAnimData
{
    BinaryBlockHeader blockHeader;
    BinString ofsName;
    BinString ofsPath;

    bit16 flag;
    u16 numUserData;
    s32 numFrame;
    u16 numVertexShapeAnim; // BindTable のサイズに影響
    u16 numKeyShapeAnim; // SubBindTable のサイズに影響
    u16 numCurve; // Context のサイズに影響
    u16 reserved;
    u32 bakedSize;

    Offset ofsBindModel;
    Offset ofsBindIndexArray; // Shape のインデックス
    Offset ofsVertexShapeAnimArray;
    Offset ofsUserDataDic;
};

//! @brief シェイプアニメーションのリソースです。
class ResShapeAnim : private ResShapeAnimData
{
    NW_G3D_RES_COMMON(ResShapeAnim);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'S', 'H', 'A') };

    //! @brief シェイプアニメーションに関するフラグです。
    enum Flag
    {
        //! @brief カーブがベイク済みです。
        CURVE_BAKED         = AnimFlag::CURVE_BAKED,

        //! @brief ループすることを表します。
        PLAYPOLICY_LOOP     = AnimFlag::PLAYPOLICY_LOOP
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 事前に ResModel と関連付けを行います。
    //!
    //! リソース同士を PreBind によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    BindResult PreBind(const ResModel* pModel);

    //! @brief 事前に ResModel と関連付けが成功するか、チェックのみを行います。
    BindResult BindCheck(const ResModel* target) const;

    //! @brief カーブをコマ化します。
    bool BakeCurve(void* pBuffer, size_t bufferSize);

    //! @brief カーブに戻します。
    void* ResetCurve();

    //! @brief シェイプアニメーションをリセットします。
    void Reset();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief ループするかどうかを取得します。
    bool IsLooped() const { return 0 != (ref().flag & PLAYPOLICY_LOOP); }

    //! @brief カーブがベイクされているかどうかを取得します。
    bool IsCurveBaked() const { return 0 != (ref().flag & CURVE_BAKED); }

    //! @brief フレーム数を取得します。
    int GetFrameCount() const { return ref().numFrame; }

    //! @brief 頂点属性アニメーション数を取得します。
    int GetKeyShapeAnimCount() const { return ref().numKeyShapeAnim; }

    //! @brief カーブ数を取得します。
    int GetCurveCount() const { return ref().numCurve; }

    //! @brief コマ化に必要なバッファサイズを取得します。
    size_t GetBakedSize() const { return ref().bakedSize; }

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_STRING_DECL(Path)
    NW_G3D_RES_FIELD_CLASS_DECL(ResModel, BindModel)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResVertexShapeAnim, VertexShapeAnim, GetName())

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

    //@}
};

}}} // namespace nw::g3d::res

#endif // NW_G3D_RES_RESSHAPEANIM_H_
