﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESSHAPE_H_
#define NW_G3D_RES_RESSHAPE_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/math/g3d_Vector3.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

class ResFile;

//--------------------------------------------------------------------------------------------------

//! @brief バッファの構造体です。
struct ResBufferData
{
    GfxBuffer_t gfxBuffer;
    Offset ofsData;
};

//! @brief バッファのリソースです。
class ResBuffer : private ResBufferData
{
    NW_G3D_RES_COMMON(ResBuffer);

public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief バッファをセットアップします。
    void Setup();

    //! @brief バッファをクリーンアップします。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief バッファを取得します。
    GfxBuffer* GetGfxBuffer()
    {
        return GfxBuffer::DownCast(&ref().gfxBuffer);
    }

    //! @brief バッファを取得します。
    const GfxBuffer* GetGfxBuffer() const
    {
        return GfxBuffer::DownCast(&ref().gfxBuffer);
    }

    //! @brief バッファのサイズを取得します。
    size_t GetSize() const { return ref().gfxBuffer.size; }

    //! @brief バッファへのポインタを取得します。
    void* GetData() { return ref().ofsData.to_ptr(); }

    //! @brief バッファへのポインタを取得します。
    const void* GetData() const { return ref().ofsData.to_ptr(); }

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief 頂点属性の構造体です。
struct ResVtxAttribData
{
    BinString ofsName;

    u8 bufferIndex; //!< シェイプ内での頂点バッファのインデックス
    u8 reserved;
    u16 offset;

    GX2AttribFormat format;
};

//! @brief 頂点属性のリソースです。
class ResVtxAttrib : private ResVtxAttribData
{
    NW_G3D_RES_COMMON(ResVtxAttrib);

public:
    //----------------------------------------
    //! @name 取得
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)

    //! @brief 頂点バッファインデクスを取得します。
    int GetBufferIndex() const { return ref().bufferIndex; }

    //! @brief 頂点属性のフォーマットを取得します。
    GX2AttribFormat GetFormat() const { return ref().format; }

    //! @brief バッファ内のオフセットを取得します。
    u32 GetOffset() const { return ref().offset; }

    //@}
};

typedef ResVtxAttrib ResVertexAttr;

//--------------------------------------------------------------------------------------------------

//! @brief 頂点情報の構造体です。
struct ResVertexData
{
    BinaryBlockHeader blockHeader;

    u8 numVtxAttrib;
    u8 numVtxBuffer;
    u16 index; //!< モデル内でのインデックス
    u32 count;
    u8 vtxSkinCount; //!< スキニングで1頂点に影響するボーンの数
    u8 reserved[3];

    Offset ofsVtxAttribArray;
    Offset ofsVtxAttribDic;
    Offset ofsVtxBufferArray;

    BinPtr pUserPtr;
};

//! @brief 頂点情報のリソースです。
class ResVertex : private ResVertexData
{
    NW_G3D_RES_COMMON(ResVertex);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'V', 'T', 'X') };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 頂点情報をセットアップします。
    void Setup();

    //! @brief 頂点情報をクリーンアップします。
    void Cleanup();

    //! @brief 頂点情報をリセットします。
    void Reset();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief モデル内でのインデクスを取得します。
    int GetIndex() const { return static_cast<int>(ref().index); }

    //! @brief 頂点数を取得します。
    u32 GetCount() const { return ref().count; }

    //! @brief 1頂点を計算するのに使用するボーン数を取得します。
    int GetVtxSkinCount() const { return ref().vtxSkinCount; }

    //! @brief GetVtxSkinCount() の別名関数です。
    int GetVertexSkinCount() const { return GetVtxSkinCount(); }

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { ref().pUserPtr.set_ptr(pUserPtr); }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return ref().pUserPtr.to_ptr<T>(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return ref().pUserPtr.to_ptr<T>(); }

    //@}

    //----------------------------------------
    //! @name 頂点属性
    //@{

    NW_G3D_RES_FIELD_CLASS_NAMED_ARRAY_DECL_DETAIL(ResVtxAttrib, VtxAttrib, "")
    NW_G3D_RES_FIELD_CLASS_NAMED_ARRAY_DECL_DETAIL_WITH_FUNCNAME(ResVtxAttrib, VertexAttr, VtxAttrib, "");

    //@}

    //----------------------------------------
    //! @name 頂点バッファ
    //@{

    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResBuffer, VtxBuffer, "")
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL_WITH_FUNCNAME(ResBuffer, VertexBuffer, VtxBuffer, "")

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief サブメッシュの構造体です。
struct ResSubMeshData
{
    u32 offset; //!< インデックスバッファの先頭からのバイトオフセット
    u32 count;
};

//! @brief サブメッシュです。
//!
//! ベース頂点インデックスを利用することによりインデックスバッファを分割し、
//! インデックスの型を小さくすることに使用します。
//! ソートの単位には影響しません。
//!
class ResSubMesh : private ResSubMeshData
{
    NW_G3D_RES_COMMON(ResSubMesh);

public:
    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief インデクスバッファの先頭からのバイトオフセットです。
    u32 GetOffset() const { return ref().offset; }

    //! @brief GetOffset() の位置から描画する頂点インデクス数を取得します。
    u32 GetCount() const { return ref().count; }

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief メッシュの構造体です。
struct ResMeshData
{
    GX2PrimitiveType primType;
    GX2IndexFormat format;
    u32 count;

    u16 numSubMesh;
    u16 reserved;
    Offset ofsSubMeshArray;
    Offset ofsIdxBuffer;
    u32 offset; // 頂点バッファ内でのオフセット
};

//! @brief メッシュのリソースです。インデックスバッファを持ちます。
class ResMesh : private ResMeshData
{
    NW_G3D_RES_COMMON(ResMesh);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief すべての SubMesh を描画します。
    void Draw(int instances = 1) const
    {
        DrawSubMesh(0, GetSubMeshCount(), instances);
    }

    //! @brief 指定した範囲の SubMesh を描画します。
    void DrawSubMesh(int submeshIndex, int submeshCount, int instances = 1) const;

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief 形状タイプを取得します。
    GX2PrimitiveType GetPrimitiveType() const { return ref().primType; }

    //! @brief インデクスデータのフォーマットを取得します。
    GX2IndexFormat GetIndexFormat() const { return ref().format; }

    //! @brief 頂点インデクス数を取得します。
    u32 GetCount() const { return ref().count; }

    //@}

    //----------------------------------------
    //! @name サブメッシュ
    //@{

    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResSubMesh, SubMesh, "")

    //@}

    //----------------------------------------
    //! @name インデクスバッファ
    //@{

    NW_G3D_RES_FIELD_CLASS_DECL(ResBuffer, IdxBuffer);

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief キーシェイプの構造体です。
struct ResKeyShapeData
{
    u8 targetAttribIndices[20]; // position, normal, tangent4, binormal4, color8, padding2 の順です。
    u8 ofsTargetAttribIndex[4]; // 廃止予定
};

//! @brief キーシェイプのリソースです。
class ResKeyShape : private ResKeyShapeData
{
    NW_G3D_RES_COMMON(ResKeyShape);

public:

    enum KeyAttrib
    {
        NUM_KEY_ATTRIB_POSITION = 1,
        NUM_KEY_ATTRIB_NORMAL = 1,
        NUM_KEY_ATTRIB_TANGENT = 4,
        NUM_KEY_ATTRIB_BINORMAL = 4,
        NUM_KEY_ATTRIB_COLOR = 8,

        KEY_ATTRIB_POSITION_OFFSET = 0,
        KEY_ATTRIB_NORMAL_OFFSET = KEY_ATTRIB_POSITION_OFFSET + NUM_KEY_ATTRIB_POSITION,
        KEY_ATTRIB_TANGENT_OFFSET = KEY_ATTRIB_NORMAL_OFFSET + NUM_KEY_ATTRIB_NORMAL,
        KEY_ATTRIB_BINORMAL_OFFSET = KEY_ATTRIB_TANGENT_OFFSET + NUM_KEY_ATTRIB_TANGENT,
        KEY_ATTRIB_COLOR_OFFSET = KEY_ATTRIB_BINORMAL_OFFSET + NUM_KEY_ATTRIB_BINORMAL
    };

    //! @brief キーシェイプがもつ位置属性のインデクスを取得します。
    int GetPositionAttribIndex() const
    {
        return ref().targetAttribIndices[KEY_ATTRIB_POSITION_OFFSET] - 1;
    }

    //! @brief キーシェイプがもつ法線属性のインデクスを取得します。
    int GetNormalAttribIndex() const
    {
        return ref().targetAttribIndices[KEY_ATTRIB_NORMAL_OFFSET] - 1;
    }

    //! @brief キーシェイプがもつ接線属性のインデクスを取得します。
    int GetTangentAttribIndex(int index = 0) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, NUM_KEY_ATTRIB_TANGENT);
        return ref().targetAttribIndices[KEY_ATTRIB_TANGENT_OFFSET + index] - 1;
    }

    //! @brief キーシェイプがもつ従法線属性のインデクスを取得します。
    int GetBinormalAttribIndex(int index = 0) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, NUM_KEY_ATTRIB_BINORMAL);
        return ref().targetAttribIndices[KEY_ATTRIB_BINORMAL_OFFSET + index] - 1;
    }

    //! @brief キーシェイプがもつカラー属性のインデクスを取得します。
    int GetColorAttribIndex(int index = 0) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, NUM_KEY_ATTRIB_COLOR);
        return ref().targetAttribIndices[KEY_ATTRIB_COLOR_OFFSET + index] - 1;
    }

    //! @brief キーシェイプが持つ頂点属性のインデックスを取得します。
    NW_G3D_DEPRECATED_FUNCTION(int GetTargetAttribIndex(int attribIndex) const)
    {
        return ref().ofsTargetAttribIndex[attribIndex] - 1;
    }
};

//--------------------------------------------------------------------------------------------------

//! @brief AABB のバウンディング情報です。
struct Bounding
{
    Vec3 center;
    Vec3 extent;
};

//! @brief シェイプの構造体です。
struct ResShapeData
{
    BinaryBlockHeader blockHeader;

    BinString ofsName;

    bit32 flag;
    u16 index;
    u16 materialIndex;
    u16 boneIndex;
    u16 vertexIndex;
    u16 numSkinBoneIndex;
    u8 vtxSkinCount; //!< スキニングで1頂点に影響するボーンの数
    u8 numMesh;
    u8 numKeyShape;
    u8 numTargetAttrib;
    u16 reserved;

    Offset ofsRadiusArray; //!< 各 LOD のバウンディング球の半径配列
    Offset ofsVertex;
    Offset ofsMeshArray;
    Offset ofsSkinBoneIndexArray; //!< スキニングで参照するボーンインデックスのソート済み配列
    Offset ofsKeyShapeDic;

    Offset ofsSubMeshBoundingArray;

    BinPtr pUserPtr;
};

//! @brief シェイプのリソースです。
class ResShape : private ResShapeData // LOD の情報は含まない
{
    NW_G3D_RES_COMMON(ResShape);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'S', 'H', 'P') };

    //! @brief シェイプに関するフラグです。
    enum Flag
    {
        //! @brief ResVertex の所有権を持ちます。
        OWN_VERTEX                  = 0x1 << 1,
        SUBMESH_BOUNDARY_CONSISTENT = 0x1 << 2,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief シェイプをセットアップします。
    void Setup();

    //! @brief シェイプをクリーンアップします。
    void Cleanup();

    //! @brief シェイプをリセットします。
    void Reset();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief モデル内でのインデクスを取得します。
    int GetIndex() const { return ref().index; }

    NW_G3D_RES_FIELD_STRING_DECL(Name)

    //! @brief マテリアルのインデクスを取得します。
    int GetMaterialIndex() const { return ref().materialIndex; }

    //! @brief ビジビリティに用いるボーンインデクスを取得します。
    //!
    //! SkinBoneIndex とは異なることがあります。
    //!
    int GetBoneIndex() const { return ref().boneIndex; }

    //! @brief 頂点情報のインデクスを取得します。
    int GetVertexIndex() const { return ref().vertexIndex; }

    //! @brief 頂点情報の所有権を持っているかどうかを取得します。
    //!
    //! 複数のシェイプが同一の頂点情報を参照している場合、その中の1つが所有権を持ちます。
    //!
    bool IsVertexOwner() const { return 0 != (ref().flag & OWN_VERTEX); }

    //! @brief スキニングで使用するボーンの総数を取得します。
    int GetSkinBoneIndexCount() const { return ref().numSkinBoneIndex; }

    //! @brief スキニングで使用するボーンのリストを取得します。
    const u16* GetSkinBoneIndexArray() const { return ref().ofsSkinBoneIndexArray.to_ptr<u16>(); }

    //! @brief 1頂点を計算するのに使用するボーン数を取得します。
    int GetVtxSkinCount() const { return ref().vtxSkinCount; }

    //! @brief GetVtxSkinCount() の別名関数です。
    int GetVertexSkinCount() const { return GetVtxSkinCount(); }

    //! @brief 剛体のシェイプかどうかを取得します。
    bool IsRigidBody() const { return ref().vtxSkinCount == 0; }

    //! @brief リジッドスキニングのシェイプかどうかを取得します。
    bool IsRigidSkinning() const { return ref().vtxSkinCount == 1; }

    //! @brief スムーススキニングのシェイプかどうかを取得します。
    bool IsSmoothSkinning() const { return ref().vtxSkinCount > 1; }

    //! @brief メッシュ間でサブメッシュ境界が維持されているかを取得します。
    //!
    //! @return サブメッシュ境界が維持されている場合は true、それ以外の場合は false を返します。
    //!
    //! @details
    //! サブメッシュ境界が維持されているということは、すべてのメッシュにおいて、サブメッシュの数が
    //! 同じであり、同一のサブメッシュインデックスで示されるサブメッシュはバウンディングボックスの
    //! 形状がほぼ等しいことを意味します。
    //!
    //! 中間ファイルオプティマイザのポリゴンリダクション機能を使用して作成された LOD モデルはサブメッシュ
    //! 境界が維持されています。他の外部ツールで作成された LOD モデルを中間ファイルオプティマイザにより
    //! 結合した場合はサブメッシュ境界は維持されません。
    //!
    bool IsSubMeshBoundaryConsistent() const
    {
        return (ToData().flag & SUBMESH_BOUNDARY_CONSISTENT) != 0;
    }

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { ref().pUserPtr.set_ptr(pUserPtr); }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return ref().pUserPtr.to_ptr<T>(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return ref().pUserPtr.to_ptr<T>(); }

    //@}

    //----------------------------------------
    //! @name 頂点情報
    //@{

    NW_G3D_RES_FIELD_CLASS_DECL(ResVertex, Vertex);

    //@}

    //----------------------------------------
    //! @name メッシュ
    //@{

    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResMesh, Mesh, GetName());

    //! @brief メッシュを取得します。
    ResMesh* GetMesh() { return ResMesh::ResCast(ref().ofsMeshArray.to_ptr<ResMeshData>()); }

    //! @brief メッシュを取得します。
    const ResMesh* GetMesh() const { return ResMesh::ResCast(ref().ofsMeshArray.to_ptr<ResMeshData>()); }

    //! @brief サブメッシュの数を取得します。
    int GetSubMeshCount() const { return ref().ofsMeshArray.to_ptr<ResMeshData>()->numSubMesh; }

    //! @brief 指定したメッシュのサブメッシュの数を取得します。
    //!
    //! @param[in] meshIndex メッシュのインデックス。
    //!
    //! @return サブメッシュの数を返します。
    //!
    //! @pre
    //! - インデックスはメッシュの数の範囲内。
    //!
    int GetSubMeshCount(int meshIndex) const
    {
        const ResMeshData* pArray = ref().ofsMeshArray.to_ptr<ResMeshData>();
        return pArray[meshIndex].numSubMesh;
    }

    //@}

    //----------------------------------------
    //! @name キーシェイプ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResKeyShape, KeyShape, GetName());

    ////! @brief キーシェイプが持つ頂点属性の数を取得します。
    int GetTargetAttribCount() const { return ref().numTargetAttrib; }

    //@}

    //----------------------------------------
    //! @name バウンディング情報
    //@{

    //! @brief AABB のバウンディングボックスを取得します。
    const Bounding& GetBounding() const
    {
        return *(ref().ofsSubMeshBoundingArray.to_ptr<Bounding>() + GetSubMeshCount());
    }

    //! @brief 指定したメッシュに対応する AABB のバウンディングボックスを取得します。
    //!
    //! @param[in] meshIndex メッシュのインデックス。
    //!
    //! @return AABB のバウンディングボックスへの参照を返します。
    //!
    //! @pre
    //! - インデックスはメッシュの数の範囲内。
    //!
    const Bounding& GetBounding(int meshIndex) const
    {
        int boundingIndex = 0;
        for (int index = 0; index <= meshIndex; ++index)
        {
            boundingIndex += GetSubMeshCount(index);
        }
        boundingIndex += meshIndex;
        const Bounding* pArray = ref().ofsSubMeshBoundingArray.to_ptr<Bounding>();
        return pArray[boundingIndex];
    }

    //! @brief AABB と中心を共有するバウンディングスフィアの半径を取得します。
    float GetRadius() const { return ref().ofsRadiusArray.to_ptr<float>()[0]; }

    //! @brief 指定したメッシュに対応する AABB と中心を共有するバウンディングスフィアの半径を取得します。
    //!
    //! @param[in] meshIndex メッシュのインデックス。
    //!
    //! @return AABB と中心を共有するバウンディングスフィアの半径を返します。
    //!
    //! @pre
    //! - インデックスはメッシュの数の範囲内。
    //!
    float GetRadius(int meshIndex) const
    {
        const float* pArray = ref().ofsRadiusArray.to_ptr<float>();
        return pArray[meshIndex];
    }
    //@}
};

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS

#endif // NW_G3D_RES_RESSHAPE_H_
