﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESSHADERPARAMANIM_H_
#define NW_G3D_RES_RESSHADERPARAMANIM_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/res/g3d_ResAnimCurve.h>
#include <nw/g3d/res/g3d_ResModel.h>
#include <nw/g3d/res/g3d_Binding.h>

namespace nw { namespace g3d { namespace res {

//! @brief パラメータアニメーションの構造体です。
struct ResShaderParamMatAnimData
{
    u16 numAnimParam;
    u16 numCurve;
    u16 numConstant;
    u16 reserved;
    s32 beginCurve; // ShaderParamAnim 内での先頭 Curve のインデックス
    s32 beginParamAnim; // ShaderParamAnim 内での ParamAnimInfo のインデックス

    BinString ofsName;

    Offset ofsParamAnimInfoArray;
    Offset ofsCurveArray;
    Offset ofsConstantArray;
};

//! @brief パラメータアニメーションのリソースです。
class ResShaderParamMatAnim : private ResShaderParamMatAnimData
{
    NW_G3D_RES_COMMON(ResShaderParamMatAnim);

public:
    //! @brief パラメータアニメーションの要素情報です。
    struct ParamAnimInfo
    {
        u16 beginCurve; // MatAnim 内での先頭 Curve のインデックス
        u16 numFloatCurve;
        u16 numIntCurve;
        u16 beginConstant; // MatAnim 内での先頭 Constant のインデックス
        u16 numConstant;
        u16 subbindIndex; // Material 内での ShaderParam のインデックス

        BinString ofsName;
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 事前に ResMaterial と関連付けを行います。
    //!
    //! リソース同士を ResMaterial によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    BindResult PreBind(const ResMaterial* target);

    //! @brief 事前に ResMaterial と関連付けが成功するか、チェックのみを行います。
    BindResult BindCheck(const ResMaterial* target) const;

    //@}

    //----------------------------------------
    //! @name 評価
    //@{

    //! @brief アニメーションの評価を行います。
    void Eval(void* pResult, float frame, const u16* pSubBindIndex) const
    {
        EvalImpl<false>(pResult, frame, pSubBindIndex, NULL);
    }

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(void* pResult, float frame, const u16* pSubBindIndex) const
    {
        Eval(pResult, frame, pSubBindIndex);
    }

    //! @brief アニメーションの評価を行います。
    void Eval(void* pResult, float frame, const u16* pSubBindIndex,
        AnimFrameCache* pFrameCache) const
    {
        EvalImpl<true>(pResult, frame, pSubBindIndex, pFrameCache);
    }

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(void* pResult, float frame, const u16* pSubBindIndex,
        AnimFrameCache* pFrameCache) const
    {
        Eval(pResult, frame, pSubBindIndex, pFrameCache);
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief アニメーション数を取得します。
    int GetParamAnimCount() const { return ref().numAnimParam; }
    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResAnimCurve, Curve, GetName())
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResAnimConstant, Constant, GetName())

    //@}

protected:
    //! @briefprivate
    template <bool useContext>
    void EvalImpl(void* pResult, float frame, const u16* pSubBindIndex,
        AnimFrameCache* pFrameCache) const;
};

//--------------------------------------------------------------------------------------------------

//! @brief シェーダパラメータアニメーションの構造体です。
struct ResShaderParamAnimData
{
    BinaryBlockHeader blockHeader;
    BinString ofsName;
    BinString ofsPath;

    bit32 flag;
    s32 numFrame;
    u16 numMatAnim; // BindTable のサイズに影響
    u16 numUserData;
    s32 numParamAnim; // SubBindTable のサイズに影響
    s32 numCurve; // Context のサイズに影響
    u32 bakedSize;

    Offset ofsBindModel;
    Offset ofsBindIndexArray; // Material のインデックス
    Offset ofsMatAnimArray;
    Offset ofsUserDataDic;
};

//! @brief シェーダパラメータアニメーションのリソースです。
class ResShaderParamAnim : private ResShaderParamAnimData
{
    NW_G3D_RES_COMMON(ResShaderParamAnim);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'S', 'H', 'U') };

    //! @brief シェーダパラメータアニメーションに関するフラグです。
    enum Flag
    {
        //! @brief カーブがベイク済みです。
        CURVE_BAKED         = AnimFlag::CURVE_BAKED,

        //! @brief ループすることを表します。
        PLAYPOLICY_LOOP     = AnimFlag::PLAYPOLICY_LOOP
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 事前に ResModel と関連付けを行います。
    //!
    //! リソース同士を PreBind によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    BindResult PreBind(const ResModel* pModel);

    //! @brief 事前に ResModel と関連付けが成功するか、チェックのみを行います。
    BindResult BindCheck(const ResModel* target) const;

    //! @brief カーブをコマ化します。
    bool BakeCurve(void* pBuffer, size_t bufferSize);

    //! @brief カーブに戻します。
    void* ResetCurve();

    //! @brief シェーダパラメータアニメーションをリセットします。
    void Reset();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief ループするかどうかを取得します。
    bool IsLooped() const { return 0 != (ref().flag & PLAYPOLICY_LOOP); }

    //! @brief カーブがベイクされているかどうかを取得します。
    bool IsCurveBaked() const { return 0 != (ref().flag & CURVE_BAKED); }

    //! @brief フレーム数を取得します。
    int GetFrameCount() const { return ref().numFrame; }

    //! @brief パラメータアニメーション数を取得します。
    int GetParamAnimCount() const { return ref().numParamAnim; }

    //! @brief カーブ数を取得します。
    int GetCurveCount() const { return ref().numCurve; }

    //! @brief コマ化に必要なバッファサイズを取得します。
    size_t GetBakedSize() const { return ref().bakedSize; }

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_STRING_DECL(Path)
    NW_G3D_RES_FIELD_CLASS_DECL(ResModel, BindModel)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResShaderParamMatAnim, MatAnim, GetName())

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

    //@}
};

}}} // namespace nw::g3d::res

#endif // NW_G3D_RES_RESSHADERPARAMANIM_H_
