﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESSCENEANIM_H_
#define NW_G3D_RES_RESSCENEANIM_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/res/g3d_ResAnimCurve.h>
#include <nw/g3d/res/g3d_Binding.h>
#include <nw/g3d/res/g3d_ResUserData.h>

namespace nw { namespace g3d { namespace res {

//! @brief カメラアニメーションの結果を格納する構造体です。
struct CameraAnimResult
{
    struct Projection
    {
        float nearZ;
        float farZ;
        float aspect;
        union
        {
            float height;
            float fovy;
        };
    } proj;
    struct View
    {
        float pos[3];
        union
        {
            float aim[3];
            float rotate[3];
        };
        float twist;
    } view;
};

//! @brief ライトアニメーションの結果を格納する構造体です。
struct LightAnimResult
{
    s32 enable;
    float pos[3];
    union
    {
        float aim[3];
        float dir[3];
    };
    float distAttn[2];
    float angleAttn[2];
    float color[2][3];
};

//! @brief フォグアニメーションの結果を格納する構造体です。
struct FogAnimResult
{
    float distAttn[2];
    float color[3];
};

class BindFuncTable;

//--------------------------------------------------------------------------------------------------

//! @brief カメラアニメーションの構造体です。
struct ResCameraAnimData
{
    BinaryBlockHeader blockHeader;
    bit16 flag;
    bit16 reserved;
    s32 numFrame;
    u8 numCurve;
    u8 reserved2;
    u16 numUserData;
    u32 bakedSize;
    BinString ofsName;
    Offset ofsCurveArray;
    Offset ofsBaseValueArray;
    Offset ofsUserDataDic;
};

//! @brief カメラアニメーションのリソースです。
class ResCameraAnim : private ResCameraAnimData
{
    NW_G3D_RES_COMMON(ResCameraAnim);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'C', 'A', 'M') };

    //! @brief カメラアニメーションに関するフラグです。
    enum Flag
    {
        //! @brief カーブがベイク済みです。
        CURVE_BAKED         = AnimFlag::CURVE_BAKED,

        //! @brief ループすることを表します。
        PLAYPOLICY_LOOP     = AnimFlag::PLAYPOLICY_LOOP,

        // RotateMode
        //! @briefprivate
        ROT_SHIFT           = 8,

        //! @brief カメラの視線を注視点から計算します。
        ROT_AIM             = 0x0 << ROT_SHIFT,

        //! @brief カメラの視線を回転角から計算します。
        ROT_EULER_ZXY       = 0x1 << ROT_SHIFT,

        //! @briefprivate
        ROT_MASK            = 0x1 << ROT_SHIFT,

        // ProjectionMode
        //! @briefprivate
        PROJ_SHIFT          = 10,

        //! @brief 平行投影で計算します。
        PROJ_ORTHO          = 0x0 << PROJ_SHIFT,

        //! @brief 透視投影で計算します。
        PROJ_PERSP          = 0x1 << PROJ_SHIFT,

        //! @briefprivate
        PROJ_MASK           = 0x1 << PROJ_SHIFT,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief CameraAnimResult の初期化を行います。
    void Init(CameraAnimResult* pResult) const;

    //! @brief Init() 関数の別名関数です。
    void Initialize(CameraAnimResult* pResult) const
    {
        Init(pResult);
    }

    //! @brief カーブをコマ化します。
    bool BakeCurve(void* pBuffer, size_t bufferSize);

    //! @brief カーブに戻します。
    void* ResetCurve();

    //! @brief カメラアニメーションをリセットします。
    void Reset();

    //@}

    //----------------------------------------
    //! @name 評価
    //@{

    //! @brief カメラアニメーションを評価します。
    void Eval(CameraAnimResult* pResult, float frame) const;

    //! @brief  Eval() 関数の別名関数です。
    void Evaluate(CameraAnimResult* pResult, float frame) const
    {
        Eval(pResult, frame);
    }

    //! @brief カメラアニメーションを評価します。
    void Eval(CameraAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const;

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(CameraAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const
    {
        Eval(pResult, frame, pFrameCache);
    }

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResAnimCurve, Curve, GetName())

    //! @brief ループするかどうかを取得します。
    bool IsLooped() const { return 0 != (ref().flag & PLAYPOLICY_LOOP); }

    //! @brief カーブがベイクされているかどうかを取得します。
    bool IsCurveBaked() const { return 0 != (ref().flag & CURVE_BAKED); }

    //! @brief フレーム数を取得します。
    int GetFrameCount() const { return ref().numFrame; }

    //! @brief 回転の計算方式を取得します。
    bit32 GetRotateMode() const { return ref().flag & ROT_MASK; }

    //! @brief プロジェクションの計算方式を取得します。
    bit32 GetProjectionMode() const { return ref().flag & PROJ_MASK; }

    //! @brief コマ化に必要なバッファサイズを取得します。
    size_t GetBakedSize() const { return ref().bakedSize; }

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief ライトアニメーションの構造体です。
struct ResLightAnimData
{
    BinaryBlockHeader blockHeader;
    bit16 flag;
    u16 numUserData;
    s32 numFrame;
    u8 numCurve;
    s8 lightTypeIndex;
    s8 distAttnFuncIndex;
    s8 angleAttnFuncIndex;
    u32 bakedSize;
    BinString ofsName;
    BinString ofsLightType;
    BinString ofsDistAttnFunc;
    BinString ofsAngleAttnFunc;
    Offset ofsCurveArray;
    Offset ofsBaseValueArray;
    Offset ofsUserDataDic;
};

//! @brief ライトアニメーションのリソースです。
class ResLightAnim : private ResLightAnimData
{
    NW_G3D_RES_COMMON(ResLightAnim);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'L', 'I', 'T') };

    //! @brief ライトアニメーションに関するフラグです。
    enum Flag
    {
        //! @brief カーブがベイク済みです。
        CURVE_BAKED         = AnimFlag::CURVE_BAKED,

        //! @brief ループすることを表します。
        PLAYPOLICY_LOOP     = AnimFlag::PLAYPOLICY_LOOP,

        //! @brief ライトの有効無効のアニメーションが存在します。
        CURVE_ENABLE        = 0x1 << 8,

        //! @brief ライトの有効無効の初期値が存在します。
        BASE_ENABLE         = 0x1 << 9,

        //! @brief 位置の初期値が存在します。
        BASE_POS            = 0x1 << 10,

        //! @brief 角度の初期値が存在します。
        BASE_DIR            = 0x1 << 11,

        //! @brief 距離減衰の初期値が存在します。
        BASE_DIST_ATTN      = 0x1 << 12,

        //! @brief 角度減衰の初期値が存在します。
        BASE_ANGLE_ATTN     = 0x1 << 13,

        //! @brief カラー0の初期値が存在します。
        BASE_COLOR0         = 0x1 << 14,

        //! @brief カラー1の初期値が存在します。
        BASE_COLOR1         = 0x1 << 15
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief LightAnimResult の初期化を行います。
    void Init(LightAnimResult* pResult) const;

    //! @brief Init() 関数の別名関数です。
    void Initialize(LightAnimResult* pResult) const
    {
        Init(pResult);
    }

    //! @brief 名前が一致する関数と参照解決を行います。
    BindResult Bind(const BindFuncTable& funcTable);

    //! @brief 参照設定を解除します。
    void Release();

    //! @brief カーブをコマ化します。
    bool BakeCurve(void* pBuffer, size_t bufferSize);

    //! @brief カーブに戻します。
    void* ResetCurve();

    //! @brief ライトアニメーションをリセットします。
    void Reset();

    //@}

    //----------------------------------------
    //! @name 評価
    //@{

    //! @brief ライトアニメーションを評価します。
    void Eval(LightAnimResult* pResult, float frame) const;

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(LightAnimResult* pResult, float frame) const
    {
        Eval(pResult, frame);
    }

    //! @brief ライトアニメーションを評価します。
    void Eval(LightAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const;

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(LightAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const
    {
        Eval(pResult, frame, pFrameCache);
    }

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_STRING_DECL(LightType)
    NW_G3D_RES_FIELD_STRING_DECL(DistAttnFunc)
    NW_G3D_RES_FIELD_STRING_DECL(AngleAttnFunc)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResAnimCurve, Curve, GetName())

    //! @brief 事前に関連付けたライトタイプのインデックスを取得します。
    int GetLightTypeIndex() const { return ref().lightTypeIndex; }

    //! @brief 事前に関連付けた距離減衰関数のインデックスを取得します。
    int GetDistAttnFuncIndex() const { return ref().distAttnFuncIndex; }

    //! @brief 事前に関連付けた角度減衰関数のインデックスを取得します。
    int GetAngleAttnFuncIndex() const { return ref().angleAttnFuncIndex; }

    //! @brief ループするかどうかを取得します。
    bool IsLooped() const { return 0 != (ref().flag & PLAYPOLICY_LOOP); }

    //! @brief カーブがベイクされているかどうかを取得します。
    bool IsCurveBaked() const { return 0 != (ref().flag & CURVE_BAKED); }

    //! @brief フレーム数を取得します。
    int GetFrameCount() const { return ref().numFrame; }

    //! @brief コマ化に必要なバッファサイズを取得します。
    size_t GetBakedSize() const { return ref().bakedSize; }

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

        //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief フォグアニメーションの構造体です。
struct ResFogAnimData
{
    BinaryBlockHeader blockHeader;
    bit16 flag;
    bit16 reserved;
    s32 numFrame;
    u8 numCurve;
    s8 distAttnFuncIndex;
    u16 numUserData;
    u32 bakedSize;
    BinString ofsName;
    BinString ofsDistAttnFunc;
    Offset ofsCurveArray;
    Offset ofsBaseValueArray;
    Offset ofsUserDataDic;
};

//! @brief フォグアニメーションのリソースです。
class ResFogAnim : private ResFogAnimData
{
    NW_G3D_RES_COMMON(ResFogAnim);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'F', 'O', 'G') };

    //! @brief フォグアニメーションに関するフラグです。
    enum Flag
    {
        //! @brief カーブがベイク済みです。
        CURVE_BAKED         = AnimFlag::CURVE_BAKED,

        //! @brief ループすることを表します。
        PLAYPOLICY_LOOP     = AnimFlag::PLAYPOLICY_LOOP
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief FogAnimResult の初期化を行います。
    void Init(FogAnimResult* pResult) const;

    //! @brief Init() 関数の別名関数です。
    void Initialize(FogAnimResult* pResult) const
    {
        Init(pResult);
    }

    //! @brief 名前が一致する関数と参照解決を行います。
    BindResult Bind(const BindFuncTable& funcTable);

    //! @brief 参照設定を解除します。
    void Release();

    //! @brief カーブをコマ化します。
    bool BakeCurve(void* pBuffer, size_t bufferSize);

    //! @brief カーブに戻します。
    void* ResetCurve();

    //! @brief フォグアニメーションをリセットします。
    void Reset();

    //@}

    //----------------------------------------
    //! @name 評価
    //@{

    //! @brief フォグアニメーションを評価します。
    void Eval(FogAnimResult* pResult, float frame) const;

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(FogAnimResult* pResult, float frame) const
    {
        Eval(pResult, frame);
    }

    //! @brief フォグアニメーションを評価します。
    void Eval(FogAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const;

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(FogAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const
    {
        Eval(pResult, frame, pFrameCache);
    }

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_STRING_DECL(DistAttnFunc)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResAnimCurve, Curve, GetName())

    //! @brief 事前に関連付けた距離減衰関数のインデックスを取得します。
    int GetDistAttnFuncIndex() const { return ref().distAttnFuncIndex; }

    //! @brief ループするかどうかを取得します。
    bool IsLooped() const { return 0 != (ref().flag & PLAYPOLICY_LOOP); }

    //! @brief カーブがベイクされているかどうかを取得します。
    bool IsCurveBaked() const { return 0 != (ref().flag & CURVE_BAKED); }

    //! @brief フレーム数を取得します。
    int GetFrameCount() const { return ref().numFrame; }

    //! @brief コマ化に必要なバッファサイズを取得します。
    size_t GetBakedSize() const { return ref().bakedSize; }

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief シーンアニメーションの構造体です。
struct ResSceneAnimData
{
    BinaryBlockHeader blockHeader;
    BinString ofsName;
    BinString ofsPath;

    u16 numUserData;
    u16 numCameraAnim;
    u16 numLightAnim;
    u16 numFogAnim;

    Offset ofsCameraAnimDic;
    Offset ofsLightAnimDic;
    Offset ofsFogAnimDic;
    Offset ofsUserDataDic;
};

//! @brief シーンアニメーションのリソースです。
class ResSceneAnim : private ResSceneAnimData
{
    NW_G3D_RES_COMMON(ResSceneAnim);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'S', 'C', 'N') };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 名前が一致する関数と参照解決を行います。
    BindResult Bind(const BindFuncTable& funcTable);

    //! @brief 参照設定を解除します。
    void Release();

    //! @brief シーンアニメーションをリセットします。
    void Reset();

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_STRING_DECL(Path)

    //@}

    //----------------------------------------
    //! @name カメラアニメーション
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResCameraAnim, CameraAnim, GetName())

    //@}

    //----------------------------------------
    //! @name ライトアニメーション
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResLightAnim, LightAnim, GetName())

    //@}

    //----------------------------------------
    //! @name フォグアニメーション
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResFogAnim, FogAnim, GetName())

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief リソースに関連付ける関数名のテーブルです。
class BindFuncTable
{
public:

    //! @brief 関数名を格納する構造体です。
    struct Name
    {
        size_t len;
        const char* str;
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    BindFuncTable()
        : m_NumLightType(0)
        , m_NumLightDistAttn(0)
        , m_NumLightAngleAttn(0)
        , m_NumFogDistAttn(0)
        , m_LightTypeArray(NULL)
        , m_LightDistAttnArray(NULL)
        , m_LightAngleAttnArray(NULL)
        , m_FogDistAttnArray(NULL)
    {
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief ライトタイプの関数名一覧を設定します。
    void SetLightTypeArray(const Name* pLightTyepArray, int count)
    {
        NW_G3D_ASSERT(pLightTyepArray != NULL || count == 0);
        m_NumLightType = count;
        m_LightTypeArray = pLightTyepArray;
    }

    //! @brief ライトタイプの関数名一覧を取得します。
    const Name* GetLightTypeArray() const { return m_LightTypeArray; }

    //! @brief ライトタイプの関数名の数を取得します。
    int GetLightTypeCount() const { return m_NumLightType; }

    //! @brief ライトの距離減衰の関数名一覧を設定します。
    void SetLightDistAttnArray(const Name* pLightDistAttnArray, int count)
    {
        NW_G3D_ASSERT(pLightDistAttnArray != NULL || count == 0);
        m_NumLightDistAttn = count;
        m_LightDistAttnArray = pLightDistAttnArray;
    }

    //! @brief ライトの距離減衰の関数名一覧を取得します。
    const Name* GetLightDistAttnArray() const { return m_LightDistAttnArray; }

    //! @brief ライトの距離減衰の関数名の数を取得します。
    int GetLightDistAttnCount() const { return m_NumLightDistAttn; }

    //! @brief ライトの角度減衰の関数名一覧を設定します。
    void SetLightAngleAttnArray(const Name* pLightAngleAttnArray, int count)
    {
        NW_G3D_ASSERT(pLightAngleAttnArray != NULL || count == 0);
        m_NumLightAngleAttn = count;
        m_LightAngleAttnArray = pLightAngleAttnArray;
    }

    //! @brief ライトの角度減衰の関数名一覧を取得します。
    const Name* GetLightAngleAttnArray() const { return m_LightAngleAttnArray; }

    //! @brief ライトの角度減衰の関数名の数を取得します。
    int GetLightAngleAttnCount() const { return m_NumLightAngleAttn; }

    //! @brief フォグの距離減衰の関数名一覧を設定します。
    void SetFogDistAttnArray(const Name* pFogDistAttnArray, int count)
    {
        NW_G3D_ASSERT(pFogDistAttnArray != NULL || count == 0);
        m_NumFogDistAttn = count;
        m_FogDistAttnArray = pFogDistAttnArray;
    }

    //! @brief フォグの距離減衰の関数名一覧を取得します。
    const Name* GetFogDistAttnArray() const { return m_FogDistAttnArray; }

    //! @brief フォグの距離減衰の関数名の数を取得します。
    int GetFogDistAttnCount() const { return m_NumFogDistAttn; }

    //@}

private:
    int m_NumLightType;
    int m_NumLightDistAttn;
    int m_NumLightAngleAttn;
    int m_NumFogDistAttn;
    const Name* m_LightTypeArray;
    const Name* m_LightDistAttnArray;
    const Name* m_LightAngleAttnArray;
    const Name* m_FogDistAttnArray;
};


}}} // namespace nw::g3d::res

#endif // NW_G3D_RES_RESSCENEANIM_H_
