﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESMATERIAL_H_
#define NW_G3D_RES_RESMATERIAL_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <nw/g3d/math/g3d_Vector3.h>
#include <nw/g3d/math/g3d_Vector4.h>
#include <nw/g3d/math/g3d_Matrix34.h>
#include <nw/g3d/fnd/g3d_GfxState.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/res/g3d_Binding.h>
#include <nw/g3d/res/g3d_ResTexture.h>
#include <nw/g3d/res/g3d_ResUserData.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

class BindCallback;
class ResFile;

//! @brief Srt2d のデータです。
//!
//! TYPE_SRT2D で用います。
//!
struct Srt2d
{
    float sx, sy, r, tx, ty;
};

//! @brief Srt3d のデータです。
//!
//! TYPE_SRT3D で用います。
//!
struct Srt3d
{
    Vec3 scale, rotate, translate;
};

//! @brief TexSrt のデータです。
//!
//! TYPE_TEXSRT で用います。
//!
struct TexSrt
{
    enum Mode
    {
        MODE_MAYA,
        MODE_3DSMAX,
        MODE_SOFTIMAGE,
        NUM_MODE
    };

    Mode mode;
    float sx, sy, r, tx, ty;

};

//! @brief TexSrtEx のデータです。
//!
//! TYPE_TEXSRT_EX で用います。
//!
struct TexSrtEx
{
    TexSrt srt;
    union
    {
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
        Mtx34* pEffectMtx;
#endif
        u32 padding;
    };
};

//--------------------------------------------------------------------------------------------------

//! @brief 描画情報の構造体です。
struct ResRenderInfoData
{
    u16 arrayLength;
    u8 type;
    u8 reserved;
    BinString ofsName;
    union
    {
        s32 iValue[1];
        f32 fValue[1];
        BinString ofsString[1];
    };
};

//! @brief 描画情報のリソースです。
class ResRenderInfo : private ResRenderInfoData
{
    NW_G3D_RES_COMMON(ResRenderInfo);

public:
    //! @brief データタイプを表します。
    enum Type
    {
        //! @brief Int 値として表されるデータです。
        INT,
        //! @brief Float 値として表されるデータです。
        FLOAT,
        //! @brief ascii の値として表されるデータです。
        STRING
    };

    //----------------------------------------
    //! @name 取得
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)

    //! @brief 配列長を取得します。
    int GetArrayLength() const { return ref().arrayLength; }

    //! @brief データのタイプを取得します。
    Type GetType() const { return static_cast<Type>(ref().type); }

    //! @brief Int 値としてデータを取得します。
    const int* GetInt() const
    {
        NW_G3D_ASSERTMSG(ref().type == INT, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
        return reinterpret_cast<const int*>(ref().iValue);
    }

    //! @brief Float 値としてデータを取得します。
    const float* GetFloat() const
    {
        NW_G3D_ASSERTMSG(ref().type == FLOAT, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
        return reinterpret_cast<const float*>(ref().fValue);
    }

    //! @brief 文字列としてデータを取得します。
    const char* GetString(int strIndex) const
    {
        NW_G3D_ASSERTMSG(ref().type == STRING, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(strIndex, ref().arrayLength, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
        return ref().ofsString[strIndex].to_ptr();
    }

    //@}
};

typedef ResRenderInfoData ResEnvRefData; // TODO: remove this
typedef ResRenderInfo ResEnvRef; // TODO: remove this

//--------------------------------------------------------------------------------------------------

//! @brief 描画ステートの構造体です。
struct ResRenderStateData
{
    bit32 flag; // mode
    GfxPolygonCtrl_t polygonCtrl;
    GfxDepthCtrl_t depthCtrl;
    GfxAlphaTest_t alphaTest;
    GfxColorCtrl_t colorCtrl;
    GfxBlendCtrl_t blendCtrl;
    GfxBlendColor_t blendColor;
};

//! @brief 描画ステートのリソースです。
class ResRenderState : private ResRenderStateData
{
    NW_G3D_RES_COMMON(ResRenderState);

public:
    //! @briefprivate
    enum Flag
    {
        MODE_SHIFT          = 0,
        MODE_MASK           = 0x3 << MODE_SHIFT,
        BLEND_SHIFT         = 4,
        BLEND_MASK          = 0x3 << BLEND_SHIFT
    };

    //! @brief レンダーステートのモードです。
    enum Mode
    {
        //! @brief カスタムモードです。
        MODE_CUSTOM         = 0,

        //! @brief 不透明モードです。
        MODE_OPAQUE         = 1,

        //! @brief 抜きモードです。
        MODE_ALPHAMASK      = 2,

        //! @brief 半透明モードです。
        MODE_TRANSLUCENT    = 3,
    };

    //! @brief ブレンドモードです。
    enum BlendMode
    {
        //! @brief ブレンド処理をせず、シェーダ出力値をそのままフレームバッファに書き込みます。
        BLEND_NONE          = 0,

        //! @brief フレームバッファとカラーブレンドします。
        BLEND_COLOR         = 1,

        //! @brief フレームバッファと論理ブレンドします。
        BLEND_LOGICAL       = 2,
    };

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief レンダーステートのモードに応じて必要な設定を更新します。
    void AdjustMode(Mode mode);

    //! @brief ブレンドモードに応じて必要な設定を更新します。
    void AdjustBlendMode(BlendMode mode);

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief レンダーステートを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief レンダーステートのモードを取得します。
    Mode GetMode() const { return static_cast<Mode>((ref().flag & MODE_MASK) >> MODE_SHIFT); }

    //! @brief レンダーステートのモードを設定します。
    //!
    //! ブレンドモードのフラグのみを設定します。
    //! AdjustMode() を呼び出すことでモードに応じてステートを設定します。
    //!
    void SetMode(Mode mode)
    {
        ref().flag = (ref().flag & ~MODE_MASK) | (mode << MODE_SHIFT);
    }

    //! @brief ブレンドモードを取得します。
    BlendMode GetBlendMode() const
    {
        return static_cast<BlendMode>((ref().flag & BLEND_MASK) >> BLEND_SHIFT);
    }

    //! @brief ブレンドモードを設定します。
    //!
    //! ブレンドモードのフラグのみを設定します。
    //! AdjustBlendMode() を呼び出すことでモードに応じてステートを設定します。
    //!
    void SetBlendMode(BlendMode mode)
    {
        ref().flag = (ref().flag & ~BLEND_MASK) | (mode << BLEND_SHIFT);
    }

    //! @brief ポリゴンを取得します。
    GfxPolygonCtrl& GetPolygonCtrl()
    {
        return GfxPolygonCtrl::DownCast(ref().polygonCtrl);
    }

    //! @brief ポリゴンを取得します。
    const GfxPolygonCtrl& GetPolygonCtrl() const
    {
        return GfxPolygonCtrl::DownCast(ref().polygonCtrl);
    }

    //! @brief 深度テストを取得します。
    GfxDepthCtrl& GetDepthCtrl()
    {
        return GfxDepthCtrl::DownCast(ref().depthCtrl);
    }

    //! @brief 深度テストを取得します。
    const GfxDepthCtrl& GetDepthCtrl() const
    {
        return GfxDepthCtrl::DownCast(ref().depthCtrl);
    }

    //! @brief アルファテストを取得します。
    GfxAlphaTest& GetAlphaTest()
    {
        return GfxAlphaTest::DownCast(ref().alphaTest);
    }

    //! @brief アルファテストを取得します。
    const GfxAlphaTest& GetAlphaTest() const
    {
        return GfxAlphaTest::DownCast(ref().alphaTest);
    }

    //! @brief カラーコントロールを取得します。
    GfxColorCtrl& GetColorCtrl()
    {
        return GfxColorCtrl::DownCast(ref().colorCtrl);
    }

    //! @brief カラーコントロールを取得します。
    const GfxColorCtrl& GetColorCtrl() const
    {
        return GfxColorCtrl::DownCast(ref().colorCtrl);
    }

    //! @brief ブレンドコントロールを取得します。
    GfxBlendCtrl& GetBlendCtrl()
    {
        return GfxBlendCtrl::DownCast(ref().blendCtrl);
    }

    //! @brief ブレンドコントロールを取得します。
    const GfxBlendCtrl& GetBlendCtrl() const
    {
        return GfxBlendCtrl::DownCast(ref().blendCtrl);
    }

    //! @brief ブレンドカラーを取得します。
    GfxBlendColor& GetBlendColor()
    {
        return GfxBlendColor::DownCast(ref().blendColor);
    }

    //! @brief ブレンドカラーを取得します。
    const GfxBlendColor& GetBlendColor() const
    {
        return GfxBlendColor::DownCast(ref().blendColor);
    }

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief シェーダ割り当て情報の構造体です。
struct ResShaderAssignData
{
    BinString ofsShaderArchiveName; //!< メイン描画パスで使用するシェーダアーカイブの名前
    BinString ofsShadingModelName; //!< メイン描画パスで使用するシェーディングモデルの名前
    u32 revision;
    u8 numAttribAssign;
    u8 numSamplerAssign;
    u16 numShaderOption;
    Offset ofsAttribAssignDic;
    Offset ofsSamplerAssignDic;
    Offset ofsShaderOptionDic;
};

//! @brief シェーダ割り当て情報です。
class ResShaderAssign : private ResShaderAssignData
{
    NW_G3D_RES_COMMON(ResShaderAssign);

public:

    //----------------------------------------
    //! @name 取得
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(ShaderArchiveName)
    NW_G3D_RES_FIELD_STRING_DECL(ShadingModelName)

    //@}

    //----------------------------------------
    //! @name 頂点属性
    //@{

    NW_G3D_RES_FIELD_STR_DIC_DECL_DETAIL(AttribAssign, GetShadingModelName()) // シェーダ側の id からモデル側の名前を取得
    NW_G3D_RES_FIELD_STR_DIC_DECL_DETAIL_WITH_FUNCNAME(AttrAssign, AttribAssign, GetShadingModelName())

    //! @brief GetAttribAssignName() の別名関数です。
    const char* GetAttrAssignId(int elemIndex) const
    {
        return GetAttribAssignName(elemIndex);
    }

    //@}

    //----------------------------------------
    //! @name サンプラー
    //@{

    NW_G3D_RES_FIELD_STR_DIC_DECL_DETAIL(SamplerAssign, GetShadingModelName()) // シェーダ側の id からモデル側の名前を取得

    //! @brief GetSamplerAssignName() の別名関数です。
    const char* GetSamplerAssignId(int elemIndex) const
    {
        return GetSamplerAssignName(elemIndex);
    }

    //@}

    //----------------------------------------
    //! @name オプション
    //@{

    NW_G3D_RES_FIELD_STR_DIC_DECL_DETAIL(ShaderOption, GetShadingModelName()) // シェーダ側の id からオプションの値を取得

    //! @brief GetShaderOptionName() の別名関数です。
    const char* GetShaderOptionId(int elemIndex) const
    {
        return GetShaderOptionName(elemIndex);
    }

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief テクスチャサンプラーの構造体です。
struct ResSamplerData
{
    GfxSampler_t gfxSampler;
    BinString ofsName;
    u8 index;
    u8 reserved0;
    u16 reserved1;
};

//! @brief テクスチャサンプラーのリソースです。
class ResSampler : private ResSamplerData
{
    NW_G3D_RES_COMMON(ResSampler);

public:
    NW_G3D_RES_FIELD_STRING_DECL(Name)

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief リソースをセットアップします。
    void Setup();

    //! @brief リソースをクリーンアップします。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief テクスチャのインデクスを取得します。
    int GetIndex() const { return ref().index; }

    //! @brief サンプラーを取得します。
    GfxSampler* GetGfxSampler() { return GfxSampler::DownCast(&ref().gfxSampler); }

    //! @brief サンプラーを取得します。
    const GfxSampler* GetGfxSampler() const { return GfxSampler::DownCast(&ref().gfxSampler); }

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief シェーダパラメーターの構造体です。
struct ResShaderParamData
{
    u8 type;
    u8 srcSize; //!< 変換前データのバッファサイズ
    u16 srcOffset; //!< 変換前データのバッファ内オフセット
    s32 offset; //!< GPU に送られる変換後データのバッファ内オフセット
    BinPtr pCallback; //!< バッファ転送時に呼び出される変換コールバック
    u16 dependedIndex; //!< 依存元パラメータのインデックス
    u16 dependIndex; //!< 依存先パラメータのインデックス

    BinString ofsId;
};

//! @brief シェーダパラメーターのリソースです。
class ResShaderParam : private ResShaderParamData
{
    NW_G3D_RES_COMMON(ResShaderParam);

public:
    //! @brief シェーダパラメータの各種タイプです。
    enum Type
    {
        TYPE_BOOL, TYPE_BOOL2, TYPE_BOOL3, TYPE_BOOL4,
        TYPE_INT, TYPE_INT2, TYPE_INT3, TYPE_INT4,
        TYPE_UINT, TYPE_UINT2, TYPE_UINT3, TYPE_UINT4,
        TYPE_FLOAT, TYPE_FLOAT2, TYPE_FLOAT3, TYPE_FLOAT4,
        TYPE_RESERVED2, TYPE_FLOAT2x2, TYPE_FLOAT2x3, TYPE_FLOAT2x4,
        TYPE_RESERVED3, TYPE_FLOAT3x2, TYPE_FLOAT3x3, TYPE_FLOAT3x4,
        TYPE_RESERVED4, TYPE_FLOAT4x2, TYPE_FLOAT4x3, TYPE_FLOAT4x4,
        TYPE_SRT2D, TYPE_SRT3D, TYPE_TEXSRT,
        NUM_TYPE
    };

    //! @brief ユニフォームブロック用のバッファへ転送される際に呼び出される変換コールバックです。
    typedef size_t (*ConvertParamCallback)(void* pDst, const void* pSrc, Type srcType, const void *pUserPtr);

    NW_G3D_RES_FIELD_STRING_DECL(Id);

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief ソースデータを変換して UniformBlock へ書き込みます。
    template <bool swap>
    void Convert(void* pDst, const void* pSrc) const;

    //! @brief ソースデータをコールバックにより変換して UniformBlock へ書き込みます。
    template <bool swap>
    void Convert(void* pDst, const void* pSrc, const void* pUserPtr) const
    {
        if (pCallback.to_ptr())
        {
            u32 tmpDst[16];
            size_t written = GetConvertParamCallback()(tmpDst, pSrc, static_cast<Type>(type), pUserPtr);
            Copy32<swap>(pDst, tmpDst, static_cast<int>((written + 3) >> 2));
        }
        else
        {
            Convert<swap>(pDst, pSrc);
        }
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief シェーダパラメータのタイプを取得します。
    Type GetType() const { return static_cast<Type>(ref().type); }

    //! @brief UniformBlock 内でのデータ先頭へのオフセットを取得します。
    s32 GetOffset() const { return ref().offset; }

    //! @brief UniformBlock 内でのデータ先頭へのオフセットを設定します。
    //!
    //! シェーダでの UniformBlock の定義によってオフセットは変わります。
    //! シェーダから取得した値を設定してください。
    //!
    void SetOffset(s32 offset) { ref().offset = offset; }

    //! @brief パラメータのソースデータのオフセットを取得します。
    s32 GetSrcOffset() const { return ref().srcOffset; }

    //! @brief UniformBlock に書き込むサイズを取得します。
    size_t GetSize() const { return GetSize(GetType()); }

    //! @brief ソースデータのサイズを取得します。
    size_t GetSrcSize() const { return ref().srcSize; }

    //! @brief UniformBlock に書き込むサイズを取得します。
    static size_t GetSize(Type type);

    //! @brief ソースデータのサイズを取得します。
    static size_t GetSrcSize(Type type);

    //! @brief 変換コールバックを取得します。
    ConvertParamCallback GetConvertParamCallback() const
    {
        void* ptr = const_cast<void*>(pCallback.to_ptr());
        return reinterpret_cast<ConvertParamCallback>(ptr); // 警告対策
    }

    //! @brief 変換コールバックを設定します。
    void SetConvertParamCallback(ConvertParamCallback callback)
    {
        ref().pCallback.set_ptr(reinterpret_cast<void*>(callback));
    }

    //! @brief 依存元のインデックスを取得します。
    int GetDependedIndex() const { return dependedIndex; }

    //! @brief 依存先のインデックスを取得します。
    int GetDependIndex() const { return dependIndex; }

    //! SRT2D 変換用のコールバック関数です。
    static size_t ConvertSrt2dCallback(void*, const void*, Type, const void*);

    //! SRT3D 変換用のコールバック関数です。
    static size_t ConvertSrt3dCallback(void*, const void*, Type, const void*);

    //! SRT2DEx 変換用のコールバック関数です。
    static size_t ConvertSrt2dExCallback(void*, const void*, Type, const void*);

    //! TexSRT 変換用のコールバック関数です。
    static size_t ConvertTexSrtCallback(void*, const void*, Type, const void*);

    //! TexSRTEx 変換用のコールバック関数です。
    static size_t ConvertTexSrtExCallback(void*, const void*, Type, const void*);

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief マテリアルの構造体です。
struct ResMaterialData
{
    BinaryBlockHeader blockHeader;

    BinString ofsName;

    bit32 flag;
    u16 index; //!< モデル内でのインデックス
    u16 numRenderInfo;
    u8 numSampler;
    u8 numTexture;

    u16 numShaderParam;
    u16 numShaderParamVolatile;
    u16 srcParamSize; //!< ShaderParam の変換前データの総サイズ
    u16 rawParamSize; //!< ShaderParam の変換後データの総サイズ
    u16 numUserData;

    Offset ofsRenderInfoDic;
    Offset ofsRenderState;
    Offset ofsShaderAssign;
    Offset ofsTextureRefArray;
    Offset ofsSamplerArray;
    Offset ofsSamplerDic;
    Offset ofsShaderParamArray;
    Offset ofsShaderParamDic;
    Offset ofsSrcParam; //!< ShaderParam の変換前データのバッファ
    Offset ofsUserDataDic;
    Offset ofsVolatileFlag;

    BinPtr pUserPtr;
};

//! @brief マテリアルのリソースです。
class ResMaterial : private ResMaterialData
{
    NW_G3D_RES_COMMON(ResMaterial);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'M', 'A', 'T') };

    //! @brief マテリアルに関するフラグです。
    enum Flag
    {
        //! @brief ビジビリティの初期値です。
        VISIBILITY                  = 0x1 << 0
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief マテリアルをセットアップします。
    void Setup();

    //! @brief マテリアルをクリーンアップします。
    void Cleanup();

    //! @brief マテリアルをリセットします。
    void Reset();

    //! @brief 外部参照されているリソースの参照解決を行います。
    BindResult Bind(const ResFile* pFile);

    //! @brief 外部参照されているリソースの参照解決を行います。
    BindResult Bind(const ResFile* pFile, BindCallback* pCallback);

    //! @brief 指定した名前のテクスチャの参照を差し替えます。
    bool ForceBind(const ResTexture* pTexture, const char* name);

    //! @brief リソースの参照設定を解除します。
    void Release();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief モデル内でのインデクスを取得します。
    int GetIndex() const { return static_cast<int>(ref().index); }

    NW_G3D_RES_FIELD_STRING_DECL(Name)

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { ref().pUserPtr.set_ptr(pUserPtr); }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return ref().pUserPtr.to_ptr<T>(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return ref().pUserPtr.to_ptr<T>(); }

    //! @brief 指定したパラメータが Volatile 属性かどうかを取得します。
    bool IsVolatile(int paramIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(paramIndex, ref().numShaderParam);
        return (ref().ofsVolatileFlag.to_ptr<bit32>()[paramIndex >> 5]
            & (0x1 << (paramIndex & 0x1F))) != 0;
    }

    //! @brief 指定したパラメータに Volatile 属性を設定します。
    void SetVolatile(int paramIndex)
    {
        bit32& flag = ref().ofsVolatileFlag.to_ptr<bit32>()[paramIndex >> 5];
        bit32 mask = (0x1 << (paramIndex & 0x1F));
        if ((flag & mask) == 0)
        {
            flag ^= mask;
            ++ref().numShaderParamVolatile;
        }
    }

    //! @brief 指定したパラメータの Volatile 属性を解除します。
    void ResetVolatile(int paramIndex)
    {
        bit32& flag = ref().ofsVolatileFlag.to_ptr<bit32>()[paramIndex >> 5];
        bit32 mask = (0x1 << (paramIndex & 0x1F));
        if ((flag & mask) != 0)
        {
            flag ^= mask;
            --ref().numShaderParamVolatile;
        }
    }

    //! @brief Volatile 属性のパラメータをもっているかどうかを取得します。
    bool HasVolatile() const { return ref().numShaderParamVolatile > 0; }

    //@}

    //----------------------------------------
    //! @name 描画情報
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResRenderInfo, RenderInfo, GetName())

    //@}

    //----------------------------------------
    //! @name 描画ステート
    //@{

    NW_G3D_RES_FIELD_CLASS_DECL(ResRenderState, RenderState)

    //@}

    //----------------------------------------
    //! @name テクスチャ
    //@{

    //! @brief インスタンスで保持するテクスチャの数を設定します。
    void SetTextureCount(int textureCount)
    {
        NW_G3D_ASSERT(GetSamplerCount() <= textureCount);
        ref().numTexture = static_cast<u8>(textureCount);
    }

    //! @brief インスタンスで保持するテクスチャの数を取得します。
    int GetTextureCount() const { return ref().numTexture; }

    //! @brief テクスチャ参照を取得します。
    ResTextureRef* GetTextureRef(int texIndex)
    {
        NW_G3D_RES_ASSERT_INDEX(Sampler, texIndex);
        ResTextureRefData* pArray = ref().ofsTextureRefArray.to_ptr<ResTextureRefData>();
        return ResTextureRef::ResCast(&pArray[texIndex]);
    }

    //! @brief テクスチャ参照を取得します。
    const ResTextureRef* GetTextureRef(int texIndex) const
    {
        NW_G3D_RES_ASSERT_INDEX(Sampler, texIndex);
        const ResTextureRefData* pArray = ref().ofsTextureRefArray.to_ptr<ResTextureRefData>();
        return ResTextureRef::ResCast(&pArray[texIndex]);
    }

    //@}

    //----------------------------------------
    //! @name サンプラー
    //@{

    NW_G3D_RES_FIELD_CLASS_NAMED_ARRAY_DECL_DETAIL(ResSampler, Sampler, GetName())

    //@}

    //----------------------------------------
    //! @name シェーダパラメータ
    //@{

    //! @brief シェーダパラメータを取得します。
    void* GetSrcParam(int paramIndex)
    {
        return AddOffset(ref().ofsSrcParam.to_ptr(), GetShaderParam(paramIndex)->GetSrcOffset());
    }

    //! @brief シェーダパラメータを取得します。
    const void* GetSrcParam(int paramIndex) const
    {
        return AddOffset(ref().ofsSrcParam.to_ptr(), GetShaderParam(paramIndex)->GetSrcOffset());
    }

    //! @brief シェーダパラメータを取得します。
    //!
    //! テンプレート引数を指定することで型チェックを行います。
    //! 真偽値は4バイトで格納されているため、BOOL、int、uint、s32、u32 などの
    //! 4バイトの型で取得する必要があります。
    //!
    template <typename T>
    T* GetSrcParam(int paramIndex)
    {
        const ResShaderParam* pParam = GetShaderParam(paramIndex);
        NW_G3D_ASSERT(sizeof(T) <= pParam->GetSrcSize());
        return AddOffset<T>(ref().ofsSrcParam.to_ptr(), pParam->GetSrcOffset());
    }

    //! @brief シェーダパラメータを取得します。
    //!
    //! テンプレート引数を指定することで型チェックを行います。
    //! 真偽値は4バイトで格納されているため、BOOL、int、uint、s32、u32 などの
    //! 4バイトの型で取得する必要があります。
    //!
    template <typename T>
    const T* GetSrcParam(int paramIndex) const
    {
        const ResShaderParam* pParam = GetShaderParam(paramIndex);
        NW_G3D_ASSERT(sizeof(T) <= pParam->GetSrcSize());
        return AddOffset<T>(ref().ofsSrcParam.to_ptr(), pParam->GetSrcOffset());
    }

    //! @brief シェーダパラメータのソースデータのバッファサイズを取得します。
    u32 GetSrcParamSize() const { return ref().srcParamSize; }

    //! @brief シェーダパラメータの UniformBlock のバッファサイズを取得します。
    u32 GetRawParamSize() const { return ref().rawParamSize; }

    //! @brief シェーダパラメータの UniformBlock のバッファサイズを設定します。
    //!
    //! シェーダでの UniformBlock の定義によってサイズは変わります。
    //! シェーダから取得した値を設定してください。
    //!
    void SetRawParamSize(u32 size) { ref().rawParamSize = static_cast<u16>(size); }

    NW_G3D_RES_FIELD_CLASS_NAMED_ARRAY_DECL_DETAIL(ResShaderParam, ShaderParam, GetName())

    //! @brief シェーダパラメータを構築します。
    //!
    //! 通常は使用する必要はありません。
    void BuildShaderParam();

    //@}

    //----------------------------------------
    //! @name シェーダ
    //@{

    NW_G3D_RES_FIELD_CLASS_DECL(ResShaderAssign, ShaderAssign)

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())


    //@}
};

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS

#endif // NW_G3D_RES_RESMATERIAL_H_
