﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_MATH_MATRIX34_H_
#define NW_G3D_MATH_MATRIX34_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/math/g3d_Vector4.h>

namespace nw { namespace g3d { namespace math {

class Vec3;
class Quat;

#if defined( _MSC_VER )
#pragma warning(push)
#pragma warning(disable:4201)
#endif
struct Mtx34_t
{
    enum { DIM_MAJOR = 3, DIM_MINOR = 4, DIM = DIM_MAJOR * DIM_MINOR, DIM_PS = DIM / 2 };

    union
    {
        struct
        {
            float m00, m01, m02, m03;
            float m10, m11, m12, m13;
            float m20, m21, m22, m23;
        };
        float a[DIM];
        float m[DIM_MAJOR][DIM_MINOR];
        Vec4 v[DIM_MAJOR];
#if NW_G3D_IS_HOST_CAFE
        f32x2 ps[DIM_PS];
#endif
    };
};
#if defined( _MSC_VER )
#pragma warning(pop)
#endif

class Mtx34 : public Mtx34_t
{
public:
    static Mtx34 Make(
        float m00, float m01, float m02, float m03,
        float m10, float m11, float m12, float m13,
        float m20, float m21, float m22, float m23);
    static Mtx34* Cast(float* a);
    static const Mtx34* Cast(const float* a);

    // セット
    Mtx34& Set(
        float m00, float m01, float m02, float m03,
        float m10, float m11, float m12, float m13,
        float m20, float m21, float m22, float m23);
    Mtx34& Set(const float* a);
    Mtx34& Set(const Mtx34& m);
    Mtx34& Set(const Vec3& x, const Vec3& y, const Vec3& z, const Vec3& w);
    Mtx34& Zero();
    Mtx34& Identity();
    Mtx34& Neg(const Mtx34& m);

    // 値の生成
    // 関係しない成分には何も書き込まない
    Mtx34& SetS(const Vec3& s);
    Mtx34& SetR(const Vec3& r);
    Mtx34& SetR(const Quat& r);
    Mtx34& SetT(const Vec3& t);
    Mtx34& SetSR(const Vec3& s, const Vec3& r);
    Mtx34& SetSR(const Vec3& s, const Quat& r);

    // 加減乗除
    Mtx34& Add(const Mtx34& lhs, const Mtx34& rhs);
    Mtx34& Sub(const Mtx34& lhs, const Mtx34& rhs);
    Mtx34& Mul(const Mtx34& lhs, const Mtx34& rhs);
    Mtx34& Mul(const Mtx34& lhs, float rhs);
    Mtx34& Div(const Mtx34& lhs, float rhs);

    //
    static float Det(const Mtx34& m);
    Mtx34& Transpose(const Mtx34& m);
    Mtx34& Adjugate(const Mtx34& m); // 行列式で除算する前の逆行列
    Mtx34& Cofactor(const Mtx34& m); // 行列式で除算する前の逆転置行列
    Mtx34& Inverse(float* pDet, const Mtx34& m);
    Mtx34& InvTranspose(float* pDet, const Mtx34& m);

    Mtx34& ScaleBases(const Mtx34& m, const Vec3& s);
    static float ExtractBaseScale(const Mtx34& m, int axis);
    static float ExtractBaseScaleSq(const Mtx34& m, int axis);
    Mtx34& LookAt(const Vec3& camPos, const Vec3& camUp, const Vec3& target);

    //! @brief テクスチャマトリクス用射影行列を視錐体を元に作成します。
    //!
    //! @param[in] l ニアクリップの左辺です。
    //! @param[in] r ニアクリップの右辺です。
    //! @param[in] b ニアクリップの下辺です。
    //! @param[in] t ニアクリップの上辺です。
    //! @param[in] n カメラからニアクリップまでの距離です。
    //!
    //! @return テクスチャマトリクス用の射影行列です。
    //!
    Mtx34& TexProjFrustum(float l, float r, float b, float t, float n);

    //! @brief テクスチャマトリクス用射影行列を作成します。
    //!
    //! @param[in] fovy 垂直視野角です。
    //! @param[in] aspect クリップ面のアスペクト比です。
    //!
    //! @return テクスチャマトリクス用の射影行列です。
    //!
    Mtx34& TexProjPerspective(float fovy, float aspect);

    //! @brief 平行射影行列を作成します。
    //!
    //! @param[in] l ニアクリップの左辺です。
    //! @param[in] r ニアクリップの右辺です。
    //! @param[in] b ニアクリップの下辺です。
    //! @param[in] t ニアクリップの上辺です。
    //!
    //! @return テクスチャマトリクス用の射影行列です。
    //!
    Mtx34& TexProjOrtho(float l, float r, float b, float t);
};

}}} // namespace nw::g3d::math

#include "inl/g3d_Matrix34-inl.h"

#endif // NW_G3D_MATH_MATRIX34_H_
