﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_FND_GFXOBJECT_H_
#define NW_G3D_FND_GFXOBJECT_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GX2Struct.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

#define NW_G3D_GFX_OBJECT(class_name)                                                              \
public:                                                                                            \
    /* ---------------------------------------- */                                                 \
    /*! @name 共通 */                                                                              \
    /*@{ */                                                                                        \
    /*! @brief コンストラクタです。 */                                                             \
    class_name();                                                                                  \
    /*! @briefprivate */                                                                       \
    typedef class_name self_type;                                                                  \
    /*! @briefprivate */                                                                       \
    typedef class_name##_t base_type;                                                              \
    /*! @brief class_name##_t から class_name に変換します。 */                                    \
    static self_type* DownCast(base_type* ptr) { return static_cast<self_type*>(ptr); }            \
    /*! @brief class_name##_t から class_name に変換します。 */                                    \
    static const self_type* DownCast(const base_type* ptr)                                         \
        { return static_cast<const self_type*>(ptr); }                                             \
    /*! @brief class_name##_t から class_name に変換します。 */                                    \
    static self_type& DownCast(base_type& ref) { return static_cast<self_type&>(ref); }            \
    /*! @brief class_name##_t から class_name に変換します。 */                                    \
    static const self_type& DownCast(const base_type& ref)                                         \
        { return static_cast<const self_type&>(ref); }                                             \
    /*@} */                                                                                        \
private:                                                                                           \
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(class_name)                                                    \


namespace nw { namespace g3d { namespace fnd {

// 1. コンストラクタで初期化またはバイナリからダウンキャスト
// 2. 必要に応じて setter で設定
// 3. Setup() で構築
// 4. 変更した場合は Endedit() で反映
// 5. Load 系関数で描画に使用
// 6. Cleanup() で破棄

//--------------------------------------------------------------------------------------------------

//! @brief 頂点バッファ、インデクスバッファ、UniformBlock に用いるバッファの構造体です。
struct GfxBuffer_t
{
    union
    {
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
        void* pData;
#endif
        u32 dataUIntPtr;
    };
    u32 size;
    u32 handle;
    u16 stride;
    u16 numBuffering;

    union
    {
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
        GX2StreamOutContext* pCtxPtr;
#endif
        u32 ctxUIntPtr;
    };
};

//! @brief 頂点バッファ、インデクスバッファ、UniformBlock に用いるバッファです。
class GfxBuffer : public GfxBuffer_t
{
    NW_G3D_GFX_OBJECT(GfxBuffer);

public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 設定された情報に基づいてバッファを構築します。
    void Setup();

    //! @brief バッファを破棄します。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

     //! @brief Setup() の後に行った変更を反映します。
    void UpdateRegs();

    //! @brief CPU のデータキャッシュをフラッシュして GPU が参照可能にします。
    void DCFlush(int bufferIndex = 0) const;

    //! @brief GPU の出力バッファをフラッシュして CPU が参照可能にします。
    void FlushExportBuffer(int bufferIndex = 0) const;

    //! @brief GPU が書き込んだデータをメモリに反映します。
    void DCRefresh();

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief PC 版ではインデクスバッファを設定します。
    void LoadIndices() const;

    //! @brief 実機版では頂点バッファを設定します。
    void LoadVertices(u32 slot, int bufferIndex = 0) const;

    //! @brief 頂点シェーダへ UniformBlock を設定します。
    void LoadVertexUniforms(u32 location, int bufferIndex = 0) const;

    //! @brief ジオメトリシェーダへ UniformBlock を設定します。
    void LoadGeometryUniforms(u32 location, int bufferIndex = 0) const;

    //! @brief フラグメントシェーダへ UniformBlock を設定します。
    void LoadFragmentUniforms(u32 location, int bufferIndex = 0) const;

    //! @brief 演算シェーダへ UniformBlock を設定します。
    void LoadComputeUniforms(u32 location, int bufferIndex = 0) const;

    //! @brief ストリームアウト先のバッファを設定します。
    void LoadStreamOutBuffer(u32 location);

    //! @brief ExportBuffer を設定します。
    void LoadExportBuffer(int bufferIndex = 0) const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GPU から参照するデータを設定します。
    void SetData(void* pData, u32 size, int bufferingCount = 1);

    //! @brief GPU から参照するデータを取得します。
    void* GetData(int bufferIndex = 0);

    //! @brief GPU から参照するデータを取得します。
    const void* GetData(int bufferIndex = 0) const;

    //! @brief バッファリング数を取得します。
    int GetBufferingCount() const { return numBuffering; }

    //! @brief データサイズを取得します。
    //!
    //! 実際にアクセスするメモリのサイズはデータサイズ×バッファリング数です。
    //!
    u32 GetSize() const { return size; }

    //! @brief 頂点データ間のストライドを設定します。
    void SetStride(u32 stride) { this->stride = static_cast<u16>(stride); }

    //! @brief 頂点データ間のストライドを取得します。
    u32 GetStride() const { return stride; }

    //! @brief ストリームアウトコンテクストを設定します。
    void SetStreamOutContext(GX2StreamOutContext* pCtx);

    //! @brief ストリームアウトコンテクストを取得します。
    GX2StreamOutContext* GetStreamOutContext();

    //! @brief ストリームアウトコンテクストを取得します。
    const GX2StreamOutContext* GetStreamOutContext() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief テクスチャサンプラーの構造体です。
struct GfxSampler_t
{
    GX2Sampler gx2Sampler;
    u32 handle;
};

//! @brief テクスチャサンプラーです。
class GfxSampler : public GfxSampler_t
{
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    NW_G3D_GFX_OBJECT(GfxSampler);

    //@}

public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 設定された情報に基づいてサンプラーを構築します。
    void Setup();

    //! @brief サンプラーを破棄します。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief Setup() の後に行った変更を反映します。
    void UpdateRegs();

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief 頂点シェーダにサンプラーを設定します。
    void LoadVertexSampler(u32 unit) const;

    //! @brief ジオメトリシェーダにサンプラーを設定します。
    void LoadGeometrySampler(u32 unit) const;

    //! @brief フラグメントシェーダにサンプラーを設定します。
    void LoadFragmentSampler(u32 unit) const;

    //! @brief 演算シェーダにサンプラーを設定します。
    void LoadComputeSampler(u32 unit) const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトを取得します。
    GX2Sampler* GetGX2Sampler()
    {
        return reinterpret_cast<GX2Sampler*>(&gx2Sampler);
    }

    //! @brief GX2 オブジェクトを取得します。
    const GX2Sampler* GetGX2Sampler() const
    {
        return reinterpret_cast<const GX2Sampler*>(&gx2Sampler);
    }

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief X 方向のクランプを設定します。
    void SetClampX(GX2TexClamp clamp);

    //! @brief Y 方向のクランプを設定します。
    void SetClampY(GX2TexClamp clamp);

    //! @brief Z 方向のクランプを設定します。
    void SetClampZ(GX2TexClamp clamp);

    //! @brief ボーダー方式を設定します。
    void SetBorderType(GX2TexBorderType type);

    //! @brief X 方向のクランプを取得します。
    GX2TexClamp GetClampX() const;

    //! @brief Y 方向のクランプを取得します。
    GX2TexClamp GetClampY() const;

    //! @brief Z 方向のクランプを取得します。
    GX2TexClamp GetClampZ() const;

    //! @brief ボーダー方式を取得します。
    GX2TexBorderType GetBorderType() const;

    //! @brief 拡大フィルタを設定します。
    void SetMagFilter(GX2TexXYFilterType filter);

    //! @brief 縮小フィルタを設定します。
    void SetMinFilter(GX2TexXYFilterType filter);

    //! @brief Z フィルタを設定します。
    void SetZFilter(GX2TexZFilterType filter);

    //! @brief ミップマップフィルタを設定します。
    void SetMipFilter(GX2TexMipFilterType filter);

    //! @brief 異方性フィルタの最大値を設定します。
    void SetMaxAniso(GX2TexAnisoRatio ratio);

    //! @brief 拡大フィルタを取得します。
    GX2TexXYFilterType GetMagFilter() const;

    //! @brief 縮小フィルタを取得します。
    GX2TexXYFilterType GetMinFilter() const;

    //! @brief Z フィルタを取得します。
    GX2TexZFilterType GetZFilter() const;

    //! @brief ミップマップフィルタを取得します。
    GX2TexMipFilterType GetMipFilter() const;

    //! @brief 異方性フィルタの最大値を取得します。
    GX2TexAnisoRatio GetMaxAniso() const;

    //! @brief サンプリングの最小 LOD を設定します。
    void SetMinLOD(float minLOD);

    //! @brief サンプリングの最大 LOD を設定します。
    void SetMaxLOD(float maxLOD);

    //! @brief LOD のバイアス値を設定します。
    void SetLODBias(float bias);

    //! @brief サンプリングの最小 LOD を取得します。
    float GetMinLOD() const;

    //! @brief サンプリングの最大 LOD を取得します。
    float GetMaxLOD() const;

    //! @brief LOD のバイアス値を取得します。
    float GetLODBias() const;

    //! @brief デプステストとステンシルテストに用いる比較関数を設定します。
    void SetCompareFunc(GX2CompareFunction func);

    //! @brief デプステストとステンシルテストに用いる比較関数を取得します。
    GX2CompareFunction GetCompareFunc() const;

    //! @brief 比較関数を有効にするかどうかを設定します。
    //!
    //! NVIDIA の GPU では depth texture をサンプルする際に、
    //! shader 内の sampler の種類によらず比較の有効無効を設定する必要があります。
    //!
    void SetCompareEnable(GX2Boolean enable);

    //@}

protected:
    //! @briefprivate
    GX2Boolean GetCompareEnable() const; // 実機では常に false なので公開しない
};

//--------------------------------------------------------------------------------------------------

//! @brief テクスチャの構造体です。
struct GfxTexture_t
{
    detail::GX2TextureData gx2Texture;
    u32 handle;
    u32 arrayLength;
};

//! @brief テクスチャです。
class GfxTexture : public GfxTexture_t
{
    NW_G3D_GFX_OBJECT(GfxTexture);

public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 設定された情報に基づいてテクスチャを構築します。
    void Setup();

    //! @brief テクスチャを破棄します。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief Setup() の後に行った変更を反映します。
    void UpdateRegs();

    //! @brief テクスチャのサイズとアライメントを計算して反映します。
    void CalcSize();

    //! @brief CPU のデータキャッシュをフラッシュして GPU が参照可能にします。
    void DCFlush() const;

    //! @brief GPU が書き込んだデータをメモリに反映します。
    void DCRefresh();

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief 頂点シェーダにテクスチャを設定します。
    void LoadVertexTexture(u32 unit) const;

    //! @brief ジオメトリシェーダにテクスチャを設定します。
    void LoadGeometryTexture(u32 unit) const;

    //! @brief フラグメントシェーダにテクスチャを設定します。
    void LoadFragmentTexture(u32 unit) const;

    //! @brief 演算シェーダにテクスチャを設定します。
    void LoadComputeTexture(u32 unit) const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトを取得します。
    GX2Texture* GetGX2Texture()
    {
        return reinterpret_cast<GX2Texture*>(&gx2Texture);
    }

    //! @brief GX2 オブジェクトを取得します。
    const GX2Texture* GetGX2Texture() const
    {
        return reinterpret_cast<const GX2Texture*>(&gx2Texture);
    }

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief 次元を取得します。
    GX2SurfaceDim GetDimension() const { return gx2Texture.surface.dim; }

    //! @brief フォーマットを取得します。
    GX2SurfaceFormat GetFormat() const { return gx2Texture.surface.format; }

    //! @brief 幅を取得します。
    u32 GetWidth() const { return gx2Texture.surface.width; }

    //! @brief 高さを取得します。
    u32 GetHeight() const { return gx2Texture.surface.height; }

    //! @brief 深度を取得します。
    u32 GetDepth() const { return gx2Texture.surface.depth; }

    //! @brief ミップマップの数を取得します。
    u32 GetMipLevels() const { return gx2Texture.surface.numMips; }

    //! @brief ミップマップ０のイメージサイズを取得します。
    u32 GetBaseSize() const { return gx2Texture.surface.imageSize; }

    //! @brief ミップマップ１以上の合計サイズを取得します。
    u32 GetMipSize() const { return gx2Texture.surface.mipSize; }

    //! @brief 画像データへのポインタを設定します。
    void SetImagePtrs(void* basePtr, void* mipPtr);

    //! @brief ミップマップ０の画素へのポインタを取得します。
    void* GetBasePtr();

    //! @brief ミップマップ０の画素へのポインタを取得します。
    const void* GetBasePtr() const;

    //! @brief ミップマップ１の画素へのポインタを取得します。
    void* GetMipPtr();

    //! @brief ミップマップ１の画素へのポインタを取得します。
    const void* GetMipPtr() const;

    //! @brief 画像データへのポインタを取得します。
    void* GetImagePtr(int mipLevel);

    //! @brief 画像データへのポインタを取得します。
    const void* GetImagePtr(int mipLevel) const;

    //! @brief アライメントを取得します。
    u32 GetAlignment() const { return gx2Texture.surface.alignment; }

    //! @brief パディングされた要素の横幅を取得します。
    u32 GetPitch() const { return gx2Texture.surface.pitch; }

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief カラーバッファの構造体です。
struct GfxColorBuffer_t
{
    detail::GX2ColorBufferData gx2ColorBuffer;
    union
    {
        u32 alignmentAux; // for GX2
        u32 handle; // for GL
    };
};

//! @brief カラーバッファです。
class GfxColorBuffer : public GfxColorBuffer_t
{
    NW_G3D_GFX_OBJECT(GfxColorBuffer);

public:
    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief 変更を反映します。
    void UpdateRegs();

    //! @brief Final TV として変更を反映します。
    void UpdateRegsFTV();

    //! @brief 補助バッファのサイズとアライメントを計算して反映します。
    void CalcAuxSize();

    //! @brief CPU のデータキャッシュをフラッシュして GPU が参照可能にします。
    void DCFlush() const;

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief カラーバッファを設定します。
    void Load(GX2RenderTarget target);

    //! @brief カラーバッファを設定します。
    void Load(GX2RenderTarget target, u32 hFrameBuffer);

    //! @brief 指定した色でクリアします。
    //!
    //! デプスステンシルバッファと同時にクリアする場合は ClearBuffers() を使用した方が高速です。
    void Clear(float r, float g, float b, float a);

    //! @brief 補助バッファを展開してカラーバッファを MSAA テクスチャとして使用可能にします。
    void ExpandAux();

    //! @brief MSAA テクスチャをシングルサンプルテクスチャに解決します。
    void Resolve(GfxTexture* pTexture, u32 mipLevel, u32 slice) const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトを取得します。
    GX2ColorBuffer* GetGX2ColorBuffer()
    {
        return reinterpret_cast<GX2ColorBuffer*>(&gx2ColorBuffer);
    }

    //! @brief GX2 オブジェクトを取得します。
    const GX2ColorBuffer* GetGX2ColorBuffer() const
    {
        return reinterpret_cast<const GX2ColorBuffer*>(&gx2ColorBuffer);
    }

    //! @brief テクスチャの情報に基づいて設定します。
    void SetTexture(GfxTexture* pTexture);

    //! @brief テクスチャの画像データへのポインタを設定します。
    void SetImagePtrs(GfxTexture* pTexture);

    //! @brief 補助バッファへのポインタを設定します。
    void SetAuxPtr(void* auxPtr);

    //! @brief 補助バッファへのポインタを取得します。
    void* GetAuxPtr();

    //! @brief 補助バッファへのポインタを取得します。
    const void* GetAuxPtr() const;

    //! @brief 補助バッファのサイズを取得します。
    u32 GetAuxSize() const;

    //! @brief 補助バッファのアライメントを取得します。
    u32 GetAuxAlignment() const;

    // Surface 関連の情報が必要であれば GfxTexture から取得する。

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief デプスステンシルバッファの構造体です。
struct GfxDepthBuffer_t
{
    detail::GX2DepthBufferData gx2DepthBuffer;
    union
    {
        u32 alignmentHiZ; // for GX2
        u32 handle; // for GL
    };
};

//! @brief デプスステンシルバッファです。
class GfxDepthBuffer : public GfxDepthBuffer_t
{
    NW_G3D_GFX_OBJECT(GfxDepthBuffer);

public:
    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief 変更を反映します。
    void UpdateRegs();

    //! @brief HiZ バッファのサイズとアライメントを計算して反映します。
    void CalcHiZSize();

    //! @brief CPU のデータキャッシュをフラッシュして GPU が参照可能にします。
    void DCFlush() const;

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief デプスステンシルバッファを設定します。
    void Load();

    //! @brief デプスステンシルバッファを設定します。
    void Load(u32 hFrameBuffer);

    //! @brief デプスバッファをクリアします。
    //!
    //! カラーバッファと同時にクリアする場合は ClearBuffers() を使用した方が高速です。
    void Clear(GX2ClearMode mode);

    //! @brief HiZ バッファを展開してデプスバッファをデプステクスチャとして使用可能にします。
    void ExpandHiZ();

    //! @brief デプスバッファをテクスチャとして使用可能なフォーマットに変換します。
    void ConvertToTexture(GfxTexture* pTexture, u32 mipLevel, u32 slice) const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトを取得します。
    GX2DepthBuffer* GetGX2DepthBuffer()
    {
        return reinterpret_cast<GX2DepthBuffer*>(&gx2DepthBuffer);
    }

    //! @brief GX2 オブジェクトを取得します。
    const GX2DepthBuffer* GetGX2DepthBuffer() const
    {
        return reinterpret_cast<const GX2DepthBuffer*>(&gx2DepthBuffer);
    }

    //! @brief テクスチャの情報に基づいて設定します。
    void SetTexture(GfxTexture* pTexture);

    //! @brief テクスチャの画像データへのポインタを設定します。
    void SetImagePtrs(GfxTexture* pTexture);

    //! @brief HiZ バッファへのポインタを設定します。
    void SetHiZPtr(void* hiZPtr);

    //! @brief HiZ バッファへのポインタを取得します。
    void* GetHiZPtr();

    //! @brief HiZ バッファへのポインタを取得します。
    const void* GetHiZPtr() const;

    //! @brief HiZ バッファのサイズを取得します。
    u32 GetHiZSize() const;

    //! @brief HiZ バッファのアライメントを取得します。
    u32 GetHiZAlignment() const;

    //! @brief HiZ バッファが有効かどうかを設定します。
    //!
    //! UpdateRegs() を呼んだ際に HiZPtr がセットされているかどうかによって再設定されます。
    void SetHiZEnable(GX2Boolean enable);

    //! @brief HiZ バッファが有効かどうかを取得します。
    GX2Boolean GetHiZEnable() const;

    //! @brief Clear に使う depth/stencil 値を設定します。
    void SetClearDepthStencil(float depth, u8 stencil);

    //! @brief Clear に使う depth 値を取得します。
    float GetClearDepth() const;

    //! @brief Clear に使う stencil 値を取得します。
    u8 GetClearStencil() const;

    // Surface 関連の情報が必要であれば GfxTexture から取得する。

    //@}
};

//--------------------------------------------------------------------------------------------------
//! @brief ストリームアウトコンテクストの構造体です。
struct GfxStreamOut_t
{
    union
    {
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
        GfxBuffer* pStream[GX2_MAX_STREAMOUT_BUFFERS];
#endif
        u32 streamUIntPtr[GX2_MAX_STREAMOUT_BUFFERS];
    };
    GX2PrimitiveType primType;
    u32 handle; // for PC
};

//! @brief ストリームアウトコンテクストです。
class GfxStreamOut : public GfxStreamOut_t
{
    NW_G3D_GFX_OBJECT(GfxStreamOut);

public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief ストリームアウトコンテクストを構築します。
    void Setup();

    //! @brief ストリームアウトコンテクストを破棄します。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! ストリームアウトコンテクストを設定します。
    void Load();

    //! @brief ストリームアウトを開始します。
    void Begin();

    //! @brief ストリームアウトを終了します。
    void End();

    //! @brief ストリームアウトを一時停止します。
    void Pause();

    //! @brief ストリームアウトを再開します。
    void Resume();

    //! @brief ストリームアウトバッファを使って描画します。
    void Draw(u32 location);

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief プリミティブタイプを設定します。
    void SetPrimitiveType(GX2PrimitiveType type) { primType = type; }

    //! @brief プリミティブタイプを取得します。
    GX2PrimitiveType GetPrimitiveType() const { return primType; }

    //! @brief ストリームアウトバッファを設定します。
    void SetStreamOutBuffer(u32 location, GfxBuffer* pBuffer);

    //! @brief ストリームアウトバッファを取得します。
    GfxBuffer* GetStreamOutBuffer(u32 location);

    //! @brief ストリームアウトバッファを取得します。
    const GfxBuffer* GetStreamOutBuffer(u32 location) const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief 頂点属性のフェッチシェーダの構造体です。
struct GfxFetchShader_t
{
    detail::GX2FetchShaderData gx2FetchShader;
    u32 ofsVFInst;
    u32 handle;
};

//! @brief 頂点属性のフェッチシェーダです。
class GfxFetchShader : public GfxFetchShader_t
{
    NW_G3D_GFX_OBJECT(GfxFetchShader);

    enum
    {
        MAX_INST_PER_FETCH_CLAUSE = 16,
        FETCH_INST_ALIGNMENT = 16,
        CF_INST_SIZE = 8,
        VF_INST_SIZE = 16
    };

public:
    enum
    {
        SHADER_ALIGNMENT = 256 // GX2_SHADER_ALIGNMENT
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 設定された情報に基づいてフェッチシェーダを構築します。
    //!
    //! SetAttribCount() で属性数を設定してから CalcSize() を呼び、
    //! SetShaderPtr() に GetShaderSize() 以上のバッファを設定して SetDefault()、
    //! 各種設定を行った後に Setup() を呼びます。
    //! shaderPtr の内容を変更した場合は描画前にキャッシュを Invalidate する必要があります。
    //! GPU が使用している間に変更しないようにしてください。
    //!
    void Setup();

    //! @brief フェッチシェーダを破棄します。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief Setup() の後に行った変更を反映します。
    void UpdateRegs();

    //! @brief CPU のデータキャッシュをフラッシュして GPU が参照可能にします。
    void DCFlush() const;

    //! @brief SetAttribCount() に基づいて ShaderSize を計算します。
    void CalcSize();

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief フェッチシェーダを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトを取得します。
    GX2FetchShader* GetGX2FetchShader()
    {
        return reinterpret_cast<GX2FetchShader*>(&gx2FetchShader);
    }

    //! @brief GX2 オブジェクトを取得します。
    const GX2FetchShader* GetGX2FetchShader() const
    {
        return reinterpret_cast<const GX2FetchShader*>(&gx2FetchShader);
    }

    //! @brief 属性数を設定します。
    void SetAttribCount(u32 count) { gx2FetchShader.num_attribs = count; }

    //! @brief 属性数を取得します。
    u32 GetAttribCount() const { return gx2FetchShader.num_attribs; }

    //! @brief フェッチシェーダに必要なサイズを取得します。
    u32 GetShaderSize() const { return gx2FetchShader.shaderSize; }

    //! @brief フェッチシェーダが用いるバッファを設定します。
    //!
    //! @param[in] ptr  SHADER_ALIGNMENT でアライメントされたバッファへのポインタです。
    //!
    void SetShaderPtr(void* ptr);

    //! @brief フェッチシェーダが用いるバッファを取得します。
    void* GetShaderPtr();

    //! @brief フェッチシェーダが用いるバッファを取得します。
    const void* GetShaderPtr() const;

    //! @brief Divisor を設定します。
    //!
    //! スロット 0 は divisor=0、スロット 1 は divisor=1 で予約されています。
    //! スロット 2 の値を divisor2 で、スロット 3 の値を divisor3 で指定します。
    //!
    void SetDivisors(u32 divisor2, u32 divisor3);

    //! @brief Divisor を取得します。
    u32 GetDivisor(int divisorIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(divisorIndex, 4);
        return divisorIndex < 2 ? divisorIndex : gx2FetchShader._divisors[divisorIndex - 2];
    }

    //! @brief 頂点バッファを設定します。
    void SetVertexBuffer(int attribIndex, const GfxBuffer* pBuffer);

    //! @brief 頂点バッファを取得します。
    const GfxBuffer* GetVertexBuffer(int attribIndex) const;

    //! @brief PC 版において VertexArrayObject に設定されている VertexBuffer を上書きします。
    //!
    //! PC 版以外では何も行いません。GetVertexBuffer() の結果には影響しません。
    //! slot が GetBufferSlot() の結果と一致する頂点属性の VertexBuffer を上書きします。
    //! シェイプアニメやシェイプの LOD のような、1つの FetchShader で異なる VertexBuffer を
    //! 差し替えながら使うような場合に使用します。
    //!
    void ReplaceVertexBuffer(u32 slot, const GfxBuffer* pBuffer) const;

    //! @brief PC 版において ReplaceVertexBuffer() の結果を元に戻します。
    //!
    //! PC 版以外では何も行いません。
    //!
    void ResetVertexBuffer() const;

#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
public:
    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault() { SetDefault(GetGX2FetchShader()->shaderPtr); }

    //! @brief 頂点属性のロケーションを設定します。
    void SetLocation(int attribIndex, u32 location)
    {
        SetLocation(GetGX2FetchShader()->shaderPtr, attribIndex, location);
    }

    //! @brief 頂点属性のロケーションを取得します。
    u32 GetLocation(int attribIndex) const
    {
        return GetLocation(GetGX2FetchShader()->shaderPtr, attribIndex);
    }

    //! @brief 頂点バッファの読み込み先スロットを設定します。
    void SetBufferSlot(int attribIndex, u32 slot)
    {
        SetBufferSlot(GetGX2FetchShader()->shaderPtr, attribIndex, slot);
    }

    //! @brief 頂点バッファの読み込み先スロットを取得します。
    u32 GetBufferSlot(int attribIndex) const
    {
        return GetBufferSlot(GetGX2FetchShader()->shaderPtr, attribIndex);
    }

    //! @brief 頂点属性のフォーマットを設定します。
    void SetFormat(int attribIndex, GX2AttribFormat format)
    {
        SetFormat(GetGX2FetchShader()->shaderPtr, attribIndex, format);
    }

    //! @brief 頂点属性のフォーマットを取得します。
    GX2AttribFormat GetFormat(int attribIndex) const
    {
        return GetFormat(GetGX2FetchShader()->shaderPtr, attribIndex);
    }

    //! @brief 頂点属性のエンディアンを設定します。
    //!
    //! SetFormat() によって上書きされるので、SetFormat() より後で設定してください。
    //!
    void SetEndianSwapMode(int attribIndex, GX2EndianSwapMode mode)
    {
        SetEndianSwapMode(GetGX2FetchShader()->shaderPtr, attribIndex, mode);
    }

    //! @brief 頂点属性のエンディアンを取得します。
    GX2EndianSwapMode GetEndianSwapMode(int attribIndex) const
    {
        return GetEndianSwapMode(GetGX2FetchShader()->shaderPtr, attribIndex);
    }

    //! @brief 頂点属性の Divisor のスロットを設定します。
    //!
    //! SetDivisors で divisor を設定したスロットの番号で設定します。
    //!
    void SetDivisorSlot(int attribIndex, u32 slot)
    {
        SetDivisorSlot(GetGX2FetchShader()->shaderPtr, attribIndex, slot);
    }

    //! @brief 頂点属性の Divisor のスロットを取得します。
    //!
    //! SetDivisors で divisor を設定したスロットの番号で取得します。
    //!
    u32 GetDivisorSlot(int attribIndex) const
    {
        return GetDivisorSlot(GetGX2FetchShader()->shaderPtr, attribIndex);
    }

    //! @brief バッファ内での頂点へのオフセットを設定します。
    void SetOffset(int attribIndex, u32 offset)
    {
        SetOffset(GetGX2FetchShader()->shaderPtr, attribIndex, offset);
    }

    //! @brief バッファ内での頂点へのオフセットを取得します。
    u32 GetOffset(int attribIndex) const
    {
        return GetOffset(GetGX2FetchShader()->shaderPtr, attribIndex);
    }

    //@}

protected:
    //! @briefprivate
    u32* GetVFInst(int attribIndex)
    {
        return GetVFInst(GetGX2FetchShader()->shaderPtr, attribIndex);
    }

    //! @briefprivate
    const u32* GetVFInst(int attribIndex) const
    {
        return GetVFInst(GetGX2FetchShader()->shaderPtr, attribIndex);
    }
#endif // NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE

    // 以下の関数は 64 ビット版のツール等に用いる場合以外は非推奨。
    // 将来的に変更される可能性があります。
public:
    //! @briefprivate
    void SetDefault(void* pShader);

    //! @briefprivate
    void SetLocation(void* pShader, int attribIndex, u32 location);

    //! @briefprivate
    u32 GetLocation(const void* pShader, int attribIndex) const;

    //! @briefprivate
    void SetBufferSlot(void* pShader, int attribIndex, u32 slot);

    //! @briefprivate
    u32 GetBufferSlot(const void* pShader, int attribIndex) const;

    //! @briefprivate
    void SetFormat(void* pShader, int attribIndex, GX2AttribFormat format);

    //! @briefprivate
    GX2AttribFormat GetFormat(const void* pShader, int attribIndex) const;

    //! @briefprivate
    void SetEndianSwapMode(void* pShader, int attribIndex, GX2EndianSwapMode mode);

    //! @briefprivate
    GX2EndianSwapMode GetEndianSwapMode(const void* pShader, int attribIndex) const;

    //! @briefprivate
    void SetDivisorSlot(void* pShader, int attribIndex, u32 slot);

    //! @briefprivate
    u32 GetDivisorSlot(const void* pShader, int attribIndex) const;

    //! @briefprivate
    void SetOffset(void* pShader, int attribIndex, u32 offset);

    //! @briefprivate
    u32 GetOffset(const void* pShader, int attribIndex) const;

protected:
    //! @briefprivate
    u32* GetVFInst(void* pShader, int attribIndex);

    //! @briefprivate
    const u32* GetVFInst(const void* pShader, int attribIndex) const;

    //! @briefprivate
    void LoadVertexAttribArray(u32 idxAttrib, const GfxBuffer* pBuffer) const;

    //! @briefprivate
    void LoadVertexAttribArray() const;

    //! @briefprivate
    void LoadVertexAttribValue() const;
};

//--------------------------------------------------------------------------------------------------

//! @brief 演算シェーダのディスパッチパラメータの構造体です。
struct GfxDispatchParams_t
{
#if NW_G3D_COMPUTE_SHADER_ENABLE
    GX2DispatchParams params;
#else
    detail::GX2DispatchParamsData params;
#endif
};

//! @brief 演算シェーダのディスパッチパラメータです。
class GfxDispatchParams : public GfxDispatchParams_t
{
    NW_G3D_GFX_OBJECT(GfxDispatchParams);

public:
    //! @brief Alignment を示す enum です。
    enum Alignment
    {
        //! @brief GfxDispatchParams クラスに要求されるアライメントです。
        CLASS_ALIGNMENT = GX2_DISPATCH_PARAMS_ALIGNMENT
    };
#if NW_G3D_COMPUTE_SHADER_ENABLE
    //! @brief GX2DispatchParams を取得します。
    GX2DispatchParams* GetGX2DispatchParams(){ return &params; }
    //! @brief GX2DispatchParams を取得します。
    const GX2DispatchParams* GetGX2DispatchParams() const { return &params; }
#endif

    //! @brief ディスパッチパラメータを設定します。
    void SetParams(uint num_gruops_x = 1, uint num_groups_y = 1, uint num_groups_z = 1);

    //! @brief GPU が参照できるように CPU のキャッシュをフラッシュします。
    void CPUFlush();

    //! @brief 設定されているパラメータで演算シェーダをディスパッチします。
    void Dispatch();
};

//--------------------------------------------------------------------------------------------------

void SetStreamOutEnable(GX2Boolean enable);

void SetRasterizerClipControl(GX2Boolean rasterizerEnable, GX2Boolean zClipEnable);

//! @brief インデクスの再描画開始位置を設定します。
void SetPrimitiveRestartIndex(u32 restartIndex);

//! @brief 指定した色とデプスステンシルでカラーバッファとデプスステンシルバッファを同時にクリアします。
//!
//! depth/stencil 値は pDepthBuffer に SetClearDepthStencil() で設定されたものを使います。
//!
void ClearBuffers(GfxColorBuffer* pColorBuffer, GfxDepthBuffer* pDepthBuffer,
    float r, float g, float b, float a, GX2ClearMode mode);

}}} // namespace nw::g3d::fnd

NW_G3D_PRAGMA_POP_WARNINGS

#endif // NW_G3D_FND_GFXOBJECT_H_
