﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_FONT_RESFONTBASE_H_
#define NW_FONT_RESFONTBASE_H_

#include <nw/types.h>
#include <nw/font/font_Font.h>
#include <nw/font/font_ResourceFormat.h>


namespace nw {
namespace font {

//---------------------------------------------------------------------------
//! @brief        リソースを扱うフォントクラスの基底クラスです。
//---------------------------------------------------------------------------
class ResFontBase : public Font
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(Font);

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */

    //! @name コンストラクタ/デストラクタ
    //@{

    //! コンストラクタです。
                            ResFontBase();

    //! デストラクタです。
    virtual                 ~ResFontBase();

    //@}

    //! @name フォント情報の取得
    //@{

    virtual int             GetWidth() const;

    virtual int             GetHeight() const;

    virtual int             GetAscent() const;

    virtual int             GetDescent() const;

    virtual int             GetMaxCharWidth() const;

    virtual Type            GetType() const;

    virtual TexFmt          GetTextureFormat() const;

    virtual int             GetLineFeed() const;

    virtual const CharWidths
                            GetDefaultCharWidths() const;

    //! @name フォント情報の設定
    //@{

    virtual void            SetLineFeed(int linefeed);

    virtual void            SetDefaultCharWidths(const CharWidths& widths);

    virtual bool            SetAlternateChar(CharCode c);

    //@}

    //! @name 文字情報の取得
    //@{

    virtual int             GetCharWidth(CharCode c) const;

    virtual const CharWidths
                            GetCharWidths(CharCode c) const;

    virtual void            GetGlyph(
                                Glyph*      pGlyph,
                                CharCode    c
                            ) const;

    virtual bool            HasGlyph(CharCode c) const;

    virtual int            GetKerning(CharCode c0, CharCode c1) const;

    //@}

    //! @name 文字列エンコーディング
    //@{

    virtual CharacterCode   GetCharacterCode() const;

    //@}

    //! @name シート情報の取得
    //@{

    virtual int             GetBaselinePos() const;
    virtual int             GetCellHeight() const;
    virtual int             GetCellWidth() const;

    //@}


    //! @name テクスチャ補間
    //@{

    virtual void            EnableLinearFilter(
                                bool    atSmall,
                                bool    atLarge);

    virtual bool            IsLinearFilterEnableAtSmall() const;

    virtual bool            IsLinearFilterEnableAtLarge() const;

    virtual u32             GetTextureWrapFilterValue() const;

    //@}

    //! @name 表示設定
    //@{

    //! @brief        カラーの白黒補間の影響の有無を取得します。
    //!
    //! @return       カラーの白黒補間の影響を受ける場合は、true を返します。
    //!
    virtual bool            IsColorBlackWhiteInterpolationEnabled() const;

    //! @brief        カラーの白黒補間の影響の有無を設定します。
    //!
    //! @details
    //! この設定でfalseを設定しても、アルファについては影響を受けます。
    //!
    //! @param[in]  flag    true を設定すると、カラーの白黒補間の影響を受けます。
    //!
    virtual void            SetColorBlackWhiteInterpolationEnabled(bool flag);

    //! @brief        縁取り効果の有無を取得します。
    //!
    //! @return       縁取り効果が有効な場合は、true を返します。
    //!
    virtual bool            IsBorderEffectEnabled() const;

    //@}

protected:
    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    typedef u16             GlyphIndex;

    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    static const GlyphIndex GLYPH_INDEX_NOT_FOUND = INVALID_GLYPH_INDEX;


    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- メンバアクセス

    FontInformation*        GetFINF()       { return m_pFontInfo; }

    const FontInformation*  GetFINF() const { return m_pFontInfo; }

    //! @brief      関連付けられているリソースフォントかどうか判断します。
    //!
    //! @param[in]  ptr 比較対象のリソースフォントへのポインタ。
    //!
    //! @return     この ResFont が ptr の指すフォントリソースと関連付けられている
    //!             ならば true を、そうでなければ false を返します。
    //!
    bool                    IsManaging(const void* ptr) const { return m_pResource == ptr; }

    //! @brief      割り当てられたバッファとそこに配置されている FINF ブロックへの
    //!             ポインタを設定します。
    //!
    //! @param[in]  pUserBuffer  割り当てられたバッファへのポインタ。
    //! @param[in]  pFontInfo    FINF ブロックへのポインタ。
    //! @param[in]  pKerningTable KRNG ブロックへのポインタ。
    //!
    void                    SetResourceBuffer(
                                void*               pUserBuffer,
                                FontInformation*    pFontInfo,
                                FontKerningTable*   pKerningTable);

    //! @brief      SetResourceBuffer で設定されたパラメータをリセットし、
    //!             割り当てられていたバッファへのポインタを返します。
    //!
    //! @return     設定されていたバッファへのポインタ。
    //!
    void*                   RemoveResourceBuffer();


    //---- グリフインデックス


    //! @brief      文字のグリフインデックスを取得します。
    //!
    //! @param[in]  c  グリフインデックスを取得する文字の文字コード。
    //!
    //! @return     文字のグリフインデックス。
    //!             フォントに対象の文字が含まれていない場合は代替文字の
    //!             グリフインデックスを返します。
    //!
    GlyphIndex              GetGlyphIndex(CharCode c) const;

    //! @brief      文字のグリフインデックスを取得します。
    //!
    //! @param[in]  c  グリフインデックスを取得する文字の文字コード。
    //!
    //! @return     文字のグリフインデックス。
    //!             フォントに対象の文字が含まれていない場合は
    //!             ResFontBase::GLYPH_INDEX_NOT_FOUND を返します。
    //!
    GlyphIndex              FindGlyphIndex(CharCode c) const;

    //! @brief      文字のグリフインデックスをCMAPブロックから探索します。
    //!
    //! @param[in]  pMap  探索対象のCMAPブロック本体へのポインタ。
    //! @param[in]  c     グリフインデックスを取得する文字の文字コード。
    //!
    //! @return     文字のグリフインデックス。
    //!             フォントに対象の文字が含まれていない場合は
    //!             ResFontBase::GLYPH_INDEX_NOT_FOUND を返します。
    //!
    GlyphIndex              FindGlyphIndex(
                                const FontCodeMap*  pMap,
                                CharCode            c
                            ) const;

    //! @brief      グリフインデックスに対応する文字の文字幅を取得します。
    //!
    //! @param[in]  index  文字幅情報を取得する文字のグリフインデックス
    //!
    //! @return     文字幅情報へのリファレンス。
    //!
    const CharWidths&       GetCharWidthsFromIndex(GlyphIndex index) const;

    //! @brief      CWDHブロック本体からグリフインデックスに対応する文字の
    //!             文字幅情報を取得します。
    //!
    //! @param[in]  pWidth  探索対象のCWDHブロック本体へのポインタ。
    //! @param[in]  index   文字幅情報を取得する文字のグリフインデックス
    //!
    //! @return     文字幅情報へのリファレンス。
    //!
    const CharWidths&       GetCharWidthsFromIndex(
                                const FontWidth*    pWidth,
                                GlyphIndex          index
                            ) const;

    //! @brief      グリフインデックスに対応するグリフデータを取得します。
    //!
    //! @param[out] glyph  グリフデータを格納するバッファへのポインタ。
    //! @param[in]  index  取得するグリフデータのグリフインデックス。
    //!
    void                    GetGlyphFromIndex(
                                Glyph*      glyph,
                                GlyphIndex  index
                            ) const;


    //! @brief      FontTextureGlyph構造体の内容をもとにGlyphのメンバを設定します。
    //!
    //! @param[out] glyph  グリフデータを格納するバッファへのポインタ。
    //! @param[in]  index  取得するグリフデータのグリフインデックス。
    //! @param[in]  tg     FontTextureGlyph構造体への参照。
    //!
    static void             SetGlyphMember(
                                Glyph*                  glyph,
                                GlyphIndex              index,
                                const FontTextureGlyph& tg);


public:
    //! @name シート情報の取得
    //@{

    //! @brief      アクティブなシートの数を取得します。
    //!
    //! @return     アクティブなシートの数を返します。
    //!
    virtual int             GetActiveSheetNum() const;

    //! @brief      テクスチャオブジェクトバッファへのポインタを取得します。
    //!
    //! @return     テクスチャオブジェクトバッファへのポインタを返します。
    //!
    internal::TextureObject*
                            GetTextureObject()
    {
        return &m_TexObj;
    }

    //! @brief      テクスチャオブジェクトバッファへのポインタを取得します。
    //!
    //! @return     テクスチャオブジェクトバッファへのポインタを返します。
    //!
    const internal::TextureObject*
                            GetTextureObject() const
    {
        return &m_TexObj;
    }

    //! テクスチャ名を生成します。
    void                    GenTextureNames();

    //! テクスチャ名を破棄します。
    void                    DeleteTextureNames();

    //@}

private:
    void LoadTexture(const internal::TextureObject* pTexObj);

    /* ------------------------------------------------------------------------
            変数
       ------------------------------------------------------------------------ */

    void*                   m_pResource;    //!< リソースへのポインタ

    //! リソース中のFINFブロック本体へのポインタ
    FontInformation*        m_pFontInfo;

    internal::TextureObject m_TexObj;       //!< テクスチャオブジェクト
    u32                     m_WrapFilter;   //!< テクスチャラップ・フィルタ値
    const FontKerningTable* m_pKerningTable; //!< カーニングの情報を保持するテーブル

};

namespace internal {

//---------------------------------------------------------------------------
//! @brief        1シートあたりのセル数を計算します。
//!
//! @param[in]    tg  FontTextureGlyph構造体への参照。
//!
//! @return       1シートあたりのセル数を返します。
//---------------------------------------------------------------------------
inline
u32
GetCellsInASheet(const FontTextureGlyph& tg)
{
    return static_cast<u32>(tg.sheetRow * tg.sheetLine);
}


}   // namespace internal

}   // namespace font
}   // namespace nw

#endif //  NW_FONT_RESFONTBASE_H_
