﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_FONT_RESFONT_H_
#define NW_FONT_RESFONT_H_

#include <nw/types.h>
#include <nw/font/font_ResFontBase.h>

namespace nw {
namespace font {

//---------------------------------------------------------------------------
//! @brief        フォントリソースに対する操作を実装するクラスです。
//---------------------------------------------------------------------------
class ResFont : public ResFontBase
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(ResFontBase);

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */

    //! @name コンストラクタ / デストラクタ
    //@{

    //! コンストラクタです。
                            ResFont();

    //! デストラクタです。
    virtual                 ~ResFont();

    //@}

    //! @name リソースの関連付け / 解除
    //@{

    //! @brief          リソースを初期状態に戻します。
    //!
    //! ResFont::SetResource() によって書き換わったリソースのバイナリを
    //! 初期状態に戻します。
    //!
    //! リソースをメモリ上で移動する場合などは、事前にこの処理を行う必要があります。
    //!
    //! @param[in, out] resource  初期状態に戻すリソースです。
    //!
    static void RevertResource(void* resource);

    //! @brief          フォントリソースを関連付けてフォントを使用可能にします。
    //!
    //! @details
    //! フォントリソースは最初に ResFont に関連付けられたときに
    //! 内容が書きかえられ、メモリ上で移動できなくなります。
    //!
    //! 使用しなくなったフォントリソースをもとに戻すには
    //! RevertResource() を使用します。
    //!
    //! @param[in,out]  bfnt   ResFont に関連付ける bfnt へのポインタ。
    //!                        128byte アライメントされている必要があります。
    //!
    //! @return         リソースの関連付けが成功したなら true を、
    //!                 そうでなければ false を返します。@n
    //!                 リソース bfnt が不正であると失敗します。
    //!
    bool                    SetResource(void* bfnt);

    //! @brief      フォントリソースの割り当てを解除します。
    //!
    //! @details
    //! この関数を呼び出しても SetResource() で書き換えられた
    //! フォントリソースは元には戻りません。
    //!
    //! 使用しなくなったフォントリソースをもとに戻すには
    //! RevertResource() を使用します。
    //!
    //! @return     リソースが関連付けされている場合は、関連付けされているリソースへの
    //!             ポインタを返します。
    //!             関連付けされていない場合は NULL を返します。
    //!
    void*                   RemoveResource();

    using ResFontBase::IsManaging;

    //@}


private:
    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */

    //! @brief      メモリ上にロードされたbffntファイルを再構築して
    //!             使用できるようにします。
    //!
    //! @param[in,out]  fileHeader  メモリ上にロードしたbffntファイルの
    //!                             ファイルヘッダへのポインタ
    //!
    //! @return     再構築に成功した場合はFINFブロック本体へのポインタを返します。
    //!             再構築に失敗した場合はNULLを返します。
    //!
    static FontInformation* Rebuild(ut::BinaryFileHeader* fileHeader);

    //! @brief      ブロックを探します。
    //!
    //! @param[in]  fileHeader  メモリ上にロードしたbffntファイルの
    //!                             ファイルヘッダへのポインタ
    //! @param[in]  sigword     シグネチャ
    //!
    //!
    //! @return     見つかった場合はそのポインタを返します。見つからなかった場合は
    //!             NULLを返します。
    //!
    static void* FindBlock(ut::BinaryFileHeader* fileHeader, ut::SigWord sigword);
};

}   // namespace font
}   // namespace nw

#endif //  NW_FONT_RESFONT_H_
