﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_FONT_FONT_H_
#define NW_FONT_FONT_H_

#if defined(_MSC_VER) && _MSC_VER >= 1500
#pragma once
#endif

#include <nw/types.h>
#include <nw/font/font_CharStrmReader.h>
#include <nw/font/font_ResourceFormat.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>
#include <climits>

#if defined(NW_PLATFORM_CAFE)
    #include <cafe/gx2.h>
#endif

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。

// Windows APIにGetCharWidthというマクロがあり、メソッド名を入れ替えてしまうため、undefする。
// GetCharWidthは既に使われなくなったWindows APIなので、undefしても問題はない。
#ifdef GetCharWidth
#undef GetCharWidth
#endif
#endif


#define NW_FONT_MIN_ASSERT(exp, min)            NW_ASSERT((exp) >= (min))

#define NW_FONT_MINMAX_ASSERT(exp, min, max)    NW_ASSERT((exp) >= (min) && (exp) <= (max))


namespace nw {
namespace font {

class Font;

//---- テクスチャフォーマット型
typedef u16     TexFmt;

namespace internal {

#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4201)
#endif
union TextureSize
{
    struct
    {
        u16                 height;   //!< 高さ
        u16                 width;    //!< 幅
    };

    u32                 size;   //!< サイズ
};
#if defined(_WIN32)
#pragma warning(pop)
#endif

class TextureObject
{
public:
    TextureObject()
    {
        Reset();
#if defined(NW_PLATFORM_CAFE)
        m_Texture.surface.imagePtr = NULL;
#endif
    }
    ~TextureObject() { Reset(); }

    void Set(
        u32         name,
        const void* pImage,
        TexFmt      format,
        u16         width,
        u16         height,
        u8          sheetNum,
        bool        blackWhiteInterpolationEnabled);

    void SetTextureSettings(const TextureObject* pSrc);

    void Reset()
    {
        Set(0, NULL, FONT_SHEET_FORMAT_BC4, 0, 0, 0, true);
    }

    u32 GetName() const
    {
        return m_Name;
    }

    void SetName(u32 texName)
    {
        m_Name = texName;
    }

    uptr GetImage() const
    {
        return m_pImage;
    }

    u8 GetFormat() const
    {
        return m_Format;
    }

    u16 GetFormatAttribute() const
    {
        return static_cast<u16>(m_FormatAttribute << FONT_TEXTUREOBJ_FORMATATTRIBUTE_SHIFT_BIT);
    }

    u8 GetSheetNum() const
    {
        return m_SheetNum;
    }

    const TextureSize GetSize() const
    {
        return m_Size;
    }

    bool IsColorBlackWhiteInterpolationEnabled() const
    {
        return m_ColorBlackWhiteInterpolationEnabled;
    }

    void SetColorBlackWhiteInterpolationEnabled(bool flag)
    {
        m_ColorBlackWhiteInterpolationEnabled = flag;
    }

#if defined(NW_PLATFORM_CAFE)
    const GX2Texture* GetTexture() const { return &m_Texture; }
    GX2Texture* GetTexture() { return &m_Texture; }
#endif

private:

    static const int FONT_TEXTUREOBJ_FORMATATTRIBUTE_SHIFT_BIT = 8; //!< フォーマット属性 を FontSheetFormat に 復元する際のビットシフト量

    u32         m_Name;                 //!< テクスチャ名
    uptr        m_pImage;               //!< イメージへのポインタ
    TextureSize m_Size;                 //!< サイズ
    u8          m_Format;               //!< フォーマット
    u8          m_FormatAttribute;      //!< フォーマット属性
    u8          m_SheetNum;             //!< シート数
    bool        m_ColorBlackWhiteInterpolationEnabled; //!< 白黒補完が有効か否か

#if defined(NW_PLATFORM_CAFE)
    GX2Texture m_Texture;
#endif
};

}   // namespace internal


//---- 文字コード型
typedef u16     CharCode;

//---------------------------------------------------------------------------
//! @brief        文字グリフの情報を保持する構造体です。
//---------------------------------------------------------------------------
struct Glyph
{
    const void* pTexture;                   //!< シートへのポインタ
    CharWidths  widths;                     //!< 文字幅情報
    u8          height;                     //!< セルの高さ
    u16         texWidth;                   //!< シートの幅
    u16         texHeight;                  //!< シートの高さ
    u16         cellX;                      //!< シート内でのセル左上隅のx座標
    u16         cellY;                      //!< シート内でのセル左上隅のy座標
    TexFmt      texFormat;                  //!< シートのフォーマット
    u8          isSheetUpdated;             //!< シートの内容が更新されている場合は真
    u8          sheetIndex;                 //!< シート番号
    const internal::TextureObject*
                pTextureObject;             //!< シートのテクスチャオブジェクトへのポインタ

    //! コンストラクタです。
                Glyph();
};

//---------------------------------------------------------------------------
//! @brief        フォント管理を行うクラスの基底クラスです。
//---------------------------------------------------------------------------
class Font
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO_ROOT();

    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    enum Type
    {
        TYPE_NULL,          //
        TYPE_RESOURCE,      // ResFont
        TYPE_SCALABLE       // ScalableFont
    };


    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    // 不正な文字コード
    static const CharCode   INVALID_CHARACTER_CODE = INVALID_CHAR_CODE;


    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */

    //! @name コンストラクタ/デストラクタ
    //@{

    //! コンストラクタです。
                            Font();

    //! デストラクタです。
    virtual                 ~Font();

    //@}


    //! @name フォント情報の取得
    //@{

    //! @brief      フォントの幅を取得します。
    //!
    //! @return     ピクセル単位でのフォントの幅を返します。
    //!
    virtual int             GetWidth() const = 0;

    //! @brief      フォントの高さを取得します。
    //!
    //! @return     ピクセル単位でのフォントの高さを返します。
    //!
    virtual int             GetHeight() const = 0;

    //! @brief      フォントのアセントを取得します。
    //!
    //! @return     ピクセル単位でのフォントのアセントを返します。
    //!
    virtual int             GetAscent() const = 0;

    //! @brief      フォントのディセントを取得します。
    //!
    //! @return     ピクセル単位でのフォントのディセントを返します。
    //!
    virtual int             GetDescent() const = 0;

    //! @brief      フォントに含まれる最大の文字幅を取得します。
    //!
    //! @return     フォントに含まれる最大の文字幅をピクセル単位で返します。
    //!
    virtual int             GetMaxCharWidth() const = 0;

    //! @brief      フォントのタイプを取得します。
    //!
    //! @return     フォントのタイプを返します。
    //!
    virtual Type            GetType() const = 0;

    //! @brief      フォントのテクスチャフォーマットを取得します。
    //!
    //! @return     グリフデータのテクスチャフォーマットを返します。
    //!
    virtual TexFmt          GetTextureFormat() const = 0;

    //! @brief      フォントが持つ改行幅を取得します。
    //!
    //! @return     フォントが持つ改行幅を返します。
    //!
    virtual int             GetLineFeed() const = 0;

    //! @brief      フォントのデフォルト文字幅情報を取得します。
    //!
    //! @return     フォントのデフォルト文字幅情報を返します。
    //!
    virtual const CharWidths
                            GetDefaultCharWidths() const = 0;

    //@}


    //! @name フォント情報の設定
    //@{

    //! @brief      フォントがもつ改行幅を設定します。
    //!
    //! @param[in]  linefeed  新しい改行幅。
    //!
    virtual void            SetLineFeed(int linefeed) = 0;

    //! @brief      フォントのデフォルト文字幅情報を設定します。
    //!
    //! @param[in]  widths  新しいデフォルト文字幅情報。
    //!
    virtual void            SetDefaultCharWidths(
                                const CharWidths& widths
                            ) = 0;

    //! @brief      フォントの代替文字を変更します。
    //!
    //! @param[in]  c  新しい代替文字の文字コード。
    //!
    //! @return     代替文字の変更に成功したら true 、失敗したら false を返します。@n
    //!             フォント内に文字コード c に対応するグリフが存在しない場合に失敗します。
    //!
    virtual bool            SetAlternateChar(CharCode c) = 0;

    //@}


    //! @name 文字情報の取得
    //@{

    //! @brief      文字の文字幅を取得します。
    //!
    //! @param[in]  c  幅を求める文字の文字コード。
    //!
    //! @return     ピクセル単位での文字幅を返します。
    //!
    virtual int             GetCharWidth(CharCode c) const = 0;

    //! @brief      文字の文字幅情報を取得します。
    //!
    //! @param[in]  c  文字幅情報を取得する文字の文字コード。
    //!
    //! @return         文字幅情報を返します。
    //!
    virtual const CharWidths
                            GetCharWidths(CharCode c) const = 0;

    //! @brief      グリフデータを取得します。
    //!
    //! @param[out] pGlyph  グリフデータを受け取るバッファへのポインタ
    //! @param[in]  c       グリフデータを取得する文字の文字コード
    //!
    virtual void            GetGlyph(
                                Glyph*      pGlyph,
                                CharCode    c
                            ) const = 0;

    //! @brief      グリフデータの有無を取得します。
    //!
    //! @param[in]  c  グリフデータの有無を取得する文字の文字コード
    //!
    //! @return     グリフデータが存在する場合は true 、存在しない場合は false を返します。
    //!
    virtual bool            HasGlyph(
                                CharCode    c
                            ) const = 0;

    //! @brief      カーニングの値を取得します。
    //!
    //! @param[in]  c0  カーニングの値を取得する、進行方向と逆側の文字
    //! @param[in]  c1  カーニングの値を取得する、進行方向側の文字
    //!
    //! @return     カーニングの値を返します。
    //!
    virtual int            GetKerning(CharCode c0, CharCode c1) const = 0;

    //@}


    //! @name 文字列エンコーディング
    //@{

    //! @brief      フォントが対応している文字コードを取得します。
    //!
    //! @return     フォントが対応している文字コードを返します。
    //!
    virtual CharacterCode   GetCharacterCode() const = 0;

    //! @brief      フォントが対応している文字コード形式の文字列を解釈する
    //!             CharStrmReader を取得します。
    //!
    //! @param[in]  dummy CharStrmReaderを取得するCharWriterが扱う文字型。
    //!             引数の値そのものは使用しません。
    //!
    //! @return     フォントが対応している文字コード形式の
    //!             文字列を解釈する CharStrmReader を返します。
    //!
    const CharStrmReader    GetCharStrmReader(char dummy) const;

    //! @brief      フォントが対応している文字コード形式の文字列を解釈する
    //!             CharStrmReader を取得します。
    //!
    //! @param[in]  dummy CharStrmReaderを取得するCharWriterが扱う文字型。
    //!             引数の値そのものは使用しません。
    //!
    //! @return     フォントが対応している文字コード形式の
    //!             文字列を解釈する CharStrmReader を返します。
    //!
    const CharStrmReader    GetCharStrmReader(char16 dummy) const;

    //@}


    //! @name シート情報の取得
    //@{

    //! @brief      ベースライン位置を取得します。
    //!
    //! @return     セルの上端からベースラインまでのピクセル数を返します。
    //!
    virtual int             GetBaselinePos() const = 0;

    //! @brief      セルの高さを取得します。
    //!
    //! @return     ピクセル単位でのセルの高さを返します。
    //!
    virtual int             GetCellHeight() const = 0;

    //! @brief      セルの幅を取得します。
    //!
    //! @return     ピクセル単位でのセルの幅を返します。
    //!
    virtual int             GetCellWidth() const = 0;

    //@}

    //! @name テクスチャ補間
    //@{

    //! @brief      文字描画時にテクスチャの線形補間を行うかどうかを指定します。
    //!
    //! @param[in]  atSmall 文字がオリジナルサイズより小さく表示される場合に線形補間を
    //!                     行うのであれば true を指定します。
    //! @param[in]  atLarge 文字がオリジナルサイズより大きく表示される場合に線形補間を
    //!                     行うのであれば true を指定します。
    //!
    virtual void            EnableLinearFilter(
                                bool    atSmall,
                                bool    atLarge
                            ) = 0;

    //! @brief      文字が小さく表示される場合に線形補間が行われるかを取得します。
    //!
    //! @return     文字がオリジナルサイズより小さく表示される場合に線形補間が行われるのであれば
    //!             true を、行われないのであれば false を返します。
    //!
    virtual bool            IsLinearFilterEnableAtSmall() const = 0;

    //! @brief      文字が大きく表示される場合に線形補間が行われるかを取得します。
    //!
    //! @return     文字がオリジナルサイズより大きく表示される場合に線形補間が行われるのであれば
    //!             true を、行われないのであれば false を返します。
    //!
    virtual bool            IsLinearFilterEnableAtLarge() const = 0;

    //! @brief      テクスチャのラップ、フィルタ設定値を取得します。
    //!
    //! @return     テクスチャのラップ、フィルタ設定値を返します。
    //!
    virtual u32             GetTextureWrapFilterValue() const = 0;

    //@}

    //! @name 表示設定
    //@{

    //! @brief        カラーの白黒補間の影響の有無を取得します。
    //!
    //! @return       カラーの白黒補間の影響を受ける場合は、true を返します。
    //!
    virtual bool            IsColorBlackWhiteInterpolationEnabled() const = 0;

    //! @brief        カラーの白黒補間の影響の有無を設定します。
    //!
    //! @details
    //! この設定でfalseを設定しても、アルファについては影響を受けます。
    //!
    //! @param[in]  flag    true を設定すると、白黒補間の影響を受けます。
    //!
    virtual void            SetColorBlackWhiteInterpolationEnabled(bool flag) = 0;

    //! @brief        縁取り効果の有無を取得します。
    //!
    //! @return       縁取り効果が有効な場合は、true を返します。
    //!
    virtual bool            IsBorderEffectEnabled() const = 0;

    //@}

    //----------------------------------------------------
    //! @name カーニングの設定
    //@{

    //! @brief        カーニングを有効か否かを取得します。
    //!
    //! @details
    //! 有効な場合も、スケーラブルフォント自体がカーニングの情報を持っていない場合は
    //! カーニングが行われないことにご注意ください。
    //!
    //! @return 有効か否か
    //!
    bool                    IsEnableKerning() const
    {
        return m_IsEnableKerning;
    }

    //! @brief        カーニングを有効か否かを設定します。
    //!
    //! @details
    //! 初期状態では有効です。
    //!
    //! @param[in]    enable true を設定すると、カーニングを有効にします。
    //!
    void                    SetEnableKerning(bool enable)
    {
        m_IsEnableKerning = enable;
    }
    //@}

    //----------------------------------------------------
    //! @name 拡張マージンの設定
    //@{

    //! @brief        拡張マージンを有効か否かを取得します。
    //!
    //! @details
    //! 有効の場合、フォントの周りに0.5ピクセル分のマージンを追加して
    //! 文字欠けをおこりにくくします。
    //!
    //! @return 有効か否か
    //!
    bool                    IsEnableExtraMargin() const
    {
        return m_IsEnableExtraMargin;
    }

    //! @brief        拡張マージンを有効か否かを設定します。
    //!
    //! @details
    //! 初期状態では有効です。
    //!
    //! @param[in]    enable true を設定すると、拡張マージンを有効にします。
    //!
    void                    SetEnableExtraMargin(bool enable)
    {
        m_IsEnableExtraMargin = enable;
    }

private:
    bool m_IsEnableKerning;
    bool m_IsEnableExtraMargin;
};

}   // namespace font
}   // namespace nw

#endif //  NW_FONT_FONT_H_
