﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_FONT_DISPSTRINGBUFFER_H_
#define NW_FONT_DISPSTRINGBUFFER_H_

#if defined(_MSC_VER) && _MSC_VER >= 1500
#pragma once
#endif

#include <nw/font/font_Types.h>
#include <nw/font/font_Font.h>

namespace nw {
namespace font {

#if defined(_MSC_VER) && _MSC_VER >= 1500
#pragma warning(push)
#pragma warning(disable: 4200) // サイズ0の配列
#endif

namespace internal
{

enum
{
    USE_TEXTURE_NUM_MAX = 8, //!< 一回の描画で扱うことの出来るテクスチャの最大数です。
    VERTEX_NUMBER_BY_LETTER = 4, //!< 文字単位の頂点数です。
};

//----------------------------------------
//! @brief テクスチャ利用情報構造体
struct TextureUseInfo
{
    const internal::TextureObject* texObj;
    u32 startIndex;
    u32 useNum;
    u32 bitFlags;
};

//----------------------------------------
//! @brief 頂点構造体です。
struct Vertex
{
    enum
    {
        POSITION_OFFSET = 0,
        COLOR_OFFSET = (sizeof(f32) * 3),
        TEXCOORD_OFFSET = (sizeof(f32) * 3) + (sizeof(u8) * 4),
    };

    math::VEC3   xyz;
    ut::Color4u8 color;
    math::VEC3   uvw; // w には シート番号が入ります。
    void Set(f32 x, f32 y, f32 z, const ut::Color4u8& c, f32 u, f32 v, f32 w)
    {
        xyz.x = x;
        xyz.y = y;
        xyz.z = z;
        color = c;
        uvw.x = u;
        uvw.y = v;
        uvw.z = w;
    }
};

//----------------------------------------
struct CharAttribute
{
public:
    enum Flags
    {
        BORDER_EFFECT = (0x1 << 0)
    };

    math::VEC4 pos;
    ut::Color4u8 color[TEXTCOLOR_MAX];
    math::VEC4 tex;
private:
    uptr texObjAndFlags;        //!< 1bit目にflags、それ以外にテクスチャオブジェクトへのポインタ
public:
    s16 italicOffset;
    u8 sheetIndex;
    u8 shadowAlpha;

    inline void SetTexObjAndFlags(const internal::TextureObject* pTexObj, u8 flags)
    {
        texObjAndFlags = reinterpret_cast<uptr>(pTexObj) | static_cast<uptr>(flags);
    }

    inline const internal::TextureObject* GetTexObj() const
    {
        return reinterpret_cast<const internal::TextureObject*>(texObjAndFlags & static_cast<uptr>(~0x01));
    }

    inline u8 GetFlags() const
    {
        return static_cast<u8>(texObjAndFlags & static_cast<uptr>(0x01));
    }
};

//----------------------------------------
template<typename TBuffer, size_t SQuantity>
class ChainBuffer
{
    NW_DISALLOW_COPY_AND_ASSIGN(ChainBuffer);
    NW_STATIC_ASSERT(0 < SQuantity);

public:
    typedef TBuffer BufferType;
    typedef ChainBuffer<BufferType, SQuantity> SelfType;

    ChainBuffer()
    : m_CurrentBufferIndex(0)
    {
        std::fill_n(m_Buffers, SQuantity, static_cast<BufferType*>(NULL));
    }

    void SetBuffer(int index, BufferType* buffer)
    {
        NW_ASSERT_MINMAXLT(index, 0, SQuantity);
        m_Buffers[index] = buffer;
    }

    BufferType* GetBuffer() { return m_Buffers[m_CurrentBufferIndex]; }

    void PassToNext()
    {
        ++m_CurrentBufferIndex;
        if (SQuantity <= m_CurrentBufferIndex) { m_CurrentBufferIndex = 0; }
    }

    BufferType* GetBufferWithIndex(int index)
    {
        NW_ASSERT_MINMAXLT(index, 0, SQuantity);
        return m_Buffers[index];
    }

private:
    BufferType* m_Buffers[SQuantity];
    u32 m_CurrentBufferIndex;
};

}   // namespace internal

//---------------------------------------------------------------------------
//! @brief 文字列表示用バッファを管理するクラスです。
//---------------------------------------------------------------------------
class DispStringBuffer
{
public:
    enum
    {
        VERTEX_BUFFER_QUANTITY = 2 //!< マルチバッファリングの数です。
    };

    //! @brief コンストラクタです。
    DispStringBuffer(u32 charNum, void* graphicsBuffer, bool shadowEnabled);

    //! @brief デストラクタです。
    ~DispStringBuffer()
    {
    }

    //! @brief コマンドをクリアします。
    void ClearCommand()
    {
        m_IsGeneratedCommand = false;
    }

    //! @brief コマンドが生成済みかどうかを判定します。
    bool IsGeneratedCommand() const
    {
        return m_IsGeneratedCommand;
    }

    //! @brief コマンドが空かどうかを判定します。
    bool IsCommandEmpty()
    {
        return ! m_IsGeneratedCommand;
    }

    //! @brief 指定したサイズの頂点列を生成したことにします。
    void SetCommandGenerated()
    {
        m_IsGeneratedCommand = true;
    }

    //! @brief 格納可能な文字数を取得します。
    u32 GetCharCountMax() const
    {
        return m_CharCountMax;
    }

    //! @brief 格納した文字数を取得します。
    u32 GetCharCount() const
    {
        return m_CharCount;
    }

    //! @brief 格納した文字数を設定します。
    void SetCharCount(u32 count)
    {
        m_CharCount = count;
    }

    //! @brief 文字属性配列を取得します。
    internal::CharAttribute* GetCharAttributes()
    {
        return m_CharAttrs;
    }

    //! @brief 文字属性配列を設定します。
    void SetCharAttributes(internal::CharAttribute* attrs)
    {
        m_CharAttrs = attrs;
    }

    //! @brief 文字属性配列を共有するかどうかを取得します。
    bool IsUseSharedCharAttrs()
    {
        return m_IsUseSharedCharAttrs;
    }

    //! @brief 文字属性配列を共有するかどうかを設定します。
    void SetUseSharedCharAttrs(bool is_use)
    {
        m_IsUseSharedCharAttrs = is_use;
    }

    //! @brief テクスチャ使用情報を取得します。
    internal::TextureUseInfo* GetTextureUseInfos()
    {
        return m_TextureUseInfos;
    }

    //! @brief テクスチャ使用情報の位置を取得します。
    u32 GetTextureUseInfoPos() const
    {
        return m_TextureUseInfoPos;
    }

    //! @brief テクスチャ使用情報の位置を取得します。
    void SetTextureUseInfoPos(u32 pos)
    {
        m_TextureUseInfoPos = pos;
    }

    ///! @brief 頂点バッファを取得します。
    internal::ChainBuffer<internal::Vertex, VERTEX_BUFFER_QUANTITY>& GetVertexBuffers()
    {
        return m_VertexBuffers;
    }

    ///! @brief 初期化時に指定したgraphicsBufferを取得します。
    void* GetGraphicsBuffer()
    {
        return m_VertexBuffers.GetBufferWithIndex(0);
    }

private:
    const u32           m_CharCountMax;         //!< 格納可能な文字数
    u32                 m_CharCount;            //!< 格納した文字数
    u32                 m_TextureUseInfoPos;    //!< テクスチャ使用情報の位置
    internal::CharAttribute* m_CharAttrs;       //!< 頂点属性配列
    bool                m_IsUseSharedCharAttrs; //!< 頂点属性配列を共有しているかどうか
    bool                m_IsGeneratedCommand;   //!< コマンドが生成された。

    internal::ChainBuffer<internal::Vertex, VERTEX_BUFFER_QUANTITY> m_VertexBuffers; //!< 頂点バッファ

    internal::TextureUseInfo m_TextureUseInfos[internal::USE_TEXTURE_NUM_MAX];  //!< テクスチャ使用情報の配列

};

#if defined(_MSC_VER) && _MSC_VER >= 1500
#pragma warning(pop)
#endif

}   // namespace font
}   // namespace nw

#endif //  NW_FONT_DISPSTRINGBUFFER_H_
