﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_FONT_CHARSTRMREADER_H_
#define NW_FONT_CHARSTRMREADER_H_

#if defined(_MSC_VER) && _MSC_VER >= 1500
#pragma once
#endif

#include <nw/types.h>
#include <nw/assert.h>

namespace nw {
namespace font {


//---------------------------------------------------------------------------
//! @brief        文字ストリームを読み取るための方法を提供します。
//---------------------------------------------------------------------------
class CharStrmReader
{
    friend class Font;

public:
    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */

    //! @name コンストラクタ / デストラクタ
    //@{

    //! @brief      コンストラクタです。
    //!
    //! @param[in]  rhs     コピー元のオブジェクト。
    //!
                            CharStrmReader(const CharStrmReader& rhs)
    : mCharStrm(rhs.mCharStrm),
      mReadFunc(rhs.mReadFunc)
    {
    }

    //! デストラクタです。
                            ~CharStrmReader()
    {
    }

    //@}


    //! @name ストリーム読み取り
    //@{

    //! @brief      CharStrmReader が解釈する文字ストリームを設定します。
    //!
    //! @param[in]  stream  新しく設定するストリームの開始点。
    //!
    void                    Set(const char* stream)
    {
        NW_ASSERT_VALID_POINTER(stream);
        NW_ASSERT(  mReadFunc == &CharStrmReader::ReadNextCharUTF8
                   || mReadFunc == &CharStrmReader::ReadNextCharCP1252
                   || mReadFunc == &CharStrmReader::ReadNextCharSJIS   );
        mCharStrm = stream;
    }

    //! @brief      CharStrmReader が解釈する文字ストリームを設定します。
    //!
    //! @param[in]  stream  新しく設定するストリームの開始点。
    //!
    void                    Set(const char16* stream)
    {
        NW_ASSERT_ALIGN(stream, 2);
        NW_ASSERT_VALID_POINTER(stream);
        NW_ASSERT(mReadFunc == &CharStrmReader::ReadNextCharUTF16);
        mCharStrm = stream;
    }

    //! @brief      ストリームの現在位置へのポインタを取得します。
    //!
    //! @return     ストリーム中の次に読み取られる文字へのポインタを返します。
    //!
    const void*             GetCurrentPos() const   { return mCharStrm; }

    //! @brief      文字ストリームの次の文字の文字コードを取得すると共に、
    //!             現在位置を一文字分進めます。
    //!
    //! @return     次の文字の文字コードを返します。
    //!
    u16                     Next()                  { return (this->*mReadFunc)(); }

    //@}

private:
    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    typedef u16 (CharStrmReader::*ReadNextCharFunc)();


    /* ------------------------------------------------------------------------
            変数
       ------------------------------------------------------------------------ */
    const void*             mCharStrm;      // ストリーム現在位置へのポインタ
    const ReadNextCharFunc  mReadFunc;      // ストリーム解釈関数へのポインタ


    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- コンストラクタ
    explicit                CharStrmReader(ReadNextCharFunc func)
    : mCharStrm(NULL),
      mReadFunc(func)
    {
    }

    //---- ストリーム操作
    template <typename CharType>
    CharType                GetChar(int offset=0) const
    {
        const CharType* charStrm = reinterpret_cast<const CharType*>(mCharStrm);
        return *(charStrm + offset);
    }

    template <typename CharType>
    void                    StepStrm(int step=1)
    {
        const CharType*& charStrm = reinterpret_cast<const CharType*&>(mCharStrm);
        charStrm += step;
    }

    //! @brief      UTF8文字列ストリームの現在位置の文字の文字コードを読み取り、
    //!             文字列ストリームの現在位置を1文字分進めます。
    //!
    //! @return     ストリームの現在位置の文字の文字コード。
    //!
    u16                     ReadNextCharUTF8();

    //! @brief      UTF16文字列ストリームの現在位置の文字の文字コードを読み取り、
    //!             文字列ストリームの現在位置を1文字分進めます。
    //!
    //! @return     ストリームの現在位置の文字の文字コード。
    //!
    u16                     ReadNextCharUTF16();

    //! @brief      CP1252文字列ストリームの現在位置の文字の文字コードを読み取り、
    //!             文字列ストリームの現在位置を1文字分進めます。
    //!
    //! @return     ストリームの現在位置の文字の文字コード。
    //!
    u16                     ReadNextCharCP1252();

    //! @brief      SJIS文字列ストリームの現在位置の文字の文字コードを読み取り、
    //!             文字列ストリームの現在位置を1文字分進めます。
    //!
    //! @return     ストリームの現在位置の文字の文字コード。
    //!
    u16                     ReadNextCharSJIS();

    // コピー演算子を禁止します。
    const CharStrmReader&   operator=( const CharStrmReader& rhs);
};



}   // namespace font
}   // namespace nw

#endif  // NW_FONT_CHARSTRMREADER_H_
