﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Heap.h>
#include <nw/ut.h>
#include <nw/eft/eftcom_Message.h>
#if EFT_IS_WIN
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <windows.h>
#endif


namespace nw    {
namespace eftvw2  {

class CmdReceiver;
class CmdSender;

#define  MAX_CMDRECEIVE_BUFF    256 * 10    //!< 受信バッファの配列サイズ
#define  MAX_CBITEM_BUFF        32  * 10    //!< 受信時のコールバック登録用の配列サイズ
#define  CAFE_CMDID 101                     //!< TBD

//------------------------------------------------------------------------------
//! @brief コマンド
//------------------------------------------------------------------------------
class Cmd
{
    friend class CmdReceiver;
    friend class CmdSender;

public:
    //---------------------------------------------------------------------------
    //! @brief TBD
    //---------------------------------------------------------------------------
    enum CmdType
    {
        CMD_TYPE_STRING     = 0,
        CMD_TYPE_BUFFER     = 1,
        CMD_TYPE_NONE       = 2,
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタ
    //---------------------------------------------------------------------------
    Cmd()
    {
        Initialize();
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタ
    //---------------------------------------------------------------------------
    ~Cmd()
    {
        if( mCmdBuffer )
        {
            mHeap->Free( mCmdBuffer );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        コマンドを取得する
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetCommand()
    {
        return mCmdId;
    }

    //---------------------------------------------------------------------------
    //! @brief        コマンドタイプを取得する
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetCommandType()
    {
        return mCmdType;
    }

    //---------------------------------------------------------------------------
    //! @brief        コマンド文字列を取得する
    //! @return TBD
    //---------------------------------------------------------------------------
    const char* GetCommandString()
    {
        return (const char *)mCmdBuffer;
    }

    //---------------------------------------------------------------------------
    //! @brief        コマンドバッファを取得する
    //! @return TBD
    //---------------------------------------------------------------------------
    void* GetCommandBuffer()
    {
        return (void *)mCmdBuffer;
    }

    //---------------------------------------------------------------------------
    //! @brief        コマンドバッファサイズを取得する
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetCommandBufferSize()
    {
        return mCmdBufferSize;
    }

private:
    //---------------------------------------------------------------------------
    //! @brief        コマンドを設定する
    //! @param[in] heap     TBD
    //! @param[in] cmdId     TBD
    //! @param[in] string     TBD
    //---------------------------------------------------------------------------
    void SetCommand( nw::eft2::Heap* heap, u32 cmdId, const char* string )
    {
        mHeap = heap;
        mCmdBuffer = mHeap->Alloc( strlen(string) + 1 );
        EFT_ASSERT( mCmdBuffer );

        mCmdId      = cmdId;
        mCmdType    = CMD_TYPE_STRING;
#if EFT_IS_WIN
        strcpy_s( (char *)mCmdBuffer, strlen(string) + 1, string );
#else
        strcpy( (char *)mCmdBuffer, string );
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        コマンドを設定する
    //! @param[in] heap     TBD
    //! @param[in] cmdId     TBD
    //! @param[in] buffer     TBD
    //! @param[in] bufferSize     TBD
    //---------------------------------------------------------------------------
    void SetCommand( nw::eft2::Heap* heap, u32 cmdId, const void* buffer, u32 bufferSize )
    {
        mHeap = heap;
        mCmdBuffer = mHeap->Alloc( bufferSize );
        EFT_ASSERT( mCmdBuffer );

        mCmdId          = cmdId;
        mCmdType        = CMD_TYPE_BUFFER;
        mCmdBufferSize  = bufferSize;
        memcpy( mCmdBuffer, buffer, bufferSize );
    }

    //---------------------------------------------------------------------------
    //! @brief        コマンドを初期化する
    //---------------------------------------------------------------------------
    void Initialize()
    {
        mCmdId          = 0;
        mCmdType        = CMD_TYPE_NONE;
        mCmdBuffer      = NULL;
        mCmdBufferSize  = 0;
    }

private:
    u32                 mCmdId;             //!< TBD
    CmdType             mCmdType;           //!< TBD
    void*               mCmdBuffer;         //!< TBD
    u32                 mCmdBufferSize;     //!< TBD
    nw::eft2::Heap*     mHeap;              //!< TBD
};


//------------------------------------------------------------------------------
//! @brief 定義
//------------------------------------------------------------------------------
typedef void (*CBFuncReceive)(void* pthis, nw::eftcom::Message* pHeader, const void* pBody );

typedef nw::ut::FixedSizeArray<Cmd*, MAX_CMDRECEIVE_BUFF>       CmdFixedArray;


//------------------------------------------------------------------------------
//! @brief コマンド受信クラス
//------------------------------------------------------------------------------
class CmdReceiver
{
    friend class PacketProcedure;
    friend class ToolConnecter;

protected:
    //---------------------------------------------------------------------------
    //! @briefprivate TBD
    //---------------------------------------------------------------------------
    struct CBItem
    {
        //! @briefprivate TBD
        //! @param[in] _pthis     TBD
        //! @param[in] _type     TBD
        //! @param[in] _func     TBD
        void Initialize( void* _pthis, nw::eftcom::Message _type, CBFuncReceive _func )
        {
            this->pThis   = _pthis;
            this->msgType = _type;
            this->cbFunc  = _func;
        }

        nw::eftcom::Message msgType;    //!< TBD
        CBFuncReceive   cbFunc;         //!< TBD
        void*           pThis;          //!< TBD
    };

    typedef nw::ut::FixedSizeArray<CBItem*, MAX_CBITEM_BUFF>        CBItemFixedArray;   //!< TBD

public:
    //---------------------------------------------------------------------------
    //! @brief TBD
    //---------------------------------------------------------------------------
    enum CBType
    {
        CBType_Before,  //!< プロシージャ前にコールされる
        CBType_After,   //!< プロシージャ後にコールされる
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタ
    //! @param[in] heap     TBD
    //---------------------------------------------------------------------------
    explicit CmdReceiver( nw::eft2::Heap* heap );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタ
    //---------------------------------------------------------------------------
    ~CmdReceiver();

    //---------------------------------------------------------------------------
    //! @brief        エフェクトコマンドを取得する
    //! @return TBD
    //---------------------------------------------------------------------------
    // エフェクトコマンドを取得する
    Cmd* GetEffectCommand()
    {
        if ( mEffectCmdQueue.empty()
            || mEffectIterator == mEffectCmdQueue.End() ) return NULL;

        return *mEffectIterator;
    }

    //---------------------------------------------------------------------------
    //! @brief        エフェクトコマンドを追加する(文字列版)
    //! @param[in] cmdId     TBD
    //! @param[in] string     TBD
    //---------------------------------------------------------------------------
    void AddEffectCommand( u32 cmdId, const char* string )
    {
        mMutex.Lock();
        Cmd* cmd = static_cast<Cmd*>( mHeap->Alloc( sizeof(Cmd) ) );
        EFT_ASSERT( cmd );

        cmd->Initialize();
        cmd->SetCommand( mHeap, cmdId,  string );
        if (mProcesing == false){
            mEffectCmdQueue.PushBack( cmd );
        } else {
            // 処理中のときは一時バッファ追加します。
            mTempEffectCmdQueue.PushBack( cmd );
        }
        mMutex.Unlock();
    }

    //---------------------------------------------------------------------------
    //! @brief        エフェクトコマンドを追加する(バッファ版)
    //! @param[in] cmdId     TBD
    //! @param[in] buffer     TBD
    //! @param[in] bufferSize     TBD
    //---------------------------------------------------------------------------
    void AddEffectCommand( u32 cmdId, const void* buffer, u32 bufferSize )
    {
        mMutex.Lock();
        Cmd* cmd = static_cast<Cmd*>( mHeap->Alloc( sizeof(Cmd) ) );
        EFT_ASSERT( cmd );

        cmd->Initialize();
        cmd->SetCommand( mHeap, cmdId,  buffer, bufferSize );
        if (mProcesing == false){
            mEffectCmdQueue.PushBack( cmd );
        } else {
            // 処理中のときは一時バッファ追加します。
            mTempEffectCmdQueue.PushBack( cmd );
        }
        mMutex.Unlock();
    }

    //---------------------------------------------------------------------------
    //! @brief        エフェクトコマンドをpopする
    //---------------------------------------------------------------------------
    void PopEffectCommand()
    {
        mMutex.Lock();

        if ( mEffectCmdQueue.empty()
            || mEffectIterator == mEffectCmdQueue.End() ) return;

        Cmd* cmd = *mEffectIterator;
        cmd->~Cmd();

        mHeap->Free( cmd );

        ++mEffectIterator;
        if (mEffectIterator == mEffectCmdQueue.End()){
            mEffectCmdQueue.Clear();
        }
        mMutex.Unlock();
    }

    //---------------------------------------------------------------------------
    //! @brief        ロックする
    //---------------------------------------------------------------------------
    void Lock()
    {
        // ロック中であれば、解除を待つ
        while( mLock ) {}
        mLock   = true;
    }

    //---------------------------------------------------------------------------
    //! @brief        アンロックする
    //---------------------------------------------------------------------------
    void Unlock()
    {
        mLock   = false;
    }

    //---------------------------------------------------------------------------
    //! @brief        処理開始（処理中にリストに追加されるかどうかを判定します）
    //---------------------------------------------------------------------------
    void BeginProc()
    {
        mMutex.Lock();
        mProcesing      = true;
        mEffectIterator = mEffectCmdQueue.Begin();
    }

    //---------------------------------------------------------------------------
    //! @brief        処理終了
    //---------------------------------------------------------------------------
    void EndProc()
    {
        mProcesing = false;
        mEffectIterator = mEffectCmdQueue.End();

        // 一時的なキュー → 本番キューへ移行います。
        if (mTempEffectCmdQueue.empty() == false)
        {
            CmdFixedArray::iterator ite = mTempEffectCmdQueue.Begin();
            CmdFixedArray::iterator end = mTempEffectCmdQueue.End();
            for ( ; ite != end ; ++ite ){
                mEffectCmdQueue.PushBack( *ite );
            }
            mTempEffectCmdQueue.Clear();
        }
        mMutex.Unlock();
    }

    //---------------------------------------------------------------------------
    //! @brief        コマンド受信用プロシジャー
    //! @param[in] pData     TBD
    //! @param[in] size     TBD
    //---------------------------------------------------------------------------
    static void ReceiveProc( const void* pData, u32 size );

//   //---------------------------------------------------------------------------
//   //! @brief        コマンド受信用プロシジャー
//   //---------------------------------------------------------------------------
//  void AddCallback( CBType cbType, void* pThis, nw::eftcom::Message msgType, CBFuncReceive func );
//
//
//   //---------------------------------------------------------------------------
//   //! @brief        タイプに該当するコールバック項目を取得します。（プロシージャ前）
//   //---------------------------------------------------------------------------
//  void DoCallbackBeforeProc( const nw::eftcom::Message* pHeader, const void* pBody );
//
//
//   //---------------------------------------------------------------------------
//   //! @brief        タイプに該当するコールバック項目を取得します。（プロシージャ後）
//   //---------------------------------------------------------------------------
//  void DoCallbackAfterProc( const nw::eftcom::Message* pHeader, const void* pBody );

private:
    CmdFixedArray           mEffectCmdQueue;        //!< エフェクトコマンドキュー
    CmdFixedArray           mTempEffectCmdQueue;    //!< 一時的なエフェクトコマンドキュー
//  CBItemFixedArray        mCallbackListBefore;    //!< プロシジャー前コールバック
//  CBItemFixedArray        mCallbackListAfter;     //!< プロシジャー後コールバック
    CmdFixedArray::iterator mEffectIterator;        //!< コマンドイテレター
    nw::eft2::Heap*         mHeap;                  //!< TODO:コマンド専用のアロケータを別枠で作成して使用するように修正。
    volatile bool           mLock;                  //!< コマンド受信中かどうか
    volatile bool           mProcesing;             //!< コマンド処理中かどうか
    nw::ut::Mutex           mMutex;                 //!< TBD

public:
    static CmdReceiver*     gCmdReceiver;           //!< TBD
};


//------------------------------------------------------------------------------
//! @brief コマンド受信クラスを生成する
//! @param[in] heap     TBD
//! @return TBD
//------------------------------------------------------------------------------
CmdReceiver* CreateCmdReceiver( nw::eft2::Heap* heap );


//------------------------------------------------------------------------------
//! @brief コマンド受信クラスを破棄する
//! @param[in] heap     TBD
//------------------------------------------------------------------------------
void DestroyCmdReceiver( nw::eft2::Heap* heap );


} // namespace eftvw2
} // namespace nw
