﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/config.h>

namespace nw   {
namespace eft2 {

//---------------------------------------------------
//!     @brief テンポラリバッファ
//!
//!     エフェクトランタイム内部で利用されるGPUが参照する一時バッファを管理します。
//---------------------------------------------------
struct TemporaryBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief        バッファサイド
    //---------------------------------------------------------------------------
    enum BufferSide
    {
        EFT_TEMPORARY_BUFFER_FIRST      = 0,        //!< テンポラリバッファ：1番目
        EFT_TEMPORARY_BUFFER_SECOND     = 1,        //!< テンポラリバッファ：2番目
        EFT_TEMPORARY_BUFFER_THIRD      = 2,        //!< テンポラリバッファ：3番目
        EFT_TEMPORARY_BUFFER_MAX        = 3,        //!< TBD
    };

    //---------------------------------------------------------------------------
    //! @brief        バッファアライン
    //---------------------------------------------------------------------------
    enum
    {
       EFT_DOUBLE_BUFFER_ALIGN      = 0x100         //!< TBD
    };


    //---------------------------------------------------------------------------
    //! @brief        初期化処理を行います。
    //!
    //! @param[in]    size         初期化するテンポラリバッファサイズ。
    //! @param[in]    useTriple    トリプルバッファを利用するかどうか。
    //!
    //! @return       なし。
    //---------------------------------------------------------------------------
    void Initialize( u32 size, bool useTriple = false );

    //---------------------------------------------------------------------------
    //! @brief        無効化処理を行います。
    //---------------------------------------------------------------------------
    void Invalidate();

    //---------------------------------------------------------------------------
    //! @brief        終了処理を行います。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        テンポラリバッファを切り替えします。
    //---------------------------------------------------------------------------
    void Swap();

    //---------------------------------------------------------------------------
    //! @brief          テンポラリバッファからメモリを確保する。
    //                  確保できない場合はNULLが返ります。
    //! @param[in] size 確保するバッファサイズ
    //! @return         先頭アドレス
    //---------------------------------------------------------------------------
    void* Alloc( u32 size );

    //---------------------------------------------------------------------------
    //! @brief  テンポラリバッファのサイズを取得します。
    //! @return テンポラリバッファのサイズ
    //---------------------------------------------------------------------------
    u32 GetSize() const { return mBufferSize; }

    //---------------------------------------------------------------------------
    //! @brief  呼び出し時のテンポラリバッファの消費サイズを取得します。
    //! @return テンポラリバッファの消費サイズ
    //---------------------------------------------------------------------------
    u32 GetUsedSize() const { return mAllocedSize[mBufferSide]; }

    //---------------------------------------------------------------------------
    //! @brief              呼び出し時のテンポラリバッファの消費サイズを取得します。
    //!
    //! @param[in]  side    取得するバッファサイド。
    //! @return             テンポラリバッファの消費サイズ
    //---------------------------------------------------------------------------
    u32 GetUsedSize( BufferSide side ) const { return mAllocedSize[side]; }

    //---------------------------------------------------------------------------
    //! @brief        テンポラリバッファの最大消費サイズを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetMaxUsedSize() const { return mMaxAllocedSize; }

    //---------------------------------------------------------------------------
    //! @brief        カレントバッファのCPUメモリキャッシュをフラッシュします。
    //---------------------------------------------------------------------------
    void FlushCache();

private:
    bool                            mInitialized;                               //!< 初期化済みかどうか
    u32                             mBufferSide;                                //!< 現在使用しているダブルバッファ
    u32                             mBufferSize;                                //!< ダブルバッファメモリ確保サイズ
    u32                             mMaxAllocedSize;                            //!< 最大メモリ確保サイズ
    u32                             mPreAllocedSize[EFT_TEMPORARY_BUFFER_MAX];  //!< 前フレームでのダブルバッファ消費サイズ
    u32                             mAllocedSize[EFT_TEMPORARY_BUFFER_MAX];     //!< ダブルバッファメモリ消費サイズ
    u32                             mFlushedSize[EFT_TEMPORARY_BUFFER_MAX];     //!< ダブルバッファメモリフラッシュしたサイズ
    void*                           mBuffer[EFT_TEMPORARY_BUFFER_MAX];          //!< ダブルバッファ確保先ポインタ
};


} // namespace eft2
} // namespace nw
