﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Enum.h>
#include <nw/eft/eft2_Config.h>
#include <nw/eft/eft2_Binary.h>
#include <nw/eft/eft2_Resource.h>
#include <nw/eft/eft2_EmitterSet.h>
#include <nw/eft/eft2_Emitter.h>
#include <nw/eft/eft2_Particle.h>
#include <nw/eft/eft2_Handle.h>
#include <nw/eft/eft2_TemporaryBuffer.h>
#include <nw/eft/eft2_EmitterCalc.h>
#include <nw/eft/eft2_View.h>
#include <nw/eft/eft2_Shader.h>
#include <nw/eft/eft2_Texture.h>
#include <nw/eft/eft2_Callback.h>

namespace nw   {
namespace eft2 {

//---------------------------------------------------
//! @briefprivate  テクスチャ初期化関数定義
//---------------------------------------------------
typedef bool (*TextureResourceInitializer)( TextureResource* texture, u64 globalId, ResTexture* resTexture, void* binary, u32 binarySize, ResourceHeap* resHeap );

//---------------------------------------------------
//! @briefprivate  テクスチャ破棄関数定義
//---------------------------------------------------
typedef bool (*TextureResourceFinalizer)( TextureResource* texture );

//---------------------------------------------------
//! @brief  エフェクト全体を管理するクラスです。
//!
//!         ユーザーはこのクラスを介してエミッタセットの生成/破棄、
//!         リソースの登録/破棄、エフェクトの計算/描画処理をおこないます。
//---------------------------------------------------
class System
{
public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief        エフェクトシステムの生成を行います。
    //!               config::Heap,config::DynamicHeapが指定されない場合アサートします。
    //!               システム初期化時に config::mHeap からメモリ確保が行われ、
    //!               それ以降は、config::mDynamicHeap からメモリ確保が行われます。
    //!
    //! @param[in]    config エフェクトシステムを生成するためのコンフィグ。
    //!
    //! @return       生成されたエフェクトシステムのインスタンス。
    //---------------------------------------------------------------------------
    explicit System( const Config &config );

    //---------------------------------------------------------------------------
    //! @brief        エフェクトシステムの破棄を行います。
    //!
    //!               ClearResourceが呼ばれていない未削除のエフェクトバイナリの
    //!               削除処理も同時に行いますが、その際、削除に用いるheapは、
    //!               EntryResource時に利用されたheapが利用されます。
    //---------------------------------------------------------------------------
    virtual ~System();

    //@}


    //----------------------------------------
    //! @name リソース管理
    //@{

    //---------------------------------------------------------------------------
    //! @brief      リソースを登録する。登録後もリソースはメモリ上に保持して下さい。
    //!
    //!             Windows版ではシェーダの遅延コンパイルが利用できます。
    //!             本メソッド内で発生するシェーダコンパイルの実行時間が問題となる場合は、
    //!             delayCompile=true とすることで、シェーダコンパイル実行を各シェーダ利用時に移行できます。
    //!
    //! @param[in]  heap                メモリ確保先のヒープ。
    //! @param[in]  resource            メモリ上のエフェクトバイナリ。128Byteアラインに配置する必要があります。
    //! @param[in]  resId               登録するリソースID。指定できるIdの上限数はConfigで指定します。
    //! @param[in]  delayCompile        シェーダ遅延コンパイル( Windows版のみ動作 )。
    //! @param[in]  residentResource    共通で利用されるテクスチャを含んだリソース。
    //!
    //! @return     リソース登録に成功すれば true, できなれば false。
    //---------------------------------------------------------------------------
    bool EntryResource( Heap* heap, void* resource, u32 resId, bool delayCompile = false, Resource* residentResource = NULL  );

    //---------------------------------------------------------------------------
    //! @brief        指定IDのリソースをエフェクトシステムから破棄します。
    //!
    //! @param[in]    heap     NULLの場合は、EntryResourceで渡されたHeapからメモリ解放処理を行います。
    //! @param[in]    resId    破棄するリソースID。
    //!
    //! @return       なし。
    //---------------------------------------------------------------------------
    void ClearResource( Heap* heap, u32 resId );

    //---------------------------------------------------------------------------
    //! @brief        指定IDのリソースを取得します。
    //!
    //! @param[in]    resId    リソースID。
    //!
    //! @return       エフェクトリソース。
    //---------------------------------------------------------------------------
    Resource* GetResource( u32 resId )
    {
        if ( m_ResourceArray[resId] && m_ResourceArray[resId]->IsAlive() )
        {
            return m_ResourceArray[resId];
        }

        return NULL;
    }

    //---------------------------------------------------------------------------
    //! @brief        指定IDのリソースを取得します。( const 版 )
    //!
    //! @param[in]    resId    リソースID。
    //!
    //! @return       エフェクトリソース。
    //---------------------------------------------------------------------------
    const Resource* GetResource( u32 resId ) const
    {
        if ( m_ResourceArray[resId] && m_ResourceArray[resId]->IsAlive() )
        {
            return m_ResourceArray[resId];
        }

        return NULL;
    }


    //---------------------------------------------------------------------------
    //! @brief        指定IDのリソースをバイナリサイズ詳細をログ出力します。
    //!
    //! @param[in]    resId    リソースID。
    //---------------------------------------------------------------------------
    void OutputResourceInfo( u32 resId )
    {
        if ( m_ResourceArray[resId] )
        {
            m_ResourceArray[resId]->OutputResourceInfo();
        }
    }
#if EFT_GX2
    //---------------------------------------------------------------------------
    //! @brief  シェーダバイナリをアンリロケート。
    //!
    //! @param[in] pData      シェーダバイナリ
    //---------------------------------------------------------------------------
    void Unrelocate( void* pData );
#endif
    //@}



    //----------------------------------------
    //! @name エミッタセット生成・破棄
    //@{

private:
    //---------------------------------------------------------------------------
    //! @brief        指定IDのエミッタセットを生成します。
    //!
    //! @param[in] handle           エフェクトハンドル。生成されたエミッタセットと紐付けられます。
    //! @param[in] emitterSetID     生成するエミッタセットのID
    //! @param[in] resourceID       生成するエミッタセットを含むリソースID
    //! @param[in] groupID          生成されたエミッタセットが所属するグループID
    //! @param[in] particleMax      マニュアル放出を利用する場合の最大パーティクル数
    //! @param[in] heap             ヒープ
    //! @param[in] delay            BegeinFrame の時点でエミッタセットリストに追加する場合はtrue
    //! @return       １つでもエミッタが生成出来れば true, できなれば false。
    //---------------------------------------------------------------------------
    bool CreateEmitterSetID( Handle* handle, s32 emitterSetID, u32 resourceID, u8 groupID, u32 particleMax, Heap* heap = NULL, bool delay =false );
    bool CreateEmitterSetId( Handle* handle, s32 emitterSetID, u32 resourceID, u8 groupID, u32 particleMax, Heap* heap = NULL, bool delay =false )
    {
        return CreateEmitterSetID( handle, emitterSetID, resourceID, groupID, particleMax, heap, delay );
    }

public:
    //---------------------------------------------------------------------------
    //! @brief        指定IDのエミッタセットを生成します。
    //!
    //! @param[in]  handle          エフェクトハンドル。
    //! @param[in]  emitterSetID    生成するエミッタセットのID。設定するエミッタセットIDは、SearchEmitterSetIDで検索します。
    //! @param[in]  resourceID      生成するエミッタセットを含むリソースID。
    //! @param[in]  groupID         グループID(0～63,63はゲーム内ビューアと共用)。
    //! @param[in]  heap            ヒープ
    //! @param[in]  delay           エミッタセットリストへの追加をBegeinFrame のタイミングで行う。
    //!
    //! @return       １つでもエミッタが生成出来れば true, できなれば false。
    //---------------------------------------------------------------------------
    bool CreateEmitterSetID( Handle* handle, s32 emitterSetID, u32 resourceID, u8 groupID, Heap* heap = NULL, bool delay =false );
    bool CreateEmitterSetId( Handle* handle, s32 emitterSetID, u32 resourceID, u8 groupID, Heap* heap = NULL, bool delay =false )
    {
        return CreateEmitterSetID( handle, emitterSetID, resourceID, groupID, heap, delay );
    }

    //---------------------------------------------------------------------------
    //! @brief        指定IDのエミッタセットを放出します。
    //!
    //! @param[in]    handle            エフェクトハンドル。
    //! @param[in]    pos               エミッタを配置するワールド座標。
    //! @param[in]    emitterSetID      エミッタセットID。設定するエミッタセットIDは、SearchEmitterSetIDで検索します。
    //! @param[in]    resourceID        リソースID。
    //! @param[in]    groupID           グループID(0～63,63はゲーム内ビューアと共用)。
    //! @param[in]    emitterMask       エミッタを出すかどうかのビットフラグ。
    //! @param[in]    delay             エミッタセットリストへの追加をBegeinFrame のタイミングで行う。
    //!
    //! @return       １つでもエミッタが生成出来れば true, できなれば false。
    //---------------------------------------------------------------------------
    bool CreateEmitterSetID( Handle* handle, const nw::math::VEC3& pos, s32 emitterSetID,
        u32 resourceID = 0, u8 groupID = 0, u32 emitterMask = 0xffffffff, bool delay = false )
    {
        EFT_UNUSED_VARIABLE(emitterMask);

        bool ret = CreateEmitterSetID( handle, emitterSetID, resourceID, groupID, NULL, delay );
        if ( ret && handle->IsValid() )
        {
            nw::math::MTX34 matrix;
            matrix.SetIdentity();
            matrix._03 = pos.x;
            matrix._13 = pos.y;
            matrix._23 = pos.z;
            handle->GetEmitterSet()->SetMatrix( matrix );
            return true;
        }

        return false;
    }
    bool CreateEmitterSetId( Handle* handle, const nw::math::VEC3& pos, s32 emitterSetID,
        u32 resourceID = 0, u8 groupID = 0, u32 emitterMask = 0xffffffff, bool delay = false )
    {
        return CreateEmitterSetID( handle, pos, emitterSetID, resourceID, groupID, emitterMask, delay );
    }

    //---------------------------------------------------------------------------
    //! @brief        指定IDのエミッタセットを放出します。引数のmtxにスケールが入っていた場合、表示が壊れます。
    //!
    //! @param[in]  handle          エフェクトハンドル。
    //! @param[in]  mtx             ワールド行列。
    //! @param[in]  emitterSetID    エミッタセットID。設定するエミッタセットIDは、SearchEmitterSetIDで検索します。
    //! @param[in]  resourceID      リソースID。
    //! @param[in]  groupID         グループID(0～63,63はゲーム内ビューアと共用)。
    //! @param[in]  emitterMask     エミッタを出すかどうかのビットフラグ。
    //! @param[in]  delay           エミッタセットリストへの追加をBegeinFrame のタイミングで行う。
    //!
    //! @return       １つでもエミッタが生成出来れば true, できなれば false。
    //---------------------------------------------------------------------------
    bool CreateEmitterSetID( Handle* handle , const nw::math::MTX34 &mtx , s32 emitterSetID ,
        u32 resourceID = 0 , u8 groupID = 0 , u32 emitterMask = 0xffffffff, bool delay = false )
    {
        EFT_UNUSED_VARIABLE(emitterMask);

        bool ret = CreateEmitterSetID( handle, emitterSetID, resourceID, groupID, NULL, delay );
        if ( ret && handle->IsValid() )
        {
            handle->GetEmitterSet()->SetMatrix( mtx );
            return true;
        }

        return false;
    }
    bool CreateEmitterSetId( Handle* handle , const nw::math::MTX34 &mtx , s32 emitterSetID ,
        u32 resourceID = 0 , u8 groupID = 0 , u32 emitterMask = 0xffffffff, bool delay = false )
    {
        return CreateEmitterSetID( handle, mtx, emitterSetID, resourceID, groupID, emitterMask, delay );
    }

    //---------------------------------------------------------------------------
    //! @brief        指定IDのエミッタセットをマニュアル放出状態で生成します。
    //!
    //!               アプリ側から放出タイミングを指定するモードでエミッタセットを生成します。
    //!
    //! @param[in]  handle              エフェクトハンドル。
    //! @param[in]  emitterSetID        エミッタセットID。設定するエミッタセットIDは、SearchEmitterSetIDで検索します。
    //! @param[in]  resourceID          リソースID。
    //! @param[in]  groupID             グループID(0～63,63はゲーム内ビューアと共用)。
    //! @param[in]  particleMax         同時放出済みパーティクル最大数。
    //! @param[in]  heap                ヒープ
    //! @param[in]  residentEmitterTime 0以上を指定した場合、この期間以上経過した後にパーティクル数が0になるとエミッタが終了します。
    //!
    //! @return       １つでもエミッタが生成出来れば true, できなれば false。
    //---------------------------------------------------------------------------
    bool CreateManualEmitEmitterSetID( Handle* handle, s32 emitterSetID, u32 resourceID, u8 groupID, u32 particleMax, Heap* heap = NULL, int residentEmitterTime = -1 );
    bool CreateManualEmitEmitterSetId( Handle* handle, s32 emitterSetID, u32 resourceID, u8 groupID, u32 particleMax, Heap* heap = NULL, int residentEmitterTime = -1 )
    {
        return CreateManualEmitEmitterSetID( handle, emitterSetID, resourceID, groupID, particleMax, heap, residentEmitterTime );
    }

    //---------------------------------------------------------------------------
    //! @brief        引数のエミッタセットを削除します。
    //!
    //! @param[in]    emitterSet   エミッタセットインスタンス。
    //! @param[in]    immediate    即時削除するか、false の場合は、beginFrameのタイミングで削除されます。
    //---------------------------------------------------------------------------
    void KillEmitterSet( EmitterSet* emitterSet, bool immediate = true );

    //---------------------------------------------------------------------------
    //! @brief        引数名のエミッタセットを削除します。
    //!
    //! @param[in]    emitterSetName   エミッタセット名。
    //! @param[in]    resId            リソースID。
    //---------------------------------------------------------------------------
    void KillEmitterSet( const char* emitterSetName, u32 resId );

    //---------------------------------------------------------------------------
    //! @briefprivate 指定エミッタセットを再生成します。
    //!
    //! @param[in] resourceID     TBD
    //! @param[in] emitterSetID     TBD
    //! @return       １つでもエミッタが生成出来れば true, できなれば false。
    //---------------------------------------------------------------------------
    bool ReCreateEmitterSet( u32 resourceID, s32 emitterSetID );

    //---------------------------------------------------------------------------
    //! @briefprivate 指定エミッタセットを再生成します。
    //!
    //! @param[in] emitterSetName     TBD
    //! @param[in] oldResId     TBD
    //! @param[in] newResId     TBD
    //---------------------------------------------------------------------------
    void RecreateEmitterSet2( const char* emitterSetName, u32 oldResId, u32 newResId );

    //---------------------------------------------------------------------------
    //! @brief        再生中の全てのエミッタを削除します。
    //---------------------------------------------------------------------------
    void KillAllEmitter( bool immediate = true )
    {
        KillAllEmitterSet( immediate );
    }

    //---------------------------------------------------------------------------
    //! @brief        再生中の全てのエミッタセットを削除します。
    //---------------------------------------------------------------------------
    void KillAllEmitterSet( bool immediate = true );

    //---------------------------------------------------------------------------
    //! @brief        指定グループに所属するエミッタを削除します。
    //!
    //! @param[in]    groupID   グループID。
    //---------------------------------------------------------------------------
    void KillEmitterGroup( u8 groupID, bool immediate = true )
    {
        return KillEmitterSetGroup( groupID, immediate );
    }

    //---------------------------------------------------------------------------
    //! @brief        指定グループに所属するエミッタを削除します。
    //!
    //! @param[in]    groupID   グループID。
    //---------------------------------------------------------------------------
    void KillEmitterSetGroup( u8 groupID, bool immediate = true );

    //@}



    //----------------------------------------
    //! @name 計算処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief        内部で確保されたメモリの遅延解放やカウンタ等のリセットを行います。
    //!
    //!               計算処理・描画処理にかかわらず、毎フレーム呼び出してください。
    //!               エミッタセットの遅延リスト追加、遅延リスト削除はこのタイミングで行われます。
    //---------------------------------------------------------------------------
    void BeginFrame();

    //---------------------------------------------------------------------------
    //! @brief        計算処理を行います。
    //! @param[in]  groupID     計算処理を行うグループID
    //! @param[in]  frameRate   フレームレート( 標準は 1.0f )
    //! @param[in]  swapUbo     ユニフォームブロックバッファをスワップするか( 標準は true )
    //---------------------------------------------------------------------------
    void Calc( u8 groupID, f32 frameRate, bool swapUbo );

    //@}


    //----------------------------------------
    //! @name 描画処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief        フレームバッファテクスチャを設定します。屈折パーティクルで利用されます。
    //!
    //! @param[in]    frameBufferTexture    フレームバッファテクスチャ。
    //---------------------------------------------------------------------------
    void SetFrameBufferTexture( TextureInterface frameBufferTexture )
    {
        CpuCore core = GetCurrentCore();
        m_FrameBufferTexture[core].Initialize( frameBufferTexture, EFT_TEXTURE_TYPE_2D );
    }

    //---------------------------------------------------------------------------
    //! @brief        デプスバッファテクスチャを設定します。ソフトパーティクル処理などで利用されます。
    //!
    //! @param[in]    depthBufferTexture    デプスバッファテクスチャ。
    //---------------------------------------------------------------------------
    void SetDepthTexture( TextureInterface depthBufferTexture )
    {
        return SetDepthBufferTexture( depthBufferTexture );
    }

    //---------------------------------------------------------------------------
    //! @brief        デプスバッファテクスチャを設定します。ソフトパーティクル処理などで利用されます。
    //!
    //! @param[in]    depthBufferTexture    デプスバッファテクスチャ。
    //---------------------------------------------------------------------------
    void SetDepthBufferTexture( TextureInterface depthBufferTexture )
    {
        CpuCore core = GetCurrentCore();
        m_DepthBufferTexture[core].Initialize( depthBufferTexture, EFT_TEXTURE_TYPE_2D );
    }

    //---------------------------------------------------------------------------
    //! @brief        各エミッタ描画で共通のカスタムシェーダユニフォームブロックを設定します。
    //!
    //                ここで設定されたUBOは、SetViewParamタイミングで設定(コマンド発行)されます。
    //                SetViewParamが呼び出されるまでは、渡したユニフォームブロックバッファをユーザー側で保持するようにしてください。
    //! @param[in]    id                ID
    //! @param[in]    uniformBlock      ユニフォームブロックへのポインタ
    //! @param[in]    uniformBlockSize  ユニフォームブロックのサイズ
    //---------------------------------------------------------------------------
    void SetCommonCustomShaderUniformBlock( CustomShaderUboID id, void* uniformBlock, u32 uniformBlockSize )
    {
        m_CommonCstmShaderUniformBlock[id]      = uniformBlock;
        m_CommonCstmShaderUniformBlockSize[id]  = uniformBlockSize;
    }

    //---------------------------------------------------------------------------
    //! @brief        描画用テンポラリバッファのスワップを行います。
    //!
    //!               計算処理内でダブルバッファに描画データを書き込むので、
    //!               Calc / CalcEmitterより前に呼び出してください。
    //---------------------------------------------------------------------------
    void SwapBuffer();

    //---------------------------------------------------------------------------
    //! @brief      ビュービューパラメータの設定を行います。
    //!
    //!             ユニフォームブロックモード時は、このメソッド内でビューユニフォームブロックを設定(コマンド発行)します。
    //! @param[in] viewParam     ビューパラメータ
    //---------------------------------------------------------------------------
    void SetViewParam( ViewParam* viewParam );

    //---------------------------------------------------------------------------
    //! @brief      ビュービューパラメータの取得を行います。
    //!
    //! @return     ビューパラメータ
    //---------------------------------------------------------------------------
    ViewParam* GetViewParam()
    {
        return &m_ViewParam[GetCurrentCore()];
    }

    //---------------------------------------------------------------------------
    //! @briefprivate   ビュー情報の取得を行います。
    //!
    //! @return         ビューパラメータ
    //---------------------------------------------------------------------------
    ViewParam* GetCoppiedViewParam()
    {
        return m_ViewParamCopy[GetCurrentCore()];
    }

    //---------------------------------------------------------------------------
    //! @brief        エミッタ描画プロファイラを設定します。
    //!
    //! @param[in] profiler プロファイラコールバック関数ポインタ
    //---------------------------------------------------------------------------
    void SetEmitterDrawProfilerCallback( DrawEmitterProfilerCallback profiler )
    {
        m_EmitterDrawProfiler = profiler;
    }

    //---------------------------------------------------------------------------
    //! @brief        描画処理を行います。
    //!
    //! @param[in]    groupID       描画を行うグループID。
    //! @param[in]    drawPathFlag  描画パスフラグ。
    //! @param[in]    sort          emittetSetのZソートを行うかどうか。
    //! @param[in]    calcStreamOut StreamOutを計算させるかどうか。
    //! @param[in]    userParam     ユーザーパラメータへのポインタ。
    //---------------------------------------------------------------------------
    void Draw( u8 groupID, u32 drawPathFlag = 0xFFFFFFFF, bool sort = false, bool calcStreamOut = true, void* userParam = NULL );

    //---------------------------------------------------------------------------
    //! @brief        指定グループIDのエミッタセットをソートバッファに追加します。
    //!
    //!               drawPathFlag で指定するフラグは、論理和で複数指定できます。
    //!               AddSortBuffer はソートバッファに積むだけの処理しか行いません。
    //!               実際の描画は、DrawSortBuffer をで行われます。
    //!
    //! @param[in]    groupID       描画を行うグループID。
    //! @param[in]    drawPathFlag  描画パスフラグ。eft::DrawPathFlag で指定してください。
    //---------------------------------------------------------------------------
    void AddSortBuffer( u8 groupID, u32 drawPathFlag = 0xFFFFFFFF );

    //---------------------------------------------------------------------------
    //! @brief        ソートバッファの描画処理を行います。
    //!
    //!               AddSortBuffer で追加されたグループをZソートして描画します。
    //!
    //! @param[in]    calcStreamOut StreamOutを計算させるかどうか。
    //! @param[in]    userParam     カスタムシェーダへ渡すアドレス。
    //---------------------------------------------------------------------------
    void DrawSortBuffer( bool calcStreamOut = true, void* userParam = NULL );

    //@}


    //----------------------------------------
    //! @name テンポラリバッファ
    //@{

    //---------------------------------------------------------------------------
    //! @brief          テンポラリバッファからメモリを確保します。
    //!
    //! @param[in] size 確保するサイズ。
    //! @return 確保したバッファの先頭アドレス
    //---------------------------------------------------------------------------
    void* AllocFromTempBuffer( u32 size );

    //---------------------------------------------------------------------------
    //! @brief          テンポラリバッファのCPUキャッシュフラッシュを行います。
    //---------------------------------------------------------------------------
    void FlushTempBuffer();

    //@}


    //----------------------------------------
    //! @name エミッタセット ID/名前 の検索
    //@{

    //---------------------------------------------------------------------------
    //! @brief        エミッタセットIDを検索します。
    //!
    //! @param[in]    emitterSetName    検索するエミッタセット名。
    //! @param[in]    resId   検索対象となるリソースID。
    //!
    //! @return       検索したエミッタセットID。見つけられなった場合は　EFT_INVALID_EMITTER_SET_ID が返ります。
    //---------------------------------------------------------------------------
    s32 SearchEmitterSetID( const char* emitterSetName, s32 resId = 0 ) const;
    s32 SearchEmitterSetId( const char* emitterSetName, s32 resId = 0 ) const
    {
        return SearchEmitterSetID( emitterSetName, resId );
    }

    //---------------------------------------------------------------------------
    //! @brief        エミッタセット名を検索します。
    //!
    //! @param[in]    emitterSetId 検索するエミッタセットID。
    //! @param[in]    resId   検索対象となるリソースID。
    //!
    //! @return       検索したエミッタセット名。見つけられなかった場合は、NULL が返ります。
    //---------------------------------------------------------------------------
    const char* SearchEmitterSetName( s32 emitterSetId, s32 resId = 0 ) /*const*/;

    //@}


    //----------------------------------------
    //! @name ストリームアウト処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief  ストリームアウトエミッタを一時リストに追加します。
    //!
    //! @param[in] streaOutEmitter     追加するエミッタ
    //---------------------------------------------------------------------------
    void AddStreamOutEmitterList( Emitter* streaOutEmitter );

    //---------------------------------------------------------------------------
    //! @brief  ストリームアウトエミッタリストを処理します。
    //!
    //!         Config::SetEnableStreamOutBatchProcess でストリームアウト処理のバッチ処理モードが有効の場合、
    //!         本メソッドを一度だけ呼び出してストリームアウト処理を行う必要があります。
    //!
    //! @param[in] userParam          ユーザーパラメータ
    //! @param[in] processEmitterFlag 処理するストリームアウトエミッタのフラグ
    //---------------------------------------------------------------------------
    void BatchCalculationStreamOutEmitter( void* userParam, u32 processEmitterFlag = 0xffffffff );

    //---------------------------------------------------------------------------
    //! @brief ストリームアウトエミッタを一括処理するか？
    //!
    //! @return trueで行う、falseで行わない
    //---------------------------------------------------------------------------
    bool IsBatchProcessStreamOutEmitter() const
    {
        return m_IsBatchProcessStreamOutEmitter;
    }

    //---------------------------------------------------------------------------
    //! @brief      ストリームアウト処理終了時にZクリップを有効/無効どちらで指定するか設定します。
    //!
    //! @param[in]  zclip_enable     Zクリップフラグ
    //---------------------------------------------------------------------------
    void SetZCllipFlagByStreamOut( bool zclip_enable ) { m_ZClipEnable = zclip_enable; }

    //---------------------------------------------------------------------------
    //! @brief      ストリームアウト処理終了時にZクリップを有効/無効どちらで指定するかフラグを取得します。
    //!
    //! @return     Zクリップフラグ
    //---------------------------------------------------------------------------
    bool GetZCllipFlagByStreamOut() const { return m_ZClipEnable; }

    //---------------------------------------------------------------------------
    //! @brief      ランタイム内部でカラーバッファへの書き込みを制御します。
    //!
    //! @param[in]  colorBufferEnable     カラーバッファフラグへの書き込み有効無効フラグ
    //---------------------------------------------------------------------------
    void SetColorBufferEnbaleFlag( bool colorBufferEnable )
    {
        CpuCore core = GetCurrentCore();
        m_ColorBufferEnable[ core ] = colorBufferEnable;
    }

    //---------------------------------------------------------------------------
    //! @brief      ランタイム内部でカラーバッファの有効/無効をどちらで指定するかフラグを取得します。
    //!
    //! @return     カラーバッファフラグ書き込み有効無効フラグ
    //---------------------------------------------------------------------------
    bool GetColorBufferEnbaleFlag() const
    {
        CpuCore core = GetCurrentCore();
        return m_ColorBufferEnable[ core ];
    }

    //@}


    //----------------------------------------
    //! @name GPUキャッシュフラッシュ
    //@{

    //---------------------------------------------------------------------------
    //! @brief        GPUキャッシュのフラッシュを行います。
    //!
    //!               必要であればGPU実行前に呼び出してください。
    //!               内部では以下の項目のGPUキャッシュフラッシュを行います。
    //!               GX2_INVALIDATE_ATTRIB_BUFFER / GX2_INVALIDATE_TEXTURE / GX2_INVALIDATE_UNIFORM_BLOCK / GX2_INVALIDATE_SHADER
    //---------------------------------------------------------------------------
    void FlushGpuCache();

    //@}


    //----------------------------------------
    //! @name 描画ビュー
    //@{

    //---------------------------------------------------------------------------
    //! @brief      描画ビューフラグを設定する。
    //!
    //! @param[in]  flag     描画ビューフラグ
    //---------------------------------------------------------------------------
    void SetDrawViewFlag( DrawViewFlag flag ) { m_DrawViewFlag[GetCurrentCore()] = flag; }

    //---------------------------------------------------------------------------
    //! @brief      描画ビューフラグを取得する。
    //!
    //! @return     描画ビューフラグ
    //---------------------------------------------------------------------------
    DrawViewFlag GetDrawViewFlag() const { return m_DrawViewFlag[GetCurrentCore()]; }

    //@}


    //----------------------------------------
    //! @name エミッタセット初期化・破棄時 コールバック処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief      エミッタセット初期化コールバック関数を設定します。
    //! @param[in]  callback エミッタセット初期化コールバック関数
    //---------------------------------------------------------------------------
    void SetEmitterSetInitializeCallback( EmitterSetInitializeCallback callback )
    {
        m_EmitterSetInitializeCallback = callback;
    }

    //---------------------------------------------------------------------------
    //! @brief      エミッタセット破棄時コールバック関数を設定します。
    //! @param[in]  callback エミッタセット破棄時コールバック関数
    //---------------------------------------------------------------------------
    void SetEmitterSetFinalizeCallback( EmitterSetFinalizeCallback callback )
    {
        m_EmitterSetFinalizeCallback = callback;
    }

    //---------------------------------------------------------------------------
    //! @brief      エミッタセット初期化コールバック関数を取得します。
    //! @return     エミッタセット初期化コールバック関数
    //---------------------------------------------------------------------------
    EmitterSetInitializeCallback GetEmitterSetInitializeCallback()
    {
        return m_EmitterSetInitializeCallback;
    }

    //---------------------------------------------------------------------------
    //! @brief      エミッタセット破棄時コールバック関数を取得します。
    //! @return     エミッタセット破棄時コールバック関数
    //---------------------------------------------------------------------------
    EmitterSetFinalizeCallback GetEmitterSetFinalizeCallback()
    {
        return m_EmitterSetFinalizeCallback;
    }

    //@}


    //----------------------------------------
    //! @name 描画パス コールバック処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief        描画パス 描画設定コールバックを設定します。
    //! @param[in] id       コールバックID
    //! @param[in] flag     描画パスフラグ
    //! @param[in] callback 描画パスコールバック関数へのポインタ
    //---------------------------------------------------------------------------
    void SetDrawPathRenderStateSetCallback( DrawPathCallBackID             id,
                                            DrawPathFlag                   flag,
                                            DrawPathRenderStateSetCallback callback );

    //---------------------------------------------------------------------------
    //! @brief        描画パス 描画設定コールバックを設定します。
    //! @param[in] flag     描画パスフラグ
    //! @param[in] callback 描画パスコールバック関数へのポインタ
    //---------------------------------------------------------------------------
    void SetDrawPathRenderStateSetCallback( DrawPathFlag                   flag,
                                            DrawPathRenderStateSetCallback callback );

    //---------------------------------------------------------------------------
    //! @brief        描画パス　描画設定コールバックを取得します。
    //! @param[in] flag     描画パスフラグ
    //! @return TBD
    //---------------------------------------------------------------------------
    DrawPathRenderStateSetCallback GetDrawPathRenderStateSetCallback( DrawPathFlag flag );

    //@}


    //----------------------------------------
    //! @name LOD / カリング コールバック処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief      エミッタ計算処理カリングコールバックを設定します。
    //!
    //! @param[in]  callback    エミッタカリングコールバック関数へのポインタ
    //---------------------------------------------------------------------------
    void SetEmitterCalcLodCallback( EmitterCalcLodCallback callback )
    {
        m_EmitterCalcLodCallback = callback;
    }

    //---------------------------------------------------------------------------
    //! @brief      エミッタ描画処理カリングコールバックを設定します。
    //!
    //! @param[in]  callback     エミッタカリングコールバック関数へのポインタ
    //---------------------------------------------------------------------------
    void SetEmitterDrawCullingCallback( EmitterDrawCullingCallback callback )
    {
        m_EmitterDrawCullingCallback = callback;
    }

    //---------------------------------------------------------------------------
    //! @brief      カスタムフィールドコールバックを設定します。
    //!
    //! @param[in]  callback    カスタムフィールドコールバック関数へのポインタ
    //---------------------------------------------------------------------------
    void SetCustomFieldCallback( CustomFieldCallback callback )
    {
        m_CustomFieldCallback = callback;
    }

    //---------------------------------------------------------------------------
    //! @brief      カスタムフィールドコールバックを取得します。
    //!
    //! @return     カスタムフィールドコールバック関数へのポインタ
    //---------------------------------------------------------------------------
    CustomFieldCallback GetCustomFieldCallback() const
    {
        return m_CustomFieldCallback;
    }

    //@}

    //----------------------------------------
    //! @name カスタムアクション / カスタムシェーダ
    //@{

    //---------------------------------------------------------------------------
    //! @brief      コールバックセットを設定します。
    //!
    //! @param[in] id   セットするコールバックID
    //! @param[in] set  コールバックセット
    //---------------------------------------------------------------------------
    void SetCallback( CallBackID id, CallbackSet set )
    {
        m_IsEnableCallback[id]    = true;
        m_Callback[id]            = set;
        m_Callback[id].endianFlip = _EndianFlipCallback;
    }

    //---------------------------------------------------------------------------
    //! @brief      指定IDのコールバックセットが有効か取得します。
    //!
    //! @param[in]  id  コールバックID
    //! @return     trueで有効、falseで無効
    //---------------------------------------------------------------------------
    bool IsEnabelCallbackSet( CallBackID id )
    {
        return m_IsEnableCallback[id];
    }

    //---------------------------------------------------------------------------
    //! @brief      コールバックセットを取得します。
    //!
    //! @param[in]  id  コールバックID
    //! @return     コールバックセット
    //---------------------------------------------------------------------------
    CallbackSet* GetCallbackSet( CallBackID id )
    {
        return &m_Callback[id];
    }

    //---------------------------------------------------------------------------
    //! @briefprivate      カスタムシェーダテクスチャの状態をリセットします。
    //!
    //! @param[in] shader   シェーダ
    //---------------------------------------------------------------------------
    void ResetCustomShaderSetting( Shader* shader )
    {
        CpuCore core = GetCurrentCore();
        m_CurrentCustomShader[core]            = shader;
        m_CurrentCustomShaderTextureSlot[core] = EFT_CUSTOM_TEXTURE_SLOT_0;
    }

    //---------------------------------------------------------------------------
    //! @brief      カスタムシェーダテクスチャをバインドします。
    //!
    //! @param[in]  csType     カスタムシェーダテクスチャタイプ
    //! @param[in]  texture    テクスチャ
    //! @return true で設定成功、falseで設定失敗
    //---------------------------------------------------------------------------
    bool BindCustomShaderTexture( CustomShaderTextureType csType, TextureInterface texture )
    {
        CpuCore core = GetCurrentCore();
        if (!m_CurrentCustomShader[core] ) return false;

        u32 fragLoc = m_CurrentCustomShader[core]->GetCustomTextureFragmentSamplerLocation( csType );
        u32 vertLoc = m_CurrentCustomShader[core]->GetCustomTextureVertexSamplerLocation( csType );

        // CustomShaderTextureType から TextureType 引く
        TextureType texType = static_cast<TextureType>( csType / EFT_CUSTOM_SHADER_TEXTURE_ID_MAX );

        if ( BindTexture( texture, texType, fragLoc, vertLoc, m_CurrentCustomShaderTextureSlot[core] ) )
        {
            m_CurrentCustomShaderTextureSlot[core] = static_cast<CustomShaderTextureSlot>( m_CurrentCustomShaderTextureSlot[core] + 1 );
            return true;
        }

        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief      カスタムシェーダテクスチャをバインドします。
    //!
    //! @param[in]  csType      カスタムシェーダテクスチャタイプ
    //! @param[in]  texture     テクスチャ
    //! @param[in]  sampler     サンプラ
    //! @return true で設定成功、falseで設定失敗
    //---------------------------------------------------------------------------
    bool BindCustomShaderTexture( CustomShaderTextureType csType, TextureInterface texture, TextureSamplerInterface sampler )
    {
        CpuCore core = GetCurrentCore();
        if ( !m_CurrentCustomShader[core] ) return false;

        u32 fragLoc = m_CurrentCustomShader[core]->GetCustomTextureFragmentSamplerLocation( csType );
        u32 vertLoc = m_CurrentCustomShader[core]->GetCustomTextureVertexSamplerLocation( csType );

        // CustomShaderTextureType から TextureType 引く
        TextureType texType = static_cast<TextureType>( csType / EFT_CUSTOM_SHADER_TEXTURE_ID_MAX );

        if ( BindTexture( texture, texType, fragLoc, vertLoc, m_CurrentCustomShaderTextureSlot[core] ) )
        {
            bool ret = BindSampler( sampler, m_CurrentCustomShaderTextureSlot[core], vertLoc, fragLoc );
            m_CurrentCustomShaderTextureSlot[core] = static_cast<CustomShaderTextureSlot>( m_CurrentCustomShaderTextureSlot[core] + 1 );
            return ret;
        }

        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief      シェーダタイプを切り替えます。
    //!
    //! @param[in]  type     シェーダタイプ
    //---------------------------------------------------------------------------
    void SetShaderType( ShaderType type )
    {
        m_CurrentShaderType[GetCurrentCore()] = type;
    }

    //---------------------------------------------------------------------------
    //! @brief      シェーダタイプを切り替えます。
    //!
    //! @param[in]  type     シェーダタイプ
    //! @param[in]  core     コアID
    //---------------------------------------------------------------------------
    void SetShaderType( ShaderType type, CpuCore core )
    {
        m_CurrentShaderType[core] = type;
    }

    //---------------------------------------------------------------------------
    //! @briefprivate   エミッタプラグインコールバックセットを設定します。
    //!
    //! @param[in] id   エミッタプラグインID
    //! @param[in] set  コールバックセット
    //---------------------------------------------------------------------------
    void SetEmitterPluginCallbackSet( EmitterPluginID id, CallbackSet set )
    {
        m_EmitterPluginCallback[id] = set;
    }

    //---------------------------------------------------------------------------
    //! @briefprivate   エミッタプラグインコールバックセットを取得します。
    //!
    //! @param[in] id   エミッタプラグインID
    //! @return         コールバックセット
    //---------------------------------------------------------------------------
    CallbackSet* GetEmitterPluginCallbackSet( EmitterPluginID id )
    {
        return &m_EmitterPluginCallback[id];
    }

    //@}


    //----------------------------------------
    //! @name テクスチャ 初期化/破棄
    //@{

    //---------------------------------------------------------------------------
    //! @briefprivate  テクスチャ 初期化/破棄 関数を登録
    //!
    //! @param[in] initializer      TBD
    //! @param[in] finalizer        TBD
    //---------------------------------------------------------------------------
    void SetupTextureTranslation( TextureResourceInitializer initializer, TextureResourceFinalizer finalizer )
    {
        m_TextureInitializer    = initializer;
        m_TextureFinalizer      = finalizer;
    }

    //---------------------------------------------------------------------------
    //! @briefprivate  テクスチャ 初期化関数を取得します。
    //!
    //! @return テクスチャ初期化関数へのポインタ
    //---------------------------------------------------------------------------
    const TextureResourceInitializer GetTextureInitializer() const { return m_TextureInitializer; }

    //---------------------------------------------------------------------------
    //! @briefprivate  テクスチャ 破棄関数を取得します。
    //!
    //! @return テクスチャ破棄関数へのポインタ
    //---------------------------------------------------------------------------
    const TextureResourceFinalizer GetTextureFinalizer() const { return m_TextureFinalizer; }

    //@}


    //----------------------------------------
    //! @name 情報取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief      最大リソース数の取得します。
    //!
    //! @return     最大リソース数
    //---------------------------------------------------------------------------
    s32 GetNumResource()    const { return GetResourceMaxNum(); }

    //---------------------------------------------------------------------------
    //! @brief      最大リソース数の取得します。
    //!
    //! @return     最大リソース数
    //---------------------------------------------------------------------------
    u32 GetResourceNum()    const { return GetResourceMaxNum(); }

    //---------------------------------------------------------------------------
    //! @brief      最大リソース数の取得します。
    //!
    //! @return     最大リソース数
    //---------------------------------------------------------------------------
    u32 GetResourceMaxNum() const { return m_ResourceNum; }

    //---------------------------------------------------------------------------
    //! @brief      処理可能な最大エミッタセット数を取得します。
    //!
    //! @return     最大エミッタセット数
    //---------------------------------------------------------------------------
    u32 GetEmitterSetMaxNum() const { return m_EmitterSetNum; }

    //---------------------------------------------------------------------------
    //! @brief      処理可能な最大エミッタ数を取得します。
    //!
    //! @return     最大エミッタ数
    //---------------------------------------------------------------------------
    u32 GetEmitterMaxNum() const{ return m_EmitterNum; }

    //---------------------------------------------------------------------------
    //! @brief      空きエミッタ数を取得します。
    //!
    //! @return     空きエミッタ数
    //---------------------------------------------------------------------------
    s32 GetNumFreeEmitter() const { return GetFreeEmitterNum(); }

    //---------------------------------------------------------------------------
    //! @brief      空きエミッタ数を取得します。
    //!
    //! @return     空きエミッタ数
    //---------------------------------------------------------------------------
    s32 GetFreeEmitterNum() const { return ( GetEmitterMaxNum() - GetProcessingEmitterCount() ); }

    //---------------------------------------------------------------------------
    //! @brief      空きエミッタ数を取得します。
    //!
    //! @return     空きエミッタ数
    //---------------------------------------------------------------------------
    s32 GetFreeEmitterActualNum() const { return m_FreeEmitterNum; }

    //---------------------------------------------------------------------------
    //! @brief  処理中のエミッタセット数を取得します。
    //!
    //! @return 処理中のエミッタセット数
    //---------------------------------------------------------------------------
    s32 GetNumEmitterSetCalc() const{ return GetProcessingEmitterSetCount(); }

    //---------------------------------------------------------------------------
    //! @brief  処理中のエミッタセット数を取得します。
    //!
    //! @return 処理中のエミッタセット数
    //---------------------------------------------------------------------------
    u32 GetProcessingEmitterSetCount() const { return m_ProcessingInfo.emitterSetNum; }

    //---------------------------------------------------------------------------
    //! @brief  処理中のエミッタ数を取得します。
    //!
    //! @return 処理中のエミッタ数
    //---------------------------------------------------------------------------
    s32 GetNumEmitterCalc() const{ return GetProcessingEmitterCount(); }

    //---------------------------------------------------------------------------
    //! @brief  処理中のエミッタ数を取得します。
    //!
    //! @return 処理中のエミッタ数
    //---------------------------------------------------------------------------
    u32 GetProcessingEmitterCount() const { return m_ProcessingInfo.emitterNum; }

    //---------------------------------------------------------------------------
    //! @brief  計算処理をスキップされたのエミッタ数を取得します。
    //!
    //! @return 計算処理をスキップされたのエミッタ数
    //---------------------------------------------------------------------------
    u32 GetCalcSkipEmitterCount() const { return m_ProcessingInfo.calcSkipEmitterNum; }

    //---------------------------------------------------------------------------
    //! @brief  処理中のCPUパーティクル数を取得します。
    //!
    //! @return 処理中のCPUパーティクル数
    //---------------------------------------------------------------------------
    s32 GetNumPtclCalc() const{ return GetProcessingCpuParticleCount(); }

    //---------------------------------------------------------------------------
    //! @brief  処理中のCPUパーティクル数を取得します。
    //!
    //! @return 処理中のCPUパーティクル数
    //---------------------------------------------------------------------------
    u32 GetProcessingCpuParticleCount() const { return m_ProcessingInfo.cpuParticleNum; }

    //---------------------------------------------------------------------------
    //! @brief  処理中のGPUパーティクル数を取得します。
    //!
    //! @return 処理中のGPUパーティクル数
    //---------------------------------------------------------------------------
    s32 GetNumGpuPtclCalc() const{ return GetProcessingGpuParticleCount(); }

    //---------------------------------------------------------------------------
    //! @brief 処理中のGPUパーティクル数を取得します。
    //!
    //! @return 処理中のGPUパーティクル数
    //---------------------------------------------------------------------------
    u32 GetProcessingGpuParticleCount() const { return m_ProcessingInfo.gpuParticleNum; }

    //---------------------------------------------------------------------------
    //! @brief  処理中のStreamOutパーティクル数を取得します。
    //!
    //! @return 処理中のStreamOutパーティクル数
    //---------------------------------------------------------------------------
    u32 GetProcessingGpuSoParticleCount() const { return m_ProcessingInfo.gpusoParticleNum; }

    //---------------------------------------------------------------------------
    //! @brief  処理中のエミッタ時間アニメ数を取得します。
    //!
    //! @return 処理中のエミッタ時間アニメ数
    //---------------------------------------------------------------------------
    u32 GetProcessingEmitterAnimNum() const { return m_ProcessingInfo.emitterAnimNum; }

    //---------------------------------------------------------------------------
    //! @brief  処理中のエミッタ時間アニメ数を取得します。
    //!
    //! @return 処理中のエミッタ時間アニメ数
    //---------------------------------------------------------------------------
    u32 GetProcessingEmitterCount( EmitterCalcType mode ) const
    {
        u32 emitterNum = 0;
        if ( mode == EFT_EMITTER_CALC_TYPE_CPU )
        {
            emitterNum = m_ProcessingInfo.cpuEmitterNum;
        }
        if ( mode == EFT_EMITTER_CALC_TYPE_GPU )
        {
            emitterNum = m_ProcessingInfo.gpuEmitterNum;
        }
        if ( mode == EFT_EMITTER_CALC_TYPE_GPU_SO )
        {
            emitterNum = m_ProcessingInfo.gpusoEmitterNum;
        }
        return emitterNum;
    }

    //---------------------------------------------------------------------------
    //! @brief  処理中のストライプ本数を取得します。
    //!
    //! @return 処理中のストライプ本数
    //---------------------------------------------------------------------------
    int GetProcessingStripeCount() const
    {
        return m_ProcessingInfo.stripeNum;
    }

    //---------------------------------------------------------------------------
    //! @brief  処理中のスーパーストライプ本数を取得します。
    //!
    //! @return 処理中のスーパーストライプ本数
    //---------------------------------------------------------------------------
    int GetProcessingSuperStripeCount() const
    {
        return m_ProcessingInfo.superStripeNum;
    }


    //---------------------------------------------------------------------------
    //! @brief  エミッタセットヘッドを取得します。( const 無し版 )
    //!
    //! @param[in]  groupID  グループID
    //! @return     エミッタセット
    //---------------------------------------------------------------------------
    EmitterSet* GetEmitterSetHead( u8 groupID ) { return m_EmitterSetHead[groupID]; }

    //---------------------------------------------------------------------------
    //! @brief        エミッタセットヘッドを取得します。
    //!
    //! @param[in]  groupID  グループID
    //! @return     エミッタセット
    //---------------------------------------------------------------------------
    const EmitterSet* GetEmitterSetHead( u8 groupID ) const { return m_EmitterSetHead[groupID]; }

    //---------------------------------------------------------------------------
    //! @brief  動的ヒープから確保したメモリサイズを取得します。
    //!
    //! @return 確保したヒープサイズ
    //---------------------------------------------------------------------------
    u32 GetAllocedFromDynamicHeapSize() const { return m_ProcessingInfo.allocedDynamicHeapSize; }

    //---------------------------------------------------------------------------
    //! @brief  動いているコアの取得します。
    //!
    //! @return コアID
    //---------------------------------------------------------------------------
    static CpuCore GetCurrentCore()
    {
#if EFT_IS_CAFE
        return (CpuCore)OSGetCoreId();
#else
        return EFT_CPU_CORE_1;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief 指定グループ、指定描画パスに描画すべきエミッタがあるか。
    //!
    //! @param[in] groupID     グループID
    //! @param[in] drawPath    描画パスフラグ
    //! @return 指定されたエミッタがあればtrue
    //---------------------------------------------------------------------------
    bool IsHasRenderingEmitter( u8 groupID, u32 drawPath ) const
    {
        for ( u32 core = 0; core < EFT_CPU_CORE_MAX; core ++ )
        {
            if ( m_EnableDrawPath[core][groupID] & drawPath )
            {
                return true;
            }
        }
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief 指定グループ、指定描画パスでフレームバッファテクスチャが要求されるか。
    //!
    //! @param[in] groupID     グループID
    //! @param[in] drawPath    描画パスフラグ
    //! @return 指定されたエミッタがあればtrue
    //---------------------------------------------------------------------------
    bool IsRequestFrameBufferTexture( u8 groupID, u32 drawPath ) const
    {
        for ( u32 core = 0; core < EFT_CPU_CORE_MAX; core ++ )
        {
            if ( m_RequestFrameBufferTexturePath[core][groupID] & drawPath )
            {
                return true;
            }
        }
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief 指定グループ、指定描画パスでデプスバッファテクスチャが要求されるか。
    //!
    //! @param[in] groupID     グループID
    //! @param[in] drawPath    描画パスフラグ
    //! @return 指定されたエミッタがあればtrue
    //---------------------------------------------------------------------------
    bool IsRequestDepthBufferTexture( u8 groupID, u32 drawPath ) const
    {
        for ( u32 core = 0; core < EFT_CPU_CORE_MAX; core ++ )
        {
            if ( m_RequestDepthBufferTexturePath[core][groupID] & drawPath )
            {
                return true;
            }
        }
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief ユニフォームブロックのCPUキャッシュフラッシュを自動的に行うか。
    //!
    //! @return trueで行う、falseで行わない
    //---------------------------------------------------------------------------
    inline bool IsUboCpuChacheFlush() const
    {
        return m_IsUboCpuChacheFlush;
    }

    //---------------------------------------------------------------------------
    //! @brief ユニフォームブロックのGPUキャッシュフラッシュを自動的に行うか。
    //!
    //! @return trueで行う、falseで行わない
    //---------------------------------------------------------------------------
    inline bool IsUboGpuChacheFlush() const
    {
        return m_IsUboGpuChacheFlush;
    }

    //---------------------------------------------------------------------------
    //! @brief        指定グループが有効なエミッタを保持するかチェックします。
    //!
    //! @param[in]    groupID    グループID。
    //!
    //! @return       有効であればtrue、無効であればfalseを返します。
    //---------------------------------------------------------------------------
    bool HasEnableEmitter( u8 groupID )
    {
        if ( m_EnableDrawPath[EFT_CPU_CORE_0][groupID] == 0 &&
             m_EnableDrawPath[EFT_CPU_CORE_1][groupID] == 0 &&
             m_EnableDrawPath[EFT_CPU_CORE_2][groupID] == 0 )
        {
            return false;
        }
        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief  フレームバッファテクスチャを取得します。
    //!
    //! @return フレームバッファテクスチャ
    //---------------------------------------------------------------------------
    TextureBase& GetFrameBufferTexture()
    {
        CpuCore core = GetCurrentCore();
        return m_FrameBufferTexture[core];
    }

    //---------------------------------------------------------------------------
    //! @brief  デプスバッファテクスチャを取得します。
    //!
    //! @return デプスバッファテクスチャ
    //---------------------------------------------------------------------------
    TextureBase& GetDepthBufferTexture()
    {
        CpuCore core = GetCurrentCore();
        return m_DepthBufferTexture[core];
    }

    //---------------------------------------------------------------------------
    //! @briefprivate  システムのバッファモードを取得します。
    //!
    //! @return バッファモード
    //---------------------------------------------------------------------------
    BufferMode GetBufferMode()  { return m_BufferMode; }

    //---------------------------------------------------------------------------
    //! @brief エラーチェックを行います。
    //!
    //!        現在は動作しておりません。コンパイル維持の為、残してあります。
    //---------------------------------------------------------------------------
    void CheckError() { return; }

    //---------------------------------------------------------------------------
    //! @brief ランタイムエラーを取得します。
    //!        現在は動作しておりません。コンパイル維持の為、残してあります。
    //!
    //! @return ランタイムエラー
    //---------------------------------------------------------------------------
    RuntimeError GetRuntimeError() const{ return EFT_RUNTIME_ERROR_NONE; }

    //---------------------------------------------------------------------------
    //! @brief テンポラリバッファの消費量を取得します。
    //!
    //! @param[in] core     コアID
    //! @return 消費テンポラリバッファサイズ
    //---------------------------------------------------------------------------
    u32 GetUsedTemporaryBufferSize( CpuCore core = EFT_CPU_CORE_1 ) const
    {
        return m_DrawTempBuffer[core].GetUsedSize();
    }

    //---------------------------------------------------------------------------
    //! @brief テンポラリバッファの消費量を取得します。
    //!
    //! @param[in]  core     コアID
    //! @return     消費テンポラリバッファサイズ
    //---------------------------------------------------------------------------
    u32 GetUsedDoubleBufferSize( CpuCore core = EFT_CPU_CORE_1 ) const
    {
        return GetUsedTemporaryBufferSize( core );
    }

    //---------------------------------------------------------------------------
    //! @brief テンポラリバッファの最大消費量を取得します。
    //!
    //! @param[in]  core     コアID
    //! @return     最大消費テンポラリバッファサイズ
    //---------------------------------------------------------------------------
    u32 GetMaxUsedTemporaryBufferSize( CpuCore core = EFT_CPU_CORE_1 ) const
    {
        return m_DrawTempBuffer[core].GetMaxUsedSize();
    }

    //---------------------------------------------------------------------------
    //! @brief テンポラリバッファのサイズを取得します。
    //!
    //! @param[in]  core     コアID
    //! @return     テンポラリバッファサイズ
    //---------------------------------------------------------------------------
    u32 GetTemporaryBufferSize( CpuCore core = EFT_CPU_CORE_1 ) const
    {
        return m_DrawTempBuffer[core].GetSize();
    }

    //---------------------------------------------------------------------------
    //! @brief ダブルバッファのサイズを取得します。
    //!
    //! @param[in]  core     コアID
    //! @return     テンポラリバッファサイズ
    //---------------------------------------------------------------------------
    u32 GetDoubleBufferSize( CpuCore core = EFT_CPU_CORE_1 ) const
    {
        return GetTemporaryBufferSize( core );
    }

    //---------------------------------------------------------------------------
    //! @brief  グローバルカウンタを取得します。
    //!
    //! @return グローバルカウンタ
    //---------------------------------------------------------------------------
    u32 GetGlobalCounter() const { return m_GlobalCounter; }


    //---------------------------------------------------------------------------
    //! @briefprivate 描画されたパーティクル数を取得します。
    //! @return パーティクル数
    //---------------------------------------------------------------------------
    s32 GetNumPtclRender() const { return 0; }

    //---------------------------------------------------------------------------
    //! @briefprivate 描画されたエミッタ数を取得します。
    //! @return エミッタ数
    //---------------------------------------------------------------------------
    s32 GetNumEmitterRender() const { return GetRenderEmitterNum(); }

    //---------------------------------------------------------------------------
    //! @briefprivate 描画されたエミッタ数を取得します。
    //! @return エミッタ数
    //---------------------------------------------------------------------------
    s32 GetRenderEmitterNum() const { return 0; }

    //---------------------------------------------------------------------------
    //! @briefprivate 現フレームに放出されたパーティクル数を取得します。
    //! @return パーティクル数
    //---------------------------------------------------------------------------
    s32 GetNumEmittedPtcl() const{ return GetEmittedParticleNum(); }

    //---------------------------------------------------------------------------
    //! @briefprivate 現フレームに放出されたパーティクル数を取得します。
    //! @return パーティクル数
    //---------------------------------------------------------------------------
    s32 GetEmittedParticleNum() const{ return 0; }

    //---------------------------------------------------------------------------
    //! @briefprivate ストライプの処理数を取得します。
    //! @return ストライプ数
    //---------------------------------------------------------------------------
    s32 GetNumStripeCalc() const{ return 0; }

    //---------------------------------------------------------------------------
    //! @briefprivate ストライプが計算している頂点数を取得します。
    //! @return 頂点数
    //---------------------------------------------------------------------------
    s32 GetNumStripeVertexCalc() const;

    //---------------------------------------------------------------------------
    //! @briefprivate ストライプが計算している頂点数を取得します。
    //! @return 頂点数
    //---------------------------------------------------------------------------
    s32 GetStripeVertexCalcNum() const { return 0; }

    //---------------------------------------------------------------------------
    //! @briefprivate ストライプが描画している頂点数を取得します。
    //! @return 頂点数
    //---------------------------------------------------------------------------
    s32 GetNumStripeVertexDraw() const { return GetRenderStripeVertexNum(); }

    //---------------------------------------------------------------------------
    //! @briefprivate ストライプが描画している頂点数を取得します。
    //! @return 頂点数
    //---------------------------------------------------------------------------
    s32 GetRenderStripeVertexNum() const { return 0; }

    //---------------------------------------------------------------------------
    //! @brief        指定グループで有効なストライプを保持するかチェックします。
    //!
    //! @param[in]    groupID    グループID。
    //!
    //! @return       有効であればtrue、無効であればfalseを返します。
    //---------------------------------------------------------------------------
    bool IsValidStripe( u8 groupID ) const
    {
        EFT_UNUSED_VARIABLE( groupID );
        return false;
    }

    //---------------------------------------------------------------------------
    //! @briefprivate 指定グループパーティクルの処理数を取得します。
    //!
    //! @param[in] groupID     グループID
    //! @return パーティクル数
    //---------------------------------------------------------------------------
    s32 GetNumPtclCalc( u8 groupID ) const { EFT_UNUSED_VARIABLE( groupID ); return 0; }

    //---------------------------------------------------------------------------
    //! @briefprivate 指定グループの先頭のエミッタを取得します。
    //!
    //! @param[in] groupID     グループID
    //! @return エミッタ数
    //---------------------------------------------------------------------------
    const Emitter *GetEmitterHead( u8 groupID ) const{ EFT_UNUSED_VARIABLE( groupID ); return NULL; }

    //---------------------------------------------------------------------------
    //! @briefprivate 指定グループの先頭のエミッタを取得します。( const 無し版 )。
    //!
    //! @param[in] groupID     グループID
    //! @return エミッタ
    //---------------------------------------------------------------------------
    Emitter* GetEmitterHead( u8 groupID ) { EFT_UNUSED_VARIABLE( groupID ); return NULL; }

    //---------------------------------------------------------------------------
    //! @briefprivate 指定グループに所属するエミッタ数を取得します。
    //!
    //! @param[in] groupID     グループID
    //! @return エミッタ数
    //---------------------------------------------------------------------------
    u32 GetNumEmitter( u8 groupID ) { EFT_UNUSED_VARIABLE( groupID ); return 0; }

    //---------------------------------------------------------------------------
    //! @briefprivate 指定グループ、指定インデックスのエミッタを取得します。
    //! @param[in] groupID     グループID
    //! @param[in] idx     インデックス
    //! @return エミッタ
    //---------------------------------------------------------------------------
    const Emitter *GetEmitter( u8 groupID, u32 idx ) const { EFT_UNUSED_VARIABLE( groupID ); EFT_UNUSED_VARIABLE( idx ); return NULL; }

    //---------------------------------------------------------------------------
    //! @briefprivate 指定グループ、指定インデックスのエミッタを取得します。( const 無し版 )。
    //! @param[in] groupID     グループID
    //! @param[in] idx     インデックス
    //! @return エミッタ
    //---------------------------------------------------------------------------
    Emitter *GetEmitter( u8 groupID, u32 idx ) { EFT_UNUSED_VARIABLE( groupID ); EFT_UNUSED_VARIABLE( idx ); return NULL; }

    //@}

public:
    //----------------------------------------
    //! @name エフェクトリソース
    //@{

    //---------------------------------------------------------------------------
    //! @briefprivate      エミッタセットを確保します。
    //! @return     エミッタセット
    //---------------------------------------------------------------------------
    EmitterSet* AllocEmitterSet();

    //---------------------------------------------------------------------------
    //! @briefprivate      エミッタを確保します。
    //! @return エミッタ
    //---------------------------------------------------------------------------
    Emitter* AllocEmitter();

    //---------------------------------------------------------------------------
    //! @briefprivate      エミッタを初期化します。
    //! @param[in] emitter     エミッタ
    //---------------------------------------------------------------------------
    void InitializeEmitter( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @briefprivate      エミッタの終了処理をします。
    //! @param[in] emitter     エミッタ
    //---------------------------------------------------------------------------
    void FinalizeEmitter( Emitter* emitter );

    //---------------------------------------------------------------------------
    //! @briefprivate      リソース更新に伴うアップデートを行います。
    //!
    //!             ビューア機能が利用する機能です。
    //! @param[in] emitterResSet     TBD
    //! @param[in] withReset     TBD
    //---------------------------------------------------------------------------
    void UpdateFromResource( EmitterResource* emitterResSet, const bool withReset );

    //@}

    //---------------------------------------------------------------------------
    //! @brief      エミッタを描画します。
    //! @param[in] emitter     エミッタ
    //! @param[in] shaderType  シェーダタイプ
    //! @param[in] userParam   ユーザーパラメータ
    //---------------------------------------------------------------------------
    void DrawEmitter( Emitter* emitter, ShaderType shaderType, void* userParam )
    {
        DrawEmitter( emitter, shaderType, true, userParam );
    }

    //---------------------------------------------------------------------------
    //! @brief      エミッタを描画します。
    //! @param[in] emitter          エミッタ
    //! @param[in] shaderType       シェーダタイプ
    //! @param[in] calcStreamOut    ストリームアウト処理を行うかどうか
    //! @param[in] userParam        ユーザーパラメータ
    //---------------------------------------------------------------------------
    void DrawEmitter( Emitter* emitter, ShaderType shaderType, bool calcStreamOut, void* userParam )
    {
        if ( ( emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_GPU_SO ) && m_IsEnableStreamOutProcess )
        {
            nw::eft2::Resource* resource        = GetResource( emitter->emitterSet->m_ResourceID );
            EFT_NULL_ASSERT( resource );
            Shader*             streamOutShader = resource->GetStreamOutShader();
            EFT_NULL_ASSERT( streamOutShader );
            u32 gcounter = emitter->GetEmitterSet()->GetSystem()->GetGlobalCounter();

            emitter->emitterCalc->CalcStreamOut( emitter, streamOutShader, gcounter, calcStreamOut, userParam );
        }
        if ( m_IsEnableDrawProcess )
        {
            emitter->emitterCalc->Draw( emitter, shaderType, userParam );
        }
    }

protected:
    //---------------------------------------------------------------------------
    //! @briefprivate システムの初期化メソッド　
    //! @param[in] heap     TBD
    //! @param[in] dynamicHeap     TBD
    //! @param[in] config     TBD
    //---------------------------------------------------------------------------
    virtual void Initialize( Heap* heap, Heap* dynamicHeap, const Config &config );

public:
    //---------------------------------------------------------------------------
    //! @briefprivate      パーティクル ソート管理 構造体
    //---------------------------------------------------------------------------
    struct ParticleSortInfo
    {
        f32             value;                                  //!< ソート用の値
        u32             index;                                  //!< パーティクルのインデックス
    };

    //---------------------------------------------------------------------------
    //! @briefprivate      生成されたエミッタセットを管理リストへ登録します。
    //! @param[in] emitterSet  エミッタセット
    //! @param[in] groupID     グループID
    //---------------------------------------------------------------------------
    void AddEmitterSetList( EmitterSet* emitterSet, u8 groupID );

    //---------------------------------------------------------------------------
    //! @briefprivate      生成されたエミッタセットを遅延生成リストへ登録します。
    //! @param[in] emitterSet  エミッタセット
    //! @param[in] groupID     グループID
    //---------------------------------------------------------------------------
    void AddDelayCreateEmitterSetList( EmitterSet* emitterSet, u8 groupID );

    //---------------------------------------------------------------------------
    //! @briefprivate      指定エミッタセットを管理リストから削除します。
    //! @param[in] emitterSet  エミッタセット
    //---------------------------------------------------------------------------
    void RemoveEmitterSetList( EmitterSet* emitterSet );

    //---------------------------------------------------------------------------
    //! @briefprivate      指定エミッタセットを遅延生成管理リストから削除します。
    //! @param[in] emitterSet  エミッタセット
    //---------------------------------------------------------------------------
    void RemoveDelayCreateEmitterSetList( EmitterSet* emitterSet );


    //----------------------------------------
    //! @name デバッグ機能
    //@{

    //---------------------------------------------------------------------------
    //! @brief  システムの計算処理を停止させます。
    //! @param[in] enableCalc     true で再生、false で停止
    //---------------------------------------------------------------------------
    void SetEnableCalcProcess( bool enableCalc )
    {
        m_IsEnableCalcProcess = enableCalc;
    }

    //---------------------------------------------------------------------------
    //! @brief  システムの描画処理を停止させます。
    //! @param[in] enableDraw       true で描画、false で未描画
    //! @param[in] enableStreamOut  true で処理、false で停止
    //---------------------------------------------------------------------------
    void SetEnableDrawProcess( bool enableDraw, bool enableStreamOut )
    {
        m_IsEnableCalcProcess      = enableDraw;
        m_IsEnableStreamOutProcess = enableStreamOut;
    }

    //---------------------------------------------------------------------------
    //! @briefprivate  パーティクルのソートを行い、その結果の配列の先頭を返します。
    //! @param[in] result     ソート結果
    //! @param[in] arraySize  配列サイズ
    //! @param[in] emitter    エミッタ
    //! @param[in] sortType   ソートタイプ
    //! @param[in] core       コアID
    //! @return trueで成功、falseで失敗
    //---------------------------------------------------------------------------
    bool GetSortedPtclList( ParticleSortInfo** result, u32* arraySize, Emitter* emitter, const ParticleSortType sortType, const CpuCore core = EFT_CPU_CORE_1 );

    //@}

    //----------------------------------------
    //! @name パーティクルソートバッファ
    //@{

    //---------------------------------------------------------------------------
    //! @brief          パーティクルソートバッファのサイズを取得します。
    //!
    //! @return バッファのサイズ
    //---------------------------------------------------------------------------
    u32 GetSortParticleCount() const { return m_ParticleSortBufferCount; }

    //---------------------------------------------------------------------------
    //! @brief          パーティクルソートバッファを取得します。
    //!
    //! @return バッファの先頭アドレス
    //---------------------------------------------------------------------------
    ParticleSortInfo* GetParticleSortBuffer() { return m_ParticleSortBuffer[GetCurrentCore()]; }

    //@}

private:
    //---------------------------------------------------------------------------
    //! @brief      エミッタセット ソート管理 構造体
    //---------------------------------------------------------------------------
    struct sortEmitterSets
    {
        EmitterSet*     emitterSet;     //!< TBD
        u32             z;              //!< TBD
    };

    //---------------------------------------------------------------------------
    //! @brief      エミッタセット ソート用関数
    //! @param[in] a     TBD
    //! @param[in] b     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    static int CompareEmitterSetViewZ( const void *a, const void  *b );

    //---------------------------------------------------------------------------
    //! @brief      エミッタ情報をダンプする
    //! @param[in] emitter     TBD
    //! @param[in] debugLevel     TBD
    //---------------------------------------------------------------------------
    void DumpEmitterInformation( Emitter* emitter, const int debugLevel );


    //---------------------------------------------------------------------------
    //! @brief TBD
    //---------------------------------------------------------------------------
    struct ProcessingInfo
    {
        u32                 emitterSetNum;                      //!< 処理されたエミッタセット数
        u32                 emitterNum;                         //!< 処理されたエミッタ数
        u32                 calcSkipEmitterNum;                 //!< 計算処理をスキップしたエミッタ数
        u32                 emitterAnimNum;                     //!< 処理されたエミッタアニメ数
        u32                 cpuParticleNum;                     //!< 処理されたパーティクル数
        u32                 gpuParticleNum;                     //!< 処理されたパーティクル数
        u32                 gpusoParticleNum;                   //!< 処理されたパーティクル数
        u32                 cpuEmitterNum;                      //!< 処理されたCpuエミッタ数
        u32                 gpuEmitterNum;                      //!< 処理されたGpuエミッタ数
        u32                 gpusoEmitterNum;                    //!< 処理されたGpusoエミッタ数
        u32                 stripeNum;                          //!< 処理されたストライプ本数
        u32                 superStripeNum;                     //!< 処理されたスーパーストライプ本数
        u32                 allocedDynamicHeapSize;             //!< 確保された動的ヒープサイズ

        //---------------------------------------------------------------------------
        //! @brief TBD
        //---------------------------------------------------------------------------
        void Clear()
        {
            emitterSetNum           = 0;    //!< TBD
            emitterNum              = 0;    //!< TBD
            calcSkipEmitterNum      = 0;    //!< TBD
            emitterAnimNum          = 0;    //!< TBD
            cpuParticleNum          = 0;    //!< TBD
            gpuParticleNum          = 0;    //!< TBD
            gpusoParticleNum        = 0;    //!< TBD
            cpuEmitterNum           = 0;    //!< TBD
            gpuEmitterNum           = 0;    //!< TBD
            gpusoEmitterNum         = 0;    //!< TBD
            stripeNum        = 0;    //!< TBD
            superStripeNum   = 0;    //!< TBD
            allocedDynamicHeapSize  = 0;    //!< TBD
        }
    };

private:
    static  bool                    g_Initialized;                                                      //!< システム初期化済みフラグ
    bool                            m_IsUboCpuChacheFlush;                                              //!< TBD
    bool                            m_IsUboGpuChacheFlush;                                              //!< TBD
    bool                            m_IsEnableCalcProcess;                                              //!< TBD
    bool                            m_IsEnableDrawProcess;                                              //!< TBD
    bool                            m_IsEnableStreamOutProcess;                                         //!< TBD
    bool                            m_IsBatchProcessStreamOutEmitter;                                   //!< ストリームアウトエミッタを一括処理するか？
    bool                            m_ZClipEnable;                                                      //!< StreamOut処理終了時に設定するZクリップの有効/無効
    bool                            m_ColorBufferEnable[EFT_CPU_CORE_MAX];                              //!< カラーバッファの有効/無効
    Heap*                           m_StaticHeap;                                                       //!< システム初期化時に確保されるヒープ
    BufferMode                      m_BufferMode;                                                       //!< バッファモード
    DrawViewFlag                    m_DrawViewFlag[EFT_CPU_CORE_MAX];                                   //!< 描画ビューフラグ
    u32                             m_ResourceNum;                                                      //!< 最大リソース数
    u32                             m_EmitterSetNum;                                                    //!< 最大エミッタセット数
    u32                             m_EmitterNum;                                                       //!< 最大エミッタ数
    u32                             m_PtclSortBufferSize;                                               //!< 最大ソートバッファサイズ
    u32                             m_EmitterSetIdx;                                                    //!< エミッタセットインデックス(空き検索用)
    u32                             m_EmitterIdx;                                                       //!< エミッタインデックス(空き検索用)
    u32                             m_EmitterSetCreateID;                                               //!< エミッタセットの生成ＩＤ
    u32                             m_FreeEmitterNum;                                                   //!< 利用可能なエミッタ数
    u32                             m_GlobalCounter;                                                    //!< グローバルフレーム（何回計算を回したか）
    ShaderType                      m_CurrentShaderType[EFT_CPU_CORE_MAX];                              //!< カレントのシェーダタイプ
    u64                             m_ProcessingGroupID;                                                //!< 計算処理を行ったグループIDフラグ
    Resource**                      m_ResourceArray;                                                    //!< リソースクラス配列
    EmitterSet*                     m_EmitterSetArray;                                                  //!< エミッタセット配列
    Emitter*                        m_EmitterArray;                                                     //!< エミッタ配列
    EmitterDynamicUniformBlock*     m_DynamicUBOArray;                                                  //!< ダイナミックUBO配列
    EmitterSet*                     m_EmitterSetHead[EFT_GROUP_MAX];                                    //!< エミッタセットリスト先頭
    EmitterSet*                     m_EmitterSetTail[EFT_GROUP_MAX];                                    //!< エミッタセットリスト最後尾
    EmitterSet*                     m_DelayCreateEmitterSetHead[EFT_GROUP_MAX];                         //!< 遅延生成エミッタセットリスト先頭
    EmitterSet*                     m_DelayCreateEmitterSetTail[EFT_GROUP_MAX];                         //!< 遅延生成エミッタセットリスト最後尾
    EmitterSet**                    m_DelayKillEmitterSetArray[EFT_CPU_CORE_MAX];                       //!< 遅延削除エミッタセット配列
    u32                             m_DelayKillEmitterSetAddCount[EFT_CPU_CORE_MAX];                    //!< 遅延削除エミッタセット配列に追加された数
    EmitterCalc*                    m_EmitterCalc;                                                      //!< エミッタ計算クラス
    TemporaryBuffer                 m_DrawTempBuffer[EFT_CPU_CORE_MAX];	                                //!< GPU参照用の一時バッファ
    ViewParam                       m_ViewParam[EFT_CPU_CORE_MAX];		                                //!< ビューパラメータ
    ViewParam*                      m_ViewParamCopy[EFT_CPU_CORE_MAX];	                                //!< ビューパラメータ(一時バッファ)
    TextureExt                      m_FrameBufferTexture[EFT_CPU_CORE_MAX];                             //!< フレームバッファテクスチャ
    TextureExt                      m_DepthBufferTexture[EFT_CPU_CORE_MAX];                             //!< デプスバッファテクスチャ
    u32                             m_DrawPathFlag[EFT_CPU_CORE_MAX];                                   //!< 現在描画を行っている描画パス
    u32                             m_RequestFrameBufferTexturePath[EFT_CPU_CORE_MAX][EFT_GROUP_MAX];   //!< フレームバッファテクスチャを要求するパスフラグ
    u32                             m_RequestDepthBufferTexturePath[EFT_CPU_CORE_MAX][EFT_GROUP_MAX];   //!< フレームバッファテクスチャを要求するパスフラグ
    sortEmitterSets*                m_SortEmittetSet[EFT_CPU_CORE_MAX];                                 //!< エミッタセットソート用バッファ
    u32                             m_AddedSortBufferNum[EFT_CPU_CORE_MAX];                             //!< エミッタセットソート用バッファに追加した数
    u32                             m_EnableDrawPath[EFT_CPU_CORE_MAX][EFT_GROUP_MAX];                  //!< グループ単位での描画パスフラグ
    Shader*                         m_CurrentCustomShader[EFT_CPU_CORE_MAX];                            //!< カレントのシェーダ
    CustomShaderTextureSlot         m_CurrentCustomShaderTextureSlot[EFT_CPU_CORE_MAX];                 //!< カレントのテクスチャスロット
    bool                            m_IsEnableCallback[EFT_CALLBACK_ID_MAX];                            //!< カスタムアクション/シェーダ コールバック有効無効
    CallbackSet                     m_Callback[EFT_CALLBACK_ID_MAX];                                    //!< カスタムアクション/シェーダ コールバック実体
    CallbackSet                     m_EmitterPluginCallback[EFT_EMITTER_PLUGIN_CALLBACK_ID_MAX];        //!< エミッタプラグイン コールバック実体
    u32                             m_DrawPathCallbackFlag[EFT_DRAW_PATH_CALLBACK_MAX];                 //!< 描画パス
    EmitterSetInitializeCallback    m_EmitterSetInitializeCallback;                                     //!< エミッタセット初期化コールバック
    EmitterSetFinalizeCallback      m_EmitterSetFinalizeCallback;                                       //!< エミッタセット破棄時コールバック
    DrawPathRenderStateSetCallback  m_DrawPathRenderStateSetCallback[EFT_DRAW_PATH_CALLBACK_MAX];       //!< 描画パス コールバック 実体
    DrawEmitterProfilerCallback     m_EmitterDrawProfiler;                                              //!< カレントのエミッタ描画プロファイラ
    EmitterCalcLodCallback          m_EmitterCalcLodCallback;                                           //!< エミッタカリングコールバック
    EmitterDrawCullingCallback      m_EmitterDrawCullingCallback;                                       //!< カリングコールバック
    CustomFieldCallback             m_CustomFieldCallback;                                              //!< カスタムフィールドコールバック
    u32                             m_RandomWorkSize;                                                   //!< 乱数ワークサイズ
    u32                             m_DelayFreeWorkSize;                                                //!< 遅延解放用ワークサイズ
    u32                             m_ResourceWorkSize;                                                 //!< リソースワークサイズ
    u32                             m_EmitterSetWorkSize;                                               //!< エミッタセットワークサイズ
    u32                             m_EmitterWorkSize;                                                  //!< エミッタワークサイズ
    u32                             m_EmitterCalcWorkSize;                                              //!< エミッタ計算クラスワークサイズ
    u32                             m_EmitterUboWorkSize;                                               //!< エミッタUboワークサイズ
    u32                             m_EmitterSetSortWorkSize;                                           //!< エミッタセットソートワークサイズ
    ProcessingInfo                  m_ProcessingInfo;                                                   //!< システム稼動状況
    Emitter*                        m_StreamOutEmitter;                                                 //!< ストリームアウト エミッタのリスト
    Emitter*                        m_StreamOutEmitterTail;                                             //!< ストリームアウト エミッタのリスト最後尾
    u32                             m_ParticleSortBufferCount;                                          //!< パーティクルソート用バッファのサイズ
    ParticleSortInfo*               m_ParticleSortBuffer[EFT_CPU_CORE_MAX];                             //!< パーティクルソート用バッファの先頭アドレス
    TextureResourceInitializer      m_TextureInitializer;                                               //!< テクスチャ初期化関数
    TextureResourceFinalizer        m_TextureFinalizer;                                                 //!< テクスチャ破棄関数
    void*                           m_CommonCstmShaderUniformBlock[EFT_CUSTOM_SHADER_UBO_MAX];          //!< 登録された全エミッタ描画で共通で利用されるカスタムUBOバッファ
    u32                             m_CommonCstmShaderUniformBlockSize[EFT_CUSTOM_SHADER_UBO_MAX];      //!< 登録された全エミッタ描画で共通で利用されるカスタムUBOバッファサイズ
    nw::ut::Mutex                   m_Mutex;                                                            //!< 同期オブジェクト

};


} // namespace eft2
} // namespace nw
