﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/typeDef2.h>
#include <nw/eft/eft2_ShaderKey.h>

namespace nw   {
namespace eft2 {

class Heap;
struct EmitterResource;

#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4201)
#endif

//---------------------------------------------------------------------------
//! @brief  シェーダコード配列
//---------------------------------------------------------------------------
#ifdef EFT_OGL
struct ShaderCodeArg
{
    u32             codeNum;            //!< TBD
    union
    {
        //!< TBD
        struct
        {
            const GLchar*   specDec;    //!< TBD
            const GLchar*   soDef;      //!< TBD
            const GLchar*   setting;    //!< TBD
            const GLchar*   glsl;       //!< TBD
            const GLchar*   dec;        //!< TBD
            const GLchar*   custom;     //!< TBD
            const GLchar*   emPlugin;   //!< TBD
            const GLchar*   program;    //!< TBD
            const GLchar*   combiner;   //!< TBD
            const GLchar*   soDec;      //!< TBD
            const GLchar*   soCustom;   //!< TBD
            const GLchar*   streamout;  //!< TBD
        };
        const GLchar*   code[12];       //!< TBD
    };
    union
    {
        //!< TBD
        struct
        {
            GLint           specDecSize;    //!< TBD
            GLint           soDefSize;      //!< TBD
            GLint           settingSize;    //!< TBD
            GLint           glslSize;       //!< TBD
            GLint           decSize;        //!< TBD
            GLint           customSize;     //!< TBD
            GLint           emPluginSize;   //!< TBD
            GLint           programSize;    //!< TBD
            GLint           combinerSize;   //!< TBD
            GLint           soDecSize;      //!< TBD
            GLint           soCustomSize;   //!< TBD
            GLint           streamoutSize;  //!< TBD
        };
        GLint   codeSize[12];               //!< TBD
    };

    const GLchar*   specDecGenericCode;     //!< TBD
    GLint           specDecGenericCodeSize; //!< TBD
    const GLchar*   specDecTargetCode;      //!< TBD
    GLint           specDecTargetCodeSize;  //!< TBD

    const GLchar*   customGeneralCode;      //!< TBD
    GLint           customGeneralCodeSize;  //!< TBD
    const GLchar*   customCode[8];          //!< TBD
    GLint           customCodeSize[8];      //!< TBD
    GLchar**        combinerCode;           //!< TBD
    GLint*          combinerCodeSize;       //!< TBD
    u32             combinerCodeCount;      //!< TBD

    const GLchar*   emitterPluginCode[EFT_EMITTER_PLUGIN_CALLBACK_ID_MAX];      //!< TBD
    GLint           emitterPluginCodeSize[EFT_EMITTER_PLUGIN_CALLBACK_ID_MAX];  //!< TBD
};
#endif


//---------------------------------------------------------------------------
//! @brief  頂点フォーマット
//---------------------------------------------------------------------------
typedef u32 VertexFormat;
enum
{
#ifdef EFT_OGL
    FORMAT_32_SINT              = 0,                                    //!< TBD
    FORMAT_32_UINT              = 1,                                    //!< TBD
    FORMAT_32_32_FLOAT          = 2,                                    //!< TBD
    FORMAT_32_32_32_FLOAT       = 3,                                    //!< TBD
    FORMAT_32_32_32_32_FLOAT    = 4,                                    //!< TBD
    FORMAT_16_16_16_16_FLOAT    = 5,                                    //!< TBD
    FORMAT_8_SINT               = 6,                                    //!< TBD
    FORMAT_8_UINT               = 7                                     //!< TBD
#endif
#if EFT_GX2
    FORMAT_32_SINT              = GX2_ATTRIB_FORMAT_32_SINT,            //!< TBD
    FORMAT_32_UINT              = GX2_ATTRIB_FORMAT_32_UINT,            //!< TBD
    FORMAT_32_32_FLOAT          = GX2_ATTRIB_FORMAT_32_32_FLOAT,        //!< TBD
    FORMAT_32_32_32_FLOAT       = GX2_ATTRIB_FORMAT_32_32_32_FLOAT,     //!< TBD
    FORMAT_32_32_32_32_FLOAT    = GX2_ATTRIB_FORMAT_32_32_32_32_FLOAT,  //!< TBD
    FORMAT_16_16_16_16_FLOAT    = GX2_ATTRIB_FORMAT_16_16_16_16_FLOAT,  //!< TBD
    FORMAT_8_SINT               = GX2_ATTRIB_FORMAT_8_SINT,             //!< TBD
    FORMAT_8_UINT               = GX2_ATTRIB_FORMAT_8_UINT              //!< TBD
#endif
};

//---------------------------------------------------------------------------
//! @brief  シェーダリソース
//---------------------------------------------------------------------------
struct ShaderResource
{
public:
    //---------------------------------------------------------------------------
    //! @brief  カスタムシェーダID
    //---------------------------------------------------------------------------
    enum CustomShaderID
    {
        EFT_CUSTOM_SHADER_ID_0 = 0,     //!< カスタムシェーダID 0
        EFT_CUSTOM_SHADER_ID_1 = 1,     //!< カスタムシェーダID 1
        EFT_CUSTOM_SHADER_ID_2 = 2,     //!< カスタムシェーダID 2
        EFT_CUSTOM_SHADER_ID_3 = 3,     //!< カスタムシェーダID 3
        EFT_CUSTOM_SHADER_ID_4 = 4,     //!< カスタムシェーダID 4
        EFT_CUSTOM_SHADER_ID_5 = 5,     //!< カスタムシェーダID 5
        EFT_CUSTOM_SHADER_ID_6 = 6,     //!< カスタムシェーダID 6
        EFT_CUSTOM_SHADER_ID_7 = 7,     //!< カスタムシェーダID 7
        EFT_CUSTOM_SHADER_ID_MAX = 8,
    };

    //---------------------------------------------------------------------------
    //! @brief  アトリビュート最大数
    //---------------------------------------------------------------------------
    enum{ cAttributeMax = 32 };

    //---------------------------------------------------------------------------
    //! @brief  終了処理
    //! @param[in] heap TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    void Finalize( Heap* heap );

    //---------------------------------------------------------------------------
    //! @brief  無効化
    //---------------------------------------------------------------------------
    void Invalidate();

    //------------------------------------------------------------------------------
    //! @brief  シェーダを有効にする
    //------------------------------------------------------------------------------
    void Bind() const;

    //------------------------------------------------------------------------------
    //! @brief  アトリビュートを取得する
    //! @param[in] name TBD
    //! @param[in] index TBD
    //! @param[in] fmt TBD
    //! @param[in] offset TBD
    //! @param[in] instancingAttr TBD
    //! @param[in] endianSwap TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    u32 GetAttribute( const char* name, u32 index, VertexFormat fmt, u32 offset = 0, bool instancingAttr = false, bool endianSwap = false );

    //------------------------------------------------------------------------------
    //! @brief  ユニフォームロケーションを取得する
    //! @param[in] name TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    u32 GetUniformLocation( const char* name ) const;

    //------------------------------------------------------------------------------
    //! @brief  サンプラーロケーションを取得する
    //! @param[in] name TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    u32 GetFragmentSamplerLocation( const char* name );

    //------------------------------------------------------------------------------
    //! @brief  サンプラーロケーションを取得する
    //! @param[in] name TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    u32 GetVertexSamplerLocation( const char* name );

#ifdef EFT_OGL
    //---------------------------------------------------------------------------
    //! @brief  初期化
    //! @param[in] emitterResSet TBD
    //! @param[in] addCompileDef TBD
    //! @param[in] isCancelCustomShader TBD
    //---------------------------------------------------------------------------
    void Initialize( const EmitterResource* emitterResSet, const char* addCompileDef );

    //---------------------------------------------------------------------------
    //! @brief  シェーダコンパイルを行います
    //! @param[in] emitterResSet TBD
    //! @param[in] addCompileDef TBD
    //! @param[in] isCancelCustomShader TBD
    //---------------------------------------------------------------------------
    void Compile( const EmitterResource* emitterResSet, bool isCancelCustomShader );

    //---------------------------------------------------------------------------
    //! @brief  ストリームアウトシェーダの初期化
    //---------------------------------------------------------------------------
    void InitializeStreamOutShader();

    //---------------------------------------------------------------------------
    //! @brief  カスタムシェーダコード用のバッファを初期化します。
    //---------------------------------------------------------------------------
    static void InitializeCustomShader();

    //---------------------------------------------------------------------------
    //! @brief  カスタムシェーダコードを設定します。
    //! @param[in] id TBD
    //! @param[in] code TBD
    //! @param[in] codeSize TBD
    //---------------------------------------------------------------------------
    static void SetCustomShaderCode( CustomShaderID id, GLchar* code, GLint codeSize );

    //---------------------------------------------------------------------------
    //! @brief  シェーダコードをファイル出力します。
    //! @param[in] writeTextPath TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    static bool WriteVtxShaderCode( const char* writeTextPath );

    //---------------------------------------------------------------------------
    //! @brief  シェーダコードをファイル出力します。
    //! @param[in] writeTextPath TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    static bool WriteFragShaderCode( const char* writeTextPath );

    //---------------------------------------------------------------------------
    //! @brief  プログラムIDを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    GLuint GetProgramID() const { return program; }
    GLuint GetProgramId() const { return program; }

    //---------------------------------------------------------------------------
    //! @brief  シェーダコードを設定します。
    //! @param[in] vshArg TBD
    //! @param[in] fshArg TBD
    //---------------------------------------------------------------------------
    static void SetShaderCode( const ShaderCodeArg& vshArg, const ShaderCodeArg& fshArg );

    //---------------------------------------------------------------------------
    //! @brief  シェーダキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    const ShaderKey* GetShaderKey() const { return &shaderKey; }

    //---------------------------------------------------------------------------
    //! @brief  エラー状態かどうか。
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsError() const { return isError; }
#endif

#if EFT_GX2
    //---------------------------------------------------------------------------
    //! @brief  初期化
    //! @param[in] heap TBD
    //! @param[in] binary TBD
    //! @param[in] vertexIdx TBD
    //! @param[in] fragIdx TBD
    //---------------------------------------------------------------------------
    void Initialize( Heap* heap, void* binary, u32 vertexIdx, u32 fragIdx );

    //------------------------------------------------------------------------------
    //! @brief  頂点シェーダを取得する
    //------------------------------------------------------------------------------
    const GX2VertexShader* GetVertexShader() const { return vertexShader; }

    //------------------------------------------------------------------------------
    //! @brief  フラグメントシェーダを取得する
    //------------------------------------------------------------------------------
    const GX2PixelShader* GetFragmentShader() const { return fragShader; }

    //------------------------------------------------------------------------------
    //! @brief  シェーダをセットアップする
    //------------------------------------------------------------------------------
    void SetupShader( Heap* heap );
#endif


    //---------------------------------------------------------------------------
    //! @brief  シェーダコードをコンパイルする
    //! @param[in] compileSetting TBD
    //! @param[in] customShaderIndex TBD
    //! @param[in] emitterPluginShaderIndex TBD
    //! @param[in] eftCombinerShaderIndex TBD
    //! @param[in] isCancelCustomShader TBD
    //---------------------------------------------------------------------------
    void _Compile( const char* compileSetting, u32 customShaderIndex, u32 emitterPluginShaderIndex, u32 eftCombinerShaderIndex, bool isCancelCustomShader );

private:

#ifdef EFT_OGL
    //------------------------------------------------------------------------------
    //! @brief  gCurrnetVshCodeArg/gCurrnetFshCodeArg の内容でシェーダコンパイルを行う
    //! @param[in] isStreamOut TBD
    //------------------------------------------------------------------------------
    bool _CompileCurrentShader(  bool isStreamOut );

    GLuint                  program;                                            //!< シェーダプログラム
    GLuint                  vtxShader;                                          //!< 頂点シェーダ
    GLuint                  fragShader;                                         //!< フラグメントシェーダ
    ShaderKey               shaderKey;                                          //!< シェーダキー
    static  ShaderCodeArg   gOriginalVshCodeArg;                                //!< 頂点シェーダ
    static  ShaderCodeArg   gOriginalFshCodeArg;                                //!< フラグメントシェーダ
    static  ShaderCodeArg   gCurrnetVshCodeArg;                                 //!< 頂点シェーダ
    static  ShaderCodeArg   gCurrnetFshCodeArg;                                 //!< フラグメントシェーダ
    static  GLchar*         gCustomShaderCode[EFT_CUSTOM_SHADER_ID_MAX];        //!< カスタムシェーダコード
    static  GLint           gCustomShaderCodeSize[EFT_CUSTOM_SHADER_ID_MAX];    //!< カスタムシェーダコード
    bool                    isError;                                            //!< シェーダエラーかどうか
#endif
#if EFT_GX2
    GX2VertexShader*        vertexShader;                                       //!< 頂点シェーダ
    GX2PixelShader*         fragShader;                                         //!< フラグメントシェーダ
    GX2FetchShader          fetchShader;                                        //!< フェッチシェーダ
    void*                   fetchShaderBuf;                                     //!< フェッチシェーダバッファ
    u32                     attribsNum;                                         //!< アトリビュート数
    GX2AttribStream         attribs[cAttributeMax];                             //!< Gx2アトリビュート
    u32                     attribsIndex[cAttributeMax];                        //!< アトリビュート
    bool                    isInitialized;                                      //!< 初期化済みかどうか
#endif
};





//---------------------------------------------------------------------------
//! @brief  シェーダリソースベース 削除予定
//---------------------------------------------------------------------------
struct ShaderResourceBase
{
public:
#ifdef EFT_OGL
    //---------------------------------------------------------------------------
    //! @brief  コンパイル
    //! @param[in] vshShaderCode TBD
    //! @param[in] vshShaderCodeSize TBD
    //! @param[in] fshShaderCode TBD
    //! @param[in] fshShaderCodeSize TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool Compile( const GLchar* vshShaderCode, u32 vshShaderCodeSize,
                        const GLchar* fshShaderCode, u32 fshShaderCodeSize );
#endif

    //---------------------------------------------------------------------------
    //! @brief  終了処理
    //! @param[in] heap TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool Finalize( Heap* heap );

    //------------------------------------------------------------------------------
    //! @brief  アトリビュートを取得する
    //! @param[in] name TBD
    //! @param[in] index TBD
    //! @param[in] fmt TBD
    //! @param[in] offset TBD
    //! @param[in] instancingAttr TBD
    //! @param[in] endianSwap TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    u32 GetAttribute( const char* name, u32 index, VertexFormat fmt, u32 offset, bool instancingAttr, bool endianSwap );

    //------------------------------------------------------------------------------
    //! @brief  ユニフォームロケーションを取得する
    //! @param[in] name TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    u32 GetVertexUniformLocation( const char* name );

    //------------------------------------------------------------------------------
    //! @brief  ユニフォームロケーションを取得する
    //! @param[in] name TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    u32 GetFragmentUniformLocation( const char* name );

    //---------------------------------------------------------------------------
    //! @brief  フラグメントサンプラーロケーションを取得する
    //! @param[in] name TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetFragmentSamplerLocation( const char* name );

    //------------------------------------------------------------------------------
    //! @brief  シェーダを有効にする
    //! @return TBD
    //------------------------------------------------------------------------------
    void Bind() const;

    //------------------------------------------------------------------------------
    //! @brief  シェーダをセットアップします
    //! @param[in] heap TBD
    //------------------------------------------------------------------------------
    void SetupShader( Heap* heap );

#if EFT_GX2
    //---------------------------------------------------------------------------
    //! @briefprivate   シェーダを生成します
    //! @param[in] heap TBD
    //! @param[in] gshBuffer TBD
    //! @param[in] gshBufferSize TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool CreateShader( Heap* heap, const void* gshBuffer, u32 gshBufferSize );
    //---------------------------------------------------------------------------
    //! @briefprivate   シェーダを生成します
    //! @param[in] heap TBD
    //! @param[in] gshBuffer TBD
    //! @param[in] gshBufferSize TBD
    //! @param[in] index TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool CreateShader( Heap* heap, const void* gshBuffer, u32 gshBufferSize, u32 index );

    //------------------------------------------------------------------------------
    //! @brief  アトリビュートロケーションを取得する
    //! @param[in] name TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    u32 GetAttributeLocation( const char* name );

private:
    //------------------------------------------------------------------------------
    //! @briefprivate Read Vertex Shader from buffer
    //! @param[in] heap TBD
    //! @param[in] ppShader TBD
    //! @param[in] index TBD
    //! @param[in] pData TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    bool _DEMOGFDReadVertexShader( nw::eft2::Heap* heap, GX2VertexShader** ppShader, u32 index, const void* pData );

    //------------------------------------------------------------------------------
    //! @briefprivate Read Pixel Shader from buffer
    //! @param[in] heap TBD
    //! @param[in] ppShader TBD
    //! @param[in] index TBD
    //! @param[in] pData TBD
    //! @return TBD
    //------------------------------------------------------------------------------
    bool _DEMOGFDReadPixelShader( nw::eft2::Heap* heap, GX2PixelShader** ppShader, u32 index, const void* pData );
#endif

private:
    //---------------------------------------------------------------------------
    //! @brief  アトリビュート最大数
    //---------------------------------------------------------------------------
    enum{ cAttributeMax = 32 };

#ifdef EFT_OGL
    GLuint                  program;                                            //!< シェーダプログラム
    GLuint                  vtxShader;                                          //!< 頂点シェーダ
    GLuint                  fragShader;                                         //!< フラグメントシェーダ
#endif
#if EFT_GX2
    //! ディスプレイリストのサイズ
    enum
    {
        DISPLAY_LSIT_SIZE = 512
    };
    u8*                     displayList;                                        //!< ディスプレイリスト
    u32                     displayListSize;                                    //!< ディスプレイリストサイズ

    GX2VertexShader*        vertexShader;                                       //!< 頂点シェーダ
    GX2PixelShader*         fragShader;                                         //!< フラグメントシェーダ
    GX2FetchShader          fetchShader;                                        //!< フェッチシェーダ
    void*                   fetchShaderBuf;                                     //!< フェッチシェーダバッファ
    u32                     attribsNum;                                         //!< アトリビュート数
    GX2AttribStream         attribs[cAttributeMax];                             //!< Gx2アトリビュート
    u32                     attribsIndex[cAttributeMax];                        //!< アトリビュート
    bool                    isInitialized;                                      //!< 初期化済みかどうか
#endif
};


} // namespace nw
} // namespace eft2
