﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Texture.h>

namespace nw   {
namespace eft2 {

//---------------------------------------------------------------------------
//! @brief      CurlNoise パラメータ初期化処理
//!
//!             CPU参照用のカールノイズ配列とGPU参照用のカールノイズテクスチャを生成します。
//---------------------------------------------------------------------------
void InitializeCurlNoise();

//---------------------------------------------------------------------------
//! @brief      CurlNoise パラメータ終了処理
//!
//!             生成されたGPU参照用のカールノイズテクスチャを破棄します。
//---------------------------------------------------------------------------
void FinalizeCurlNoise();

//---------------------------------------------------------------------------
//! @brief      CurlNoise テクスチャでヒープから確保したメモリサイズを取得する
//! @return     ヒープから確保したメモリサイズ
//---------------------------------------------------------------------------
u32 GetCurlNoiseTextureAllocedSize();

//---------------------------------------------------------------------------
//! @brief      カールノイズテクスチャを取得します。
//! @return     テクスチャへの参照
//---------------------------------------------------------------------------
TextureBase& GetCurlNoiseTexture();

//---------------------------------------------------------------------------
//! @brief          カールノイズテクスチャを取得します。
//! @param[in]      emitter エミッタへのポインタ
//! @param[in,out]  pos     位置
//! @param[in,out]  vec     速度
//! @param[in]      attr    パーティクルアトリビュートへのポインタ
//---------------------------------------------------------------------------
void _CalcParticleBehavior_FieldCurlNoise( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr );

//---------------------------------------------------
//! @brief カールノイズテクスチャ
//---------------------------------------------------
struct CurlNoiseTexture
{
    //---------------------------------------------------------------------------
    //! @brief  初期化処理
    //---------------------------------------------------------------------------
    void Initialize();

    //---------------------------------------------------------------------------
    //! @brief  終了処理
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief  無効化
    //---------------------------------------------------------------------------
    void Invalidate();

    //---------------------------------------------------------------------------
    //! @brief      カールノイズテクスチャを取得
    //! @return     テクスチャへの参照
    //---------------------------------------------------------------------------
    TextureBase& GetTexture()
    {
        return texture;
    }

    TextureExt  texture;        //!< テクスチャ

#ifdef EFT_OGL
    GLuint      textureid;      //!< テクスチャID
#endif
#if EFT_GX2
    GX2Texture  gx2Texture;     //!< テクスチャ
#endif
    void*       textureImage;   //!< テクスチャ画像
    u32         allocedSize;    //!< メモリサイズ
};


//---------------------------------------------------
//! @brief カールノイズ クラス
//!
//!        カールノイズテーブルへアクセスする為のクラスです。
//---------------------------------------------------
class CurlNoiseTbl
{
public:
    //---------------------------------------------------------------------------
    //! @brief  S8ベクトル構造体
    //---------------------------------------------------------------------------
    struct Vec3S8
    {
        s8 x;       //!< x成分
        s8 y;       //!< y成分
        s8 z;       //!< z成分
    };

    //---------------------------------------------------------------------------
    //! @brief  初期化処理です。
    //! @param[in] bufWidth バッファ幅
    //! @param[in] pTbl     テーブルへのポインタ
    //---------------------------------------------------------------------------
    void Initialize( s32 bufWidth, void* pTbl )
    {
        m_BufWidth = bufWidth;
        m_pCurlNoiseBufS8 = (Vec3S8*)pTbl;
    }

    //---------------------------------------------------------------------------
    //! @brief  カールノイズのs8の値をVector3で取り出します。
    //! @param[in] x x成分
    //! @param[in] y y成分
    //! @param[in] z z成分
    //! @return      カールノイズ成分をVector3Dで返す
    //---------------------------------------------------------------------------
    nw::math::Vector3 GetCurlNoiseS8( s32 x, s32 y, s32 z )
    {
        if( x < 0 ) x *= -1;
        if( y < 0 ) y *= -1;
        if( z < 0 ) z *= -1;
        x %= m_BufWidth;
        y %= m_BufWidth;
        z %= m_BufWidth;

        s32 idx = x*m_BufWidth*m_BufWidth + y*m_BufWidth + z;
        nw::math::Vector3 ret;
        ret.x = m_pCurlNoiseBufS8[idx].x / 127.0f;
        ret.y = m_pCurlNoiseBufS8[idx].y / 127.0f;
        ret.z = m_pCurlNoiseBufS8[idx].z / 127.0f;
        return ret;
    }

    //---------------------------------------------------------------------------
    //! @brief  カールノイズのs8の値を近傍8点補間してVector3で取り出します。
    //! @param[in] x    x成分
    //! @param[in] y    y成分
    //! @param[in] z    z成分
    //! @return         カールノイズ成分をVector3Dで返す
    //---------------------------------------------------------------------------
    nw::math::Vector3  GetCurlNoiseS8Interpolate( f32 x, f32 y, f32 z )
    {
        s32 ix = (s32)x;
        s32 iy = (s32)y;
        s32 iz = (s32)z;
        f32 dx = x - (f32)ix;
        f32 dy = y - (f32)iy;
        f32 dz = z - (f32)iz;
        f32 dxi = 1.0f - dx;
        f32 dyi = 1.0f - dy;
        f32 dzi = 1.0f - dz;

        nw::math::Vector3 ret000 = GetCurlNoiseS8( ix,   iy,   iz );
        nw::math::Vector3 ret001 = GetCurlNoiseS8( ix+1, iy,   iz );
        nw::math::Vector3 ret010 = GetCurlNoiseS8( ix,   iy+1, iz );
        nw::math::Vector3 ret011 = GetCurlNoiseS8( ix+1, iy+1, iz );

        nw::math::Vector3 ret100 = GetCurlNoiseS8( ix,   iy,   iz+1 );
        nw::math::Vector3 ret101 = GetCurlNoiseS8( ix+1, iy,   iz+1 );
        nw::math::Vector3 ret110 = GetCurlNoiseS8( ix,   iy+1, iz+1 );
        nw::math::Vector3 ret111 = GetCurlNoiseS8( ix+1, iy+1, iz+1 );

        nw::math::Vector3 ret00x = ret000 * dxi + ret001 * dx;
        nw::math::Vector3 ret01x = ret010 * dxi + ret011 * dx;
        nw::math::Vector3 ret0y  = ret00x * dyi + ret01x * dy;

        nw::math::Vector3 ret10x = ret100 * dxi + ret101 * dx;
        nw::math::Vector3 ret11x = ret110 * dxi + ret111 * dx;
        nw::math::Vector3 ret1y  = ret10x * dyi + ret11x * dy;

        nw::math::Vector3 ret  = ret0y * dzi + ret1y * dz;
        return ret;
    }

    s32     m_BufWidth;         //!< バッファ幅
    Vec3S8* m_pCurlNoiseBufS8;  //!< カールノイズバッファ
};

}
}
