﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_PRIMITIVE_RENDERER_H_
#define NW_DEV_PRIMITIVE_RENDERER_H_

#include <nw/dev/dev_PrimitiveRendererBase.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
#include <nw/dev/gl/dev_PrimitiveRendererGL.h>
#elif defined(NW_PLATFORM_CAFE)
#include <nw/dev/cafe/dev_PrimitiveRendererCafe.h>
#endif

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief        基本図形を描画するクラスです。
//---------------------------------------------------------------------------
class PrimitiveRenderer
{
public:
    //----------------------------------------
    //! @name 引数クラス
    //  @{

    //! @brief Quad 描画用の引数クラスです。
    class QuadArg
    {
    public:
        //! @brief コンストラクタです。
        QuadArg()
            : m_Center ( nw::math::VEC3::Zero() ),
              m_Size ( 1.f, 1.f ),
              m_Horizonal ( false )
              {}

        //! @brief 中心座標で位置を指定します。
        //! @param[in] vec 中心座標です。
        //! @return 自身のオブジェクトの参照を返します。
        QuadArg& SetCenter( const nw::math::VEC3& vec ) { m_Center = vec; return *this; }

        //! @brief 中心座標で位置を指定します。
        //! @param[in] x 中心座標の x の値です。
        //! @param[in] y 中心座標の y の値です。
        //! @param[in] z 中心座標の z の値です。
        //! @return 自身のオブジェクトの参照を返します。
        QuadArg& SetCenter( f32 x, f32 y, f32 z ) { m_Center.Set( x, y, z ); return *this; }

        //! @brief サイズを指定します。
        //! @param[in] size サイズです。
        //! @return 自身のオブジェクトの参照を返します。
        QuadArg& SetSize( const nw::math::VEC2& size ) { m_Size = size; return *this; }

        //! @brief サイズを指定します。
        //! @param[in] x サイズの x の値です。
        //! @param[in] y サイズの y の値です。
        //! @return 自身のオブジェクトの参照を返します。
        QuadArg& SetSize( f32 x, f32 y ) { m_Size.Set( x, y ); return *this; }

        //! @brief 左上の点の座標で位置とサイズを指定します。
        //! @param[in] vec 左上の点の座標です。
        //! @param[in] size サイズです。
        //! @return 自身のオブジェクトの参照を返します。
        QuadArg& SetCornerAndSize( const nw::math::VEC3& vec, const nw::math::VEC2& size )
        {
            m_Center.Set( size.x * 0.5f + vec.x, size.y * 0.5f + vec.y, vec.z );
            m_Size = size;
            return *this;
        }

        //! @brief 左上の点の座標で位置とサイズを指定します。
        //! @param[in] x 左上の点の x 座標です。
        //! @param[in] y 左上の点の y 座標です。
        //! @param[in] z 左上の点の z 座標です。
        //! @param[in] size サイズです。
        //! @return 自身のオブジェクトの参照を返します。
        QuadArg& SetCornerAndSize( f32 x, f32 y, f32 z, const nw::math::VEC2& size )
        {
            return SetCornerAndSize( nw::math::VEC3( x, y, z ), size );
        }

        //! @brief 左上の点の座標で位置とサイズを指定します。
        //! @param[in] vec 左上の点の座標です。
        //! @param[in] w 幅です。
        //! @param[in] h 高さです。
        //! @return 自身のオブジェクトの参照を返します。
        QuadArg& SetCornerAndSize( const nw::math::VEC3& vec, f32 w, f32 h )
        {
            return SetCornerAndSize( vec, nw::math::VEC2( w, h ) );
        }

        //! @brief 左上の点の座標で位置とサイズを指定します。
        //! @param[in] x 左上の点の x 座標です。
        //! @param[in] y 左上の点の y 座標です。
        //! @param[in] z 左上の点の z 座標です。
        //! @param[in] w 幅です。
        //! @param[in] h 高さです。
        //! @return 自身のオブジェクトの参照を返します。
        QuadArg& SetCornerAndSize( f32 x, f32 y, f32 z, f32 w, f32 h )
        {
            return SetCornerAndSize( nw::math::VEC3( x, y, z ), nw::math::VEC2( w, h ) );
        }

        //! @brief 上側の色と下側の色を設定します。
        //! @param[in] colorTop 上側の色です。
        //! @param[in] colorBottom 下側の色です。
        //! @return 自身のオブジェクトの参照を返します。
        QuadArg& SetColor( const nw::ut::Color4u8& colorTop, const nw::ut::Color4u8& colorBottom )
        {
            m_Horizonal = false;
            m_Color0 = colorTop;
            m_Color1 = colorBottom;
            return *this;
        }

        //! @brief 単色を設定します。
        //! @param[in] color 色です。
        //! @return 自身のオブジェクトの参照を返します。
        QuadArg& SetColor( const nw::ut::Color4u8& color ) { return SetColor( color, color ); }

        //! @brief 左側の色と右側の色を設定します。
        //! @param[in] colorLeft 左側の色です。
        //! @param[in] colorRight 右側の色です。
        //! @return 自身のオブジェクトの参照を返します。
        QuadArg& SetColorHorizontal( const nw::ut::Color4u8& colorLeft, const nw::ut::Color4u8& colorRight )
        {
            m_Horizonal = true;
            m_Color0 = colorLeft;
            m_Color1 = colorRight;
            return *this;
        }

        //! @brief 中心座標を取得します。
        //! @return 中心座標を返します。
        const nw::math::VEC3& GetCenter() const { return m_Center; }

        //! @brief サイズを取得します。
        //! @return サイズを返します。
        const nw::math::VEC2& GetSize() const { return m_Size; }

        //! @brief 色 0 (左または上) を取得します。
        //! @return 色 0 を返します。
        const nw::ut::Color4u8& GetColor0() const { return m_Color0; }

        //! @brief 色 1 (右または下) を取得します。
        //! @return 色 1 を返します。
        const nw::ut::Color4u8& GetColor1() const { return m_Color1; }

        //! @brief 縦にグラデーションするモードかどうか取得します。
        //! @return 縦にグラデーションするモードの場合は true を返します。
        bool IsHorizonal() const { return m_Horizonal; }

    private:
        nw::math::VEC3 m_Center;
        nw::math::VEC2 m_Size;
        nw::ut::Color4u8  m_Color0;
        nw::ut::Color4u8  m_Color1;
        bool           m_Horizonal;
    };

    //! @brief Quad にテクスチャを貼り付ける場合の、 UV 値の指定用引数クラスです。
    class UVArg
    {
    public:
        //! @brief コンストラクタです。
        UVArg() : m_UVSrc( 0.f, 0.f ), m_UVSize( 1.f, 1.f ) {}

        //! @brief 左下の点とサイズを設定します。
        //! @param[in] srcU 左下の点の U の値です。
        //! @param[in] srcV 左下の点の V の値です。
        //! @param[in] sizeU U 方向のサイズです。
        //! @param[in] sizeV V 方向のサイズです。
        //! @return 自身のオブジェクトの参照を返します。
        UVArg& Set( f32 srcU, f32 srcV, f32 sizeU, f32 sizeV )
        {
            m_UVSrc.Set( srcU, srcV );
            m_UVSize.Set( sizeU, sizeV );
            return *this;
        }

        //! @brief 左下の点の UV 値を設定します。
        //! @param[in] vec 左下の点の UV 値です。
        //! @return 自身のオブジェクトの参照を返します。
        UVArg& SetUVSrc( const nw::math::VEC2& vec ) { m_UVSrc = vec; return *this; }

        //! @brief 左下の点の UV 値を設定します。
        //! @param[in] u 左下の点の u の値です。
        //! @param[in] v 左下の点の v の値です。
        //! @return 自身のオブジェクトの参照を返します。
        UVArg& SetUVSrc( f32 u, f32 v ) { m_UVSrc.Set( u, v ); return *this; }

        //! @brief UV のサイズを設定します(右上点の UV - 左下点の UV)。
        //! @param[in] vec UV のサイズです。
        //! @return 自身のオブジェクトの参照を返します。
        UVArg& SetUVSize( const nw::math::VEC2& vec ) { m_UVSize = vec; return *this; }

        //! @brief UV のサイズを設定します(右上点の UV - 左下点の UV)。
        //! @param[in] u U のサイズです。
        //! @param[in] v V のサイズです。
        //! @return 自身のオブジェクトの参照を返します。
        UVArg& SetUVSize( f32 u, f32 v ) { m_UVSize.Set( u, v ); return *this; }

        //! @brief 左下の点の UV 値を取得します。
        //! @return 左下の点の UV 値を返します。
        const nw::math::VEC2& GetUVSrc() const { return m_UVSrc; }

        //! @brief UV のサイズを取得します(右上点の UV - 左下点の UV)。
        //! @return UV のサイズを返します。
        const nw::math::VEC2& GetUVSize() const { return m_UVSize; }

    private:
        nw::math::VEC2 m_UVSrc;
        nw::math::VEC2 m_UVSize;
    };

    //! @brief Cube 描画用の引数クラスです。
    class CubeArg
    {
    public:
        //! @brief コンストラクタです。
        CubeArg()
           : m_Center ( nw::math::VEC3::Zero() ),
             m_Size ( 1.f, 1.f, 1.f )
            {}

        //! @brief 中心座標で位置を指定します。
        //! @param[in] vec 中心座標です。
        //! @return 自身のオブジェクトの参照を返します。
        CubeArg& SetCenter( const nw::math::VEC3& vec ) { m_Center = vec ; return *this; }

        //! @brief 中心座標で位置を指定します。
        //! @param[in] x 中心座標の x の値です。
        //! @param[in] y 中心座標の y の値です。
        //! @param[in] z 中心座標の z の値です。
        //! @return 自身のオブジェクトの参照を返します。
        CubeArg& SetCenter( f32 x, f32 y, f32 z ) { m_Center.Set( x, y, z ); return *this; }

        //! @brief サイズを指定します。
        //! @param[in] size サイズです。
        //! @return 自身のオブジェクトの参照を返します。
        CubeArg& SetSize( const nw::math::VEC3& size ) { m_Size = size; return *this; }

        //! @brief サイズを指定します。
        //! @param[in] x サイズの x の値です。
        //! @param[in] y サイズの y の値です。
        //! @param[in] z サイズの z の値です。
        //! @return 自身のオブジェクトの参照を返します。
        CubeArg& SetSize( f32 x, f32 y, f32 z ) { m_Size.Set( x, y, z ); return *this; }

        //! @brief 角の座標で位置とサイズを指定します。
        //! @param[in] vec 角の座標です。
        //! @param[in] size サイズです。
        //! @return 自身のオブジェクトの参照を返します。
        CubeArg& SetCornerAndSize( const nw::math::VEC3& vec, const nw::math::VEC3& size )
        {
            m_Center.Set( size.x * 0.5f + vec.x, size.y * 0.5f + vec.y, vec.z );
            m_Size = size;
            return *this;
        }

        //! @brief 角の座標で位置とサイズを指定します。
        //! @param[in] x 角の x 座標です。
        //! @param[in] y 角の y 座標です。
        //! @param[in] z 角の z 座標です。
        //! @param[in] size サイズです。
        //! @return 自身のオブジェクトの参照を返します。
        CubeArg& SetCornerAndSize( f32 x, f32 y, f32 z, const nw::math::VEC3& size )
        {
            return SetCornerAndSize( nw::math::VEC3( x, y, z ), size );
        }

        //! @brief 角の座標で位置とサイズを指定します。
        //! @param[in] vec 角の座標です。
        //! @param[in] w 幅です。
        //! @param[in] h 高さです。
        //! @param[in] d 奥行です。
        //! @return 自身のオブジェクトの参照を返します。
        CubeArg& SetCornerAndSize( const nw::math::VEC3& vec, f32 w, f32 h, f32 d )
        {
            return SetCornerAndSize( vec, nw::math::VEC3( w, h, d ) );
        }

        //! @brief 角の座標で位置とサイズを指定します。
        //! @param[in] x 角の x 座標です。
        //! @param[in] y 角の y 座標です。
        //! @param[in] z 角の z 座標です。
        //! @param[in] w 幅です。
        //! @param[in] h 高さです。
        //! @param[in] d 奥行です。
        //! @return 自身のオブジェクトの参照を返します。
        CubeArg& SetCornerAndSize( f32 x, f32 y, f32 z, f32 w, f32 h, f32 d )
        {
            return SetCornerAndSize( nw::math::VEC3( x, y, z ), nw::math::VEC3( w, h, d ) );
        }

        //! @brief 手前側の色と奥側の色を設定します。
        //! @param[in] colorFront 手前側の色です。
        //! @param[in] colorBack 奥側の色です。
        //! @return 自身のオブジェクトの参照を返します。
        CubeArg& SetColor( const nw::ut::Color4u8& colorFront, const nw::ut::Color4u8& colorBack )
        {
            m_Color0 = colorFront;
            m_Color1 = colorBack;
            return *this;
        }

        //! @brief 単色を設定します。
        //! @param[in] color 色です。
        //! @return 自身のオブジェクトの参照を返します。
        CubeArg& SetColor( const nw::ut::Color4u8& color ) { return SetColor( color, color ); }

        //! @brief 中心座標を取得します。
        //! @return 中心座標を返します。
        const nw::math::VEC3& GetCenter() const { return m_Center; }

        //! @brief サイズを取得します。
        //! @return サイズを返します。
        const nw::math::VEC3& GetSize() const { return m_Size; }

        //! @brief 色 0 (手前)を取得します。
        //! @return 色 0 を返します。
        const nw::ut::Color4u8& GetColor0() const { return m_Color0; }

        //! @brief 色 1 (奥)を取得します。
        //! @return 色 1 を返します。
        const nw::ut::Color4u8& GetColor1() const { return m_Color1; }

    private:
        nw::math::VEC3 m_Center;
        nw::math::VEC3 m_Size;
        nw::ut::Color4u8  m_Color0;
        nw::ut::Color4u8  m_Color1;
    };

    //  @}


    //---------------------------------------------------------------------------
    //! @brief        インスタンスを取得します。
    //!
    //! @return       シングルトンのインスタンスを返します。
    //---------------------------------------------------------------------------
    static PrimitiveRenderer* GetInstance()
    {
        static PrimitiveRenderer s_Instance;
        return &s_Instance;
    }

    //---------------------------------------------------------------------------
    //! @brief        シェーダバイナリから初期化を行います。
    //!
    //! @param[in]    allocator   アロケーターです。
    //! @param[in]    binaryData  シェーダバイナリへのポインタです。
    //! @param[in]    binarySize  シェーダバイナリのサイズです。
    //---------------------------------------------------------------------------
    void InitializeFromBinary( nw::ut::IAllocator* allocator, const void* binaryData = NULL, u32 binarySize = 0 );

    //---------------------------------------------------------------------------
    //! @brief        初期化を行います。
    //!
    //! @param[in]    allocator   アロケーターです。
    //! @param[in]    shaderPath  シェーダパスです。
    //---------------------------------------------------------------------------
    void Initialize( nw::ut::IAllocator* allocator, const char* shaderPath = NULL );

    //---------------------------------------------------------------------------
    //! @brief        終了処理を行います。
    //!
    //! @param[in]    allocator   アロケーターです。
    //---------------------------------------------------------------------------
    void Finalize( nw::ut::IAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief        頂点バッファをバインドします。
    //!
    //! @param[in]    buf         バッファの構造体です。
    //---------------------------------------------------------------------------
    void CreateVertexBuffer( DrawBuffer& buf );


    //----------------------------------------
    //! @name 行列設定／取得
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        PrimitiveRenderer で使うビュー行列を設定します。
    //!
    //! @param[in]    viewMtx     設定するビュー行列です。
    //---------------------------------------------------------------------------
    void SetViewMtx( const nw::math::MTX34* viewMtx );

    //---------------------------------------------------------------------------
    //! @brief        PrimitiveRenderer で使うプロジェクション行列を設定します。
    //!
    //! @param[in]    projectionMtx   設定するプロジェクション行列です。
    //---------------------------------------------------------------------------
    void SetProjectionMtx( const nw::math::MTX44* projectionMtx );

    //---------------------------------------------------------------------------
    //! @brief        PrimitiveRenderer で使うビューポートを設定します。
    //!
    //! @param[in]    x           設定する x 座標です。
    //! @param[in]    y           設定する y 座標です。
    //! @param[in]    width       設定する幅です。
    //! @param[in]    height      設定する高さです。
    //---------------------------------------------------------------------------
    void SetViewport( const s32 x, const s32 y, const s32 width, const s32 height );

    //---------------------------------------------------------------------------
    //! @brief        以降の描画でモデルに掛ける変換行列を設定します。
    //!
    //! @param[in]    modelToWorldMtx   設定する変換行列です。
    //---------------------------------------------------------------------------
    void SetModelMatrix( const nw::math::MTX34& modelToWorldMtx )
    {
        m_ModelMtx = modelToWorldMtx;
    }

    //---------------------------------------------------------------------------
    //! @brief        モデルに掛ける変換行列を取得します。
    //!
    //! @return       モデルに掛ける変換行列を返します。
    //---------------------------------------------------------------------------
    const nw::math::MTX34& GetModelMatrix() { return m_ModelMtx; }

    //  @}


    //----------------------------------------
    //! @name 描画
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        描画に必要な設定を行います。
    //---------------------------------------------------------------------------
    void Begin();

    //---------------------------------------------------------------------------
    //! @brief        描画を終了します。
    //---------------------------------------------------------------------------
    void End();

    //---------------------------------------------------------------------------
    //! @brief        平面の三角形を描画します。
    //!
    //! @param[in]    color   頂点色です。
    //---------------------------------------------------------------------------
    void DrawTriangle( const nw::ut::Color4u8& color );

    //---------------------------------------------------------------------------
    //! @brief        バッファを指定して三角形を描画します。
    //!
    //! @param[in]    color       頂点色です。
    //! @param[in]    buf         バッファの構造体です。
    //---------------------------------------------------------------------------
    void DrawTriangle( const nw::ut::Color4u8& color, const DrawBuffer& buf );

    //---------------------------------------------------------------------------
    //! @brief        平面の四角形を描画します。
    //!
    //! @param[in]    colorLeft   左側の点の頂点色です。
    //! @param[in]    colorRight  右側の点の頂点色です。
    //---------------------------------------------------------------------------
    void DrawQuad( const nw::ut::Color4u8& colorLeft, const nw::ut::Color4u8& colorRight );

    //---------------------------------------------------------------------------
    //! @brief        単色の平面の四角形を描画します。
    //!
    //! @param[in]    color       頂点色です。
    //---------------------------------------------------------------------------
    void DrawQuad( const nw::ut::Color4u8& color ) { DrawQuad( color, color ); }

    //---------------------------------------------------------------------------
    //! @brief        テクスチャを張り込んだ平面の四角形を描画します。
    //!
    //! @param[in]    texture     貼り付けるテクスチャです。
    //! @param[in]    colorLeft   左側の点の頂点色です。
    //! @param[in]    colorRight  右側の点の頂点色です。
    //! @param[in]    uvSrc       左下の点の UV 値です。
    //! @param[in]    uvSize      UV 値の範囲です(右上の点の UV 値 - 左下の点の UV 値)。
    //---------------------------------------------------------------------------
    void DrawQuad(
        const nw::gfnd::Texture& texture,
        const nw::ut::Color4u8& colorLeft,
        const nw::ut::Color4u8& colorRight,
        const nw::math::VEC2& uvSrc = nw::math::VEC2::Zero(),
        const nw::math::VEC2& uvSize = nw::math::VEC2( 1.f, 1.f )
    );

    //---------------------------------------------------------------------------
    //! @brief        テクスチャを張り込んだ単色の平面の四角形を描画します。
    //!
    //! @param[in]    texture     貼り付けるテクスチャです。
    //! @param[in]    color       頂点色です。
    //! @param[in]    uvSrc       左下の点の UV 値です。
    //! @param[in]    uvSize      UV 値の範囲です(右上の点の UV 値 - 左下の点の UV 値)。
    //---------------------------------------------------------------------------
    void DrawQuad(
        const nw::gfnd::Texture& texture,
        const nw::ut::Color4u8& color = nw::ut::Color4u8(),
        const nw::math::VEC2& uvSrc = nw::math::VEC2::Zero(),
        const nw::math::VEC2& uvSize = nw::math::VEC2( 1.f, 1.f )
    )
    {
        DrawQuad( texture, color, color, uvSrc, uvSize );
    }

    //---------------------------------------------------------------------------
    //! @brief        平面の四角形を引数に設定した条件で描画します。
    //!
    //! @param[in]    arg         Quad 描画用の引数です。
    //---------------------------------------------------------------------------
    void DrawQuad( const nw::dev::PrimitiveRenderer::QuadArg& arg );

    //---------------------------------------------------------------------------
    //! @brief        テクスチャを張り込んだ平面の四角形を引数に設定した条件で描画します。
    //!
    //! @param[in]    texture     貼り付けるテクスチャです。
    //! @param[in]    arg         Quad 描画用の引数です。
    //! @param[in]    uvArg       UV 値指定用の引数です。
    //---------------------------------------------------------------------------
    void DrawQuad(
        const nw::gfnd::Texture& texture,
        const nw::dev::PrimitiveRenderer::QuadArg& arg,
        const nw::dev::PrimitiveRenderer::UVArg& uvArg = nw::dev::PrimitiveRenderer::UVArg()
    );

    //---------------------------------------------------------------------------
    //! @brief        四角形の枠線を描画します。
    //!
    //! @param[in]    colorLeft   左側の点の頂点色です。
    //! @param[in]    colorRight  右側の点の頂点色です。
    //---------------------------------------------------------------------------
    void DrawBox( const nw::ut::Color4u8& colorLeft, const nw::ut::Color4u8& colorRight );

    //---------------------------------------------------------------------------
    //! @brief        単色の四角形の枠線を描画します。
    //!
    //! @param[in]    color       頂点色です。
    //---------------------------------------------------------------------------
    void DrawBox( const nw::ut::Color4u8& color ) { DrawBox( color, color ); }

    //---------------------------------------------------------------------------
    //! @brief        四角形の枠線を引数に設定した条件で描画します。
    //!
    //! @param[in]    arg         Quad 描画用の引数です。
    //---------------------------------------------------------------------------
    void DrawBox( const nw::dev::PrimitiveRenderer::QuadArg& arg );

    //---------------------------------------------------------------------------
    //! @brief        直方体を描画します。
    //!
    //! @param[in]    color0      手前左上の頂点色です。
    //! @param[in]    color1      奥右下の頂点色です。
    //---------------------------------------------------------------------------
    void DrawCube( const nw::ut::Color4u8& color0, const nw::ut::Color4u8& color1 );

    //---------------------------------------------------------------------------
    //! @brief        単色の直方体を描画します。
    //!
    //! @param[in]    color       頂点色です。
    //---------------------------------------------------------------------------
    void DrawCube( const nw::ut::Color4u8& color ) { DrawCube( color, color ); }

    //---------------------------------------------------------------------------
    //! @brief        直方体を引数に設定した条件で描画します。
    //!
    //! @param[in]    arg         Cube 描画用の引数です。
    //---------------------------------------------------------------------------
    void DrawCube( const nw::dev::PrimitiveRenderer::CubeArg& arg );

    //---------------------------------------------------------------------------
    //! @brief        ワイヤーフレームの直方体を描画します。
    //!
    //! @param[in]    color0      手前左上の頂点色です。
    //! @param[in]    color1      奥右下の頂点色です。
    //---------------------------------------------------------------------------
    void DrawWireCube( const nw::ut::Color4u8& color0, const nw::ut::Color4u8& color1 );

    //--------------------------------------------------------------------------
    //! @brief        ワイヤーフレームで単色の直方体を描画します。
    //!
    //! @param[in]    color       頂点色です。
    //---------------------------------------------------------------------------
    void DrawWireCube( const nw::ut::Color4u8& color ) { DrawWireCube( color, color ); }

    //---------------------------------------------------------------------------
    //! @brief        ワイヤーフレームの直方体を引数に設定した条件で描画します。
    //!
    //! @param[in]    arg         Cube 描画用の引数です。
    //---------------------------------------------------------------------------
    void DrawWireCube( const nw::dev::PrimitiveRenderer::CubeArg& arg );

    //---------------------------------------------------------------------------
    //! @brief        線分を描画します。
    //!
    //! @param[in]    color0      1 つ目の頂点色です。
    //! @param[in]    color1      2 つ目の頂点色です。
    //! @param[in]    lineWidth   線分の幅です。
    //---------------------------------------------------------------------------
    void DrawLine( const nw::ut::Color4u8& color0, const nw::ut::Color4u8& color1, const f32 lineWidth = 1.f );

    //---------------------------------------------------------------------------
    //! @brief        単色の線分を描画します。
    //!
    //! @param[in]    color       線の色です。
    //! @param[in]    lineWidth   線分の幅です。
    //---------------------------------------------------------------------------
    void DrawLine( const nw::ut::Color4u8& color, const f32 lineWidth = 1.f ) { DrawLine( color, color, lineWidth ); }

    //---------------------------------------------------------------------------
    //! @brief        from と to を結ぶ線分を描画します。
    //!
    //! @param[in]    from        線分の始点です。
    //! @param[in]    to          線分の終点です。
    //! @param[in]    color0      始点の頂点色です。
    //! @param[in]    color1      終点の頂点色です。
    //! @param[in]    lineWidth   線分の幅です。
    //---------------------------------------------------------------------------
    void DrawLine(
        const nw::math::VEC3& from,
        const nw::math::VEC3& to,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1,
        const f32 lineWidth = 1.f
    );

    //---------------------------------------------------------------------------
    //! @brief        from と to を結ぶ線分を描画します。
    //!
    //! @param[in]    from        線分の始点です。
    //! @param[in]    to          線分の終点です。
    //! @param[in]    color       頂点色です。
    //! @param[in]    lineWidth   線分の幅です。
    //---------------------------------------------------------------------------
    void DrawLine(
        const nw::math::VEC3& from,
        const nw::math::VEC3& to,
        const nw::ut::Color4u8& color,
        const f32 lineWidth = 1.f
    )
    {
        DrawLine( from, to, color, color, lineWidth );
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファを指定して線分を描画します。
    //!
    //! @param[in]    color       頂点色です。
    //! @param[in]    buf         バッファの構造体です。
    //! @param[in]    lineWidth   線分の幅です。
    //---------------------------------------------------------------------------
    void DrawLine( const nw::ut::Color4u8& color, const DrawBuffer& buf, const f32 lineWidth = 1.f );

    //---------------------------------------------------------------------------
    //! @brief        4 x 8 分割、直径 1 の球を描画します。
    //!
    //! @param[in]    north       北極(y が + 側)の頂点色です。
    //! @param[in]    south       南極(y が - 側)の頂点色です。
    //---------------------------------------------------------------------------
    void DrawSphere4x8( const nw::ut::Color4u8& north, const nw::ut::Color4u8& south );

    //---------------------------------------------------------------------------
    //! @brief        4 x 8 分割、直径 1 の単色の球を描画します。
    //!
    //! @param[in]    color       球の色です。
    //---------------------------------------------------------------------------
    void DrawSphere4x8( const nw::ut::Color4u8& color ) { DrawSphere4x8( color, color ); }

    //---------------------------------------------------------------------------
    //! @brief        指定した位置に 4 x 8 の球を描画します。
    //!
    //! @param[in]    pos         中心点です。
    //! @param[in]    radius      半径です。
    //! @param[in]    north       北極側の色です。
    //! @param[in]    south       南極側の色です。
    //---------------------------------------------------------------------------
    void DrawSphere4x8(
        const nw::math::VEC3& pos,
        f32 radius,
        const nw::ut::Color4u8& north,
        const nw::ut::Color4u8& south
    );

    //---------------------------------------------------------------------------
    //! @brief        指定した位置に単色の 4 x 8 の球を描画します。
    //!
    //! @param[in]    pos         中心点です。
    //! @param[in]    radius      半径です。
    //! @param[in]    color       色です。
    //---------------------------------------------------------------------------
    void DrawSphere4x8(
        const nw::math::VEC3& pos,
        f32 radius,
        const nw::ut::Color4u8& color
    )
    {
        DrawSphere4x8( pos, radius, color, color );
    }

    //---------------------------------------------------------------------------
    //! @brief        8 x 16 分割、直径 1 の球を描画します。
    //!
    //! @param[in]    north       北極(y が + 側)の頂点色です。
    //! @param[in]    south       南極(y が - 側)の頂点色です。
    //---------------------------------------------------------------------------
    void DrawSphere8x16( const nw::ut::Color4u8& north, const nw::ut::Color4u8& south );

    //---------------------------------------------------------------------------
    //! @brief        8 x 16 分割、直径 1 の単色の球を描画します。
    //!
    //! @param[in]    color       球の色です。
    //---------------------------------------------------------------------------
    void DawSphere8x16( const nw::ut::Color4u8& color ) { DrawSphere8x16( color, color ); }

    //---------------------------------------------------------------------------
    //! @brief        指定した位置に 8 x 16 の球を描画します。
    //!
    //! @param[in]    pos         中心点です。
    //! @param[in]    radius      半径です。
    //! @param[in]    north       北極側の色です。
    //! @param[in]    south       南極側の色です。
    //---------------------------------------------------------------------------
    void DrawSphere8x16(
        const nw::math::VEC3& pos,
        f32 radius,
        const nw::ut::Color4u8& north,
        const nw::ut::Color4u8& south
    );

    //---------------------------------------------------------------------------
    //! @brief        指定した位置に単色の 8 x 16 の球を描画します。
    //!
    //! @param[in]    pos         中心点です。
    //! @param[in]    radius      半径です。
    //! @param[in]    color       色です。
    //---------------------------------------------------------------------------
    void DrawSphere8x16(
        const nw::math::VEC3& pos,
        f32 radius,
        const nw::ut::Color4u8& color
    )
    {
        DrawSphere8x16( pos, radius, color, color );
    }

    //---------------------------------------------------------------------------
    //! @brief        16 分割、直径 1 の円を描画します。
    //!
    //! @param[in]    color       円の色です。
    //---------------------------------------------------------------------------
    void DrawCircle16( const nw::ut::Color4u8& color );

    //---------------------------------------------------------------------------
    //! @brief        指定した位置に 16 分割の円を描画します。
    //!
    //! @param[in]    pos         中心点です。
    //! @param[in]    radius      半径です。
    //! @param[in]    color       色です。
    //---------------------------------------------------------------------------
    void DrawCircle16( const nw::math::VEC3& pos, f32 radius, const nw::ut::Color4u8& color );

    //---------------------------------------------------------------------------
    //! @brief        32 分割、直径 1 の円を描画します。
    //!
    //! @param[in]    color       円の色です。
    //---------------------------------------------------------------------------
    void DrawCircle32( const nw::ut::Color4u8& color );

    //---------------------------------------------------------------------------
    //! @brief        指定した位置に 32 分割の円を描画します。
    //!
    //! @param[in]    pos         中心点です。
    //! @param[in]    radius      半径です。
    //! @param[in]    color       色です。
    //---------------------------------------------------------------------------
    void DrawCircle32( const nw::math::VEC3& pos, f32 radius, const nw::ut::Color4u8& color );

    //  @}

protected:
    //! @brief コンストラクタです。
    PrimitiveRenderer()
      : m_RendererImpl( NULL ),
        m_ModelMtx( nw::math::MTX34::Identity() )
    {
        if (s_pInstance != NULL)
        {
            NW_FATAL_ERROR("nw::dev::PrimitiveRenderer instance already exists.");
        }
        s_pInstance = this;
    }

    //! @brief デストラクタです。
    virtual ~PrimitiveRenderer() {}

private:
    //! @brief インスタンスの生成などを行います。
    void Prepare( nw::ut::IAllocator* allocator );

    nw::dev::PrimitiveRendererBase* m_RendererImpl; //!< グラフィックスエンジン毎の具体的な実装です。
    nw::math::MTX34 m_ModelMtx;                     //!< モデル行列です。

    static PrimitiveRenderer* s_pInstance;  //!< シングルトンオブジェクトへのポインタです。
};

} // namespace dev
} // namespace nw

#endif  // NW_DEV_PRIMITIVE_RENDERER_H_

