﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if defined(NW_PLATFORM_CAFE)
#include <cafe/gx2.h>
#endif

namespace nw
{
namespace dev
{

//------------------------------------------------------------------------------
template<s32 N>
void
MultiGPUMeter<N>::Initialize(
#if defined(NW_PLATFORM_CAFE)
    u64* gpuCountersBuf
#endif
)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL( gpuCountersBuf );
    m_GpuCounters = gpuCountersBuf;

#else
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        glGenQueries( N * 2, m_GpuQueries );
        NW_GL_ASSERT();
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
#endif
}

//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
template<s32 N>
u32
MultiGPUMeter<N>::GetBufferSize()
{
    return nw::ut::RoundUp(sizeof(u64) * N * 2, GX2_DEFAULT_BUFFER_ALIGNMENT);
}
#endif

//------------------------------------------------------------------------------
template<s32 N>
void
MultiGPUMeter<N>::BeginMeasure( const nw::ut::Color4u8& color )
{
#if defined(NW_PLATFORM_CAFE)
    GX2SampleTopGPUCycle( &m_GpuCounters[ this->m_SectionNum[ this->m_CurrentBuffer ] * 2 + 0 ] );

#else
    glGetInteger64v( GL_TIMESTAMP, &frameBeginTimeGL[ m_SectionNum[ m_CurrentBuffer ] ] );
    glQueryCounter( m_GpuQueries[ m_SectionNum[ m_CurrentBuffer ] * 2 + 0 ], GL_TIMESTAMP );
#endif

    LoadMeterBase::SetBeginMeasure( nw::ut::Tick::GetSystemCurrent(), color );
}

//------------------------------------------------------------------------------
template<s32 N>
void
MultiGPUMeter<N>::EndMeasure()
{
#if defined(NW_PLATFORM_CAFE)
    if ( this->m_OverNum == 0 )
    {
        GX2SampleBottomGPUCycle( &m_GpuCounters[ this->m_TopSection * 2 + 1 ] );
    }

#else
    glQueryCounter( m_GpuQueries[ m_TopSection * 2 + 1 ], GL_TIMESTAMP );
#endif

    // GPU 計測は OnEndFrame() 時に更新するので Tick 値は 0 でセクションを終了します。
    LoadMeterBase::SetEndMeasure( nw::ut::Tick() );
}

//------------------------------------------------------------------------------
template<s32 N>
void
MultiGPUMeter<N>::OnEndFrame()
{
    // フレーム終了時に GPU 計測結果を更新します。

#if defined(NW_PLATFORM_CAFE)
    DCInvalidateRange( m_GpuCounters, sizeof(u64) * N * 2 );

    for ( int i = 0; i < this->m_SectionNum[ this->m_CurrentBuffer ]; ++i )
    {
        LoadMeterBase::Section* section = &this->m_SectionArray[this->m_CurrentBuffer].at( i );
        section->begin = nw::ut::Tick( static_cast<s64>( GX2GPUTimeToCPUTime( m_GpuCounters[ i * 2 + 0 ] ) ) );
        section->end = nw::ut::Tick( static_cast<s64>( GX2GPUTimeToCPUTime( m_GpuCounters[ i * 2 + 1 ] ) ) );
        this->m_FinalEnd[ this->m_CurrentBuffer ] = section->end;
    }

#else
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        for ( int i = 0; i < m_SectionNum[ m_CurrentBuffer ]; ++i )
        {
            GLuint64 start;
            GLuint64 end;

            glGetQueryObjectui64v( m_GpuQueries[ i * 2 + 0 ], GL_QUERY_RESULT, &start );
            glGetQueryObjectui64v( m_GpuQueries[ i * 2 + 1 ], GL_QUERY_RESULT, &end );
            NW_GL_ASSERT();

            Section* section = &m_SectionArray[m_CurrentBuffer].at( i );
            nw::ut::Tick begin = section->begin;
            section->begin = begin + nw::ut::TimeSpan::FromNanoSeconds( start - frameBeginTimeGL[i] );
            section->end = section->begin +  nw::ut::TimeSpan::FromNanoSeconds( end - start );
            m_FinalEnd[ m_CurrentBuffer ] = section->end;
        }
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
#endif

    LoadMeterBase::OnEndFrame();
}

} // namespace dev
} // namespace nw

