﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ウェブページ / オフライン HTML で開いたページとのメッセージを通信する為の API 宣言
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/web/web_SessionTypes.h>

namespace nn { namespace web {

/**
 * @brief ウェブページ / オフライン HTML とセッションを構築する為のクラスです。
 */
class Session
{
public:

    //------------------------------------------------------------------------
    /**
     * @brief 初期化に必要なメモリサイズを返します。
     *
     * @return 初期化に必要なメモリサイズ
     */
    static size_t GetWorkBufferSize() NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief Session を利用するのに必要な最低限の情報を持ったクラスを作成します。
     */
    Session() NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 初期化します。
     *
     * @param[in] pWorkBuffer       SessionHandle 初期化に利用するワークバッファへのポインタ
     * @param[in] workBufferSize    SessionHandle 初期化に利用するワークバッファのサイズ
     *
     * @pre
     * - workBufferSize == GetWorkBufferSize()
     * - pWorkBuffer が 4 バイトアライメントされている
     *
     */
    void Initialize(void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 終了します。
     *
     * @pre
     *  - initialize() 済み
     */
    void Finalize() NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief Sessionの状態を取得します。
     *
     * @return 有効なら true を返します。
     */
    bool IsValid() const NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief メッセージを受信します。
     *
     * @pre
     *  - Initialize() 済み
     *
     * @return 受信に成功したら true を返します。
     *
     * @details
     * 毎フレーム呼び出してください。
     * １度の呼び出して１メッセージを受信します。
     * 複数のメッセージを受信したい場合は false が返る迄呼び出してください。
     */
    bool TryReciveMessage(SessionMessage* pOutMessage) NN_NOEXCEPT;

    //! @brief 内部用機能のため使用禁止です。
    void* Impl() NN_NOEXCEPT;

private:
    /**
     * @internal
     */
    void* m_pImpl;
};

}} // namespace nn::web
