﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief オフライン HTML を開くのに利用される引数の宣言(非公開)
 */

#pragma once

#include <nn/nn_ApplicationId.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet.h>
#include <nn/account/account_Types.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/web/web_OfflineTypes.h>
#include <nn/web/web_OfflineTypesPrivate.h>
#include <nn/web/web_OfflineShowArg.h>

namespace nn { namespace web {

/**
 * @brief アプリケーションの「ソフトリーガル情報」を開くための情報を構築するクラスです。
 */
class ShowApplicationLegalInformationPageArg
{
public:
    /**
     * @brief NULL 終端を含むドキュメントパスの最大バイト数です。
     */
    static const size_t DocumentPathMaxLength = 3072;

    /**
     * @brief アプリケーションの「ソフトリーガル情報」を開く為の最低限の情報を持つクラスを作成します。
     *
     * @param[in]   id              ApplicationId を指定してください。
     * @param[in]   pDocumentPath  ドキュメントパスを指定してください。
     *
     * @details pDocumentPathは NULL 終端されている必要があります。
     *          DocumentPathMaxLength -1 文字以下にする必要があります。
     */
    explicit ShowApplicationLegalInformationPageArg(const nn::ApplicationId& id,
        const char* pDocumentPath) NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief   LoginPage を開く為の変数を一つのストレージにまとめ、ライブラリアプレットの入力チャンネルにプッシュします。
     *
     * @param[in]   libraryAppletHandle     対象のライブラリアプレットのハンドルを指定します。
     */
    void PushToInChannel(nn::applet::LibraryAppletHandle handle) const NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief   対象とするフッターボタンが起動時に表示されるかどうかを設定します。
     *
     * @param[in]   buttonId     対象のフッターボタンの ID を指定します。
     * @param[in]   visible      対象のフッターボタンを起動時に表示するかどうかを指定します。
     *
     * @details デフォルト値は A ボタン、 B ボタンが true で、それ以外は false です。
     *          同じフッターボタンに対して二回呼び出した場合、そのフッターボタンに対する設定は上書きされます。
     */
    void SetBootFooterButtonVisible(const OfflineFooterButtonId buttonId, const bool visible) NN_NOEXCEPT;

protected:
    /**
     * @internal
     * @brief 情報を格納するために使用するデータのサイズ(バイト)です。
     */
    static const size_t DataSize = 8192;

    /**
     * @internal
     */
    NN_ALIGNAS(4) uint8_t m_Data[DataSize];

private:
    /**
     * @internal
     */
    ShowApplicationLegalInformationPageArg() NN_NOEXCEPT;
};

//------------------------------------------------------------------------
/**
 * @brief 「システムデータ」を開くための情報を構築するクラスです。
 */
class ShowSystemDataPageArg
{
public:
    /**
     * @brief NULL 終端を含むドキュメントパスの最大バイト数です。
     */
    static const size_t DocumentPathMaxLength = 3072;

    /**
     * @brief システムデータを開く為の最低限の情報を持つクラスを作成します。
     *
     * @param[in]   id              SystemDataId を指定ください。
     * @param[in]   pDocumentPath   ドキュメントパスを指定してください。
     *
     * @details pDocumentPath は NULL 終端されている必要があります。
     *          DocumentPathMaxLength -1 文字以下にする必要があります。
     */
    explicit ShowSystemDataPageArg(const nn::ncm::SystemDataId& id,
        const char* pDocumentPath) NN_NOEXCEPT;

    /**
     * @brief フッター表示のデフォルトの制御方法を設定します。
     *
     * @param[in]   offlineFooterFixedKind  フッター表示の制御方法を指定してください。
     *
     * @details デフォルト値は OfflineFooterFixedKind_Auto です。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetFooterFixedKind(const OfflineFooterFixedKind offlineFooterFixedKind) NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief   「システムデータ」を開く際に必要なデータを一つのストレージにまとめ、ライブラリアプレットの入力チャンネルにプッシュします。
     *
     * @param[in]   libraryAppletHandle     対象のライブラリアプレットのハンドルを指定します。
     *
     */
    void PushToInChannel(nn::applet::LibraryAppletHandle handle) const NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief   対象とするフッターボタンが起動時に表示されるかどうかを設定します。
     *
     * @param[in]   buttonId     対象のフッターボタンの ID を指定します。
     * @param[in]   visible      対象のフッターボタンを起動時に表示するかどうかを指定します。
     *
     * @details デフォルト値は A ボタン、 B ボタンが true で、それ以外は false です。
     *          同じフッターボタンに対して二回呼び出した場合、そのフッターボタンに対する設定は上書きされます。
     */
    void SetBootFooterButtonVisible(const OfflineFooterButtonId buttonId, const bool visible) NN_NOEXCEPT;

protected:
    /**
     * @internal
     * @brief 情報を格納するために使用するデータのサイズ(バイト)です。
     */
    static const size_t DataSize = 8192;

    /**
     * @internal
     */
    NN_ALIGNAS(4) uint8_t m_Data[DataSize];

private:
    /**
     * @internal
     */
    ShowSystemDataPageArg() NN_NOEXCEPT;
};

}} // namespace nn::web
