﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ロビーページを開くのに利用される引数の宣言(非公開)
 */

#pragma once

#include <nn/account/account_Types.h>
#include <nn/web/web_LobbyTypesPrivate.h>

namespace nn { namespace web {

/**
 * @brief ロビーページを開くための情報を構築するクラスです。
 */
class ShowLobbyPageArg
{
public:
    /**
     * @brief NULL 終端を含めた追加パラメータの最大バイト数です。
     */
    static const size_t LobbyParameterMaxLength = 256;

    /**
     * @brief ロビーページを開くための最低限の情報を持つクラスを作成します。
     */
    explicit ShowLobbyPageArg() NN_NOEXCEPT;

    /**
     * @brief Uid を設定します。
     *
     * @param[in]   uid アカウントの Uid を指定してください。
     *
     * @details 指定する値は、アカウントシステムに登録されているユーザーの Uid である必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetUid(const nn::account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief リクエスト URL に付加する追加パラメータを設定します。
     *
     * @param[in]   pLobbyParameter 追加パラメータを指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          LobbyParameterMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetLobbyParameter(const char* pLobbyParameter) NN_NOEXCEPT;

protected:
    /**
     * @internal
     * @brief 情報を格納するために使用するデータのサイズ(バイト)です。
     */
    static const size_t DataSize = 8192;

    /**
     * @internal
     */
    NN_ALIGNAS(4) uint8_t m_Data[DataSize];
};

//--------------------------------------------------------------------------
/**
 * @brief ロビーページを開いた結果の情報を提供するクラスです。
 */
class LobbyPageReturnValue
{
public:
    /**
     * @brief NULL 終端を含む最後に開いていたページの URL の最大バイト数です。
     */
    static const size_t LastUrlMaxLength = 4096;

    /**
     * @brief 有効な情報を持たないクラスを作成します。
     */
    LobbyPageReturnValue() NN_NOEXCEPT;

    /**
     * @brief 終了理由を返します。
     *
     * @return 終了理由を表す定数(@ref LobbyExitReason)
     */
    LobbyExitReason GetLobbyExitReason() const NN_NOEXCEPT;

    /**
     * @brief 最後に開いていたページの URL を取得します。
     *
     * @return ページの URL を返します。
     *
     * @pre
     *     GetLobbyExitReason() == LobbyExitReason_CallbackUrlReached
     */
    const char* GetLastUrl() const NN_NOEXCEPT;

    /**
     * @brief 最後に開いていたページの URL のバイト数を取得します。
     *
     * @return URL のサイズを返します。
     *
     * @pre
     *     GetLobbyExitReason() == LobbyExitReason_CallbackUrlReached
     */
    size_t GetLastUrlSize() const NN_NOEXCEPT;

protected:
    /**
     * @internal
     */
    uint64_t m_ExitReason;
    /**
     * @internal
     */
    char m_LastUrl[LastUrlMaxLength];
    /**
     * @internal
     */
    uint64_t m_LastUrlSize;
};

}} // namespace nn::web
