﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ウェブページを開くのに利用する API 宣言(内部開発者向け)
 */

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/web/web_ShowArgPrivate.h>

namespace nn { namespace web { namespace detail {

//--------------------------------------------------------------------------
/**
 * @brief ウェブページを開くための情報を書き込む為のクラスです。
 */
class ShowWebPageArgWriter
    : public ShowWebPageArg
{
public:
    /**
     * @internal
     *
     * @brief   LoginPage を開く為の変数を一つのストレージにまとめ、ライブラリアプレットの入力チャンネルにプッシュします。
     *
     * @param[in]   libraryAppletHandle     対象のライブラリアプレットのハンドルを指定します。
     */
    void PushToInChannel(nn::applet::LibraryAppletHandle handle) const NN_NOEXCEPT;
};

//------------------------------------------------------------------------
/**
 * @brief ウェブページを開いた戻り値を設定するクラスです。
 */
class WebPageReturnValueDetail
    : protected WebPageReturnValue
{
public:
    /**
     * @brief 終了理由を設定します。
     *
     * @param[in]   webExitReason 終了理由を指定してください。
     */
    inline void SetWebExitReason(WebExitReason webExitReason) NN_NOEXCEPT;

    /**
     * @brief 終了時開いていたページの URL を設定します。
     *
     * @param[in]   lastUrl 終了時開いていたページの URL を指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          LastUrlMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    inline void SetLastUrl(const char* lastUrl) NN_NOEXCEPT;
};

//------------------------------------------------------------------------
void WebPageReturnValueDetail::SetWebExitReason(WebExitReason webExitReason) NN_NOEXCEPT
{
    m_ExitReason = static_cast<uint64_t>(webExitReason);
}

//------------------------------------------------------------------------
void WebPageReturnValueDetail::SetLastUrl(const char* lastUrl) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(lastUrl);
    size_t size = std::strlen(lastUrl) + 1;
    NN_SDK_ASSERT(size <= LastUrlMaxLength);

    std::memcpy(m_LastUrl, lastUrl, size);

    m_LastUrlSize = static_cast<uint64_t>(size);
}

}}} // namespace nn::web::detail
