﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/nn_Result.h>

namespace nn { namespace rid {

    class NetworkConnection;

    /**
     * @brief    無効な ApplicationId です。
     */
    const ncm::ApplicationId InvalidApplicationId = ncm::ApplicationId::GetInvalidId();

    /**
     * @brief    アプリケーションと Asset の対応を表すクラスです。
     *           アプリケーションを持たないコンテンツの場合、demoApplicationId に rid::InvalidApplicationId を指定してください。
     */
    struct ApplicationAsset
    {
        int index;
        ncm::ApplicationId demoApplicationId;
        ncm::ApplicationId aocApplicationId;
    };

    /**
     * @brief    コンテンツのダウンロード結果を表すクラスです。
     */
    struct DownloadResult
    {
        ncm::ApplicationId applicationId;
        Result result;
    };

    /**
     * @brief    アプリケーション更新の進行状況です。
     */
    struct ApplicationUpdateProgress
    {
        enum class State
        {
            DoNothing,              //!< アプリケーション更新処理が始まっていない
            Downloading,            //!< アプリケーション更新をダウンロードしている
            Completed,              //!< アプリケーション更新が完了した
            Failed,                 //!< アプリケーション更新に失敗した
            Cancelled,              //!< アプリケーション更新がキャンセルされた
        }state;

        int64_t done;               //!< 終了したステップ数です。
        int64_t total;              //!< トータルのステップ数です。
    };

    /**
     * @brief    アプリケーション更新を実行する処理を行うクラスです。
     */
    class ApplicationUpdater
    {
    public:
        ApplicationUpdater() NN_NOEXCEPT;
        ~ApplicationUpdater() NN_NOEXCEPT {};

        /**
         * @brief        アプリケーション更新を実行します。
         *
         * @details      アプリケーション更新に必要な処理を行います。
         *               assetList の配列に存在するアプリケーションと Asset はダウンロードと更新が行われます。
         *
         * @param[out]   outResultListCount          ダウンロード結果を格納したコンテンツの数
         * @param[out]   outResultList               ダウンロード結果の配列
         * @param[in]    resultListCount             outResultList の要素数
         * @param[in]    assetList                   ダウンロードを行うアプリケーションと Asset のインデックスの組の配列
         * @param[in]    count                       assetList の要素数
         * @param[in]    deletesUnlistedContents     assetList にないコンテンツの削除を行うか
         * @param[in]    requiredFreeSpaceSize       最低限残す本体ストレージの空き領域[Byte]
         *
         * @return       処理の結果が返ります。
         * @retval       ResultNetworkNotConnected                  ネットワークに接続されていないので更新に失敗しました。
         * @retval       ResultNotEnoughSpaceForDownloadApplication コンテンツのダウンロードに必要な容量が不足しています。
         * @retval       ResultDownloadApplicationFailed            ダウンロードに失敗したコンテンツがあります。
         * @retval       ResultNoRightsContentExists                権利が付与されていないコンテンツが存在します。
         * @retval       ResultDownloadApplicationCancelled         コンテンツのダウンロードがキャンセルされました。
         */
        Result Execute(int* outResultListCount, DownloadResult outResultList[], int resultListCount, ApplicationAsset assetList[], int assetListCount, bool deletesUnlistedContents, int64_t requiredFreeSpaceSize = 0) NN_NOEXCEPT;

        /**
         * @brief        アプリケーション更新をキャンセルします。
         */
        void Cancel() NN_NOEXCEPT;

        /**
         * @brief        アプリケーション更新の進行状況を取得します。
         *
         * @return       アプリケーション更新の進行状況が返ります。
         */
        const ApplicationUpdateProgress GetProgress() NN_NOEXCEPT;

        /**
        * @brief        コミットが必要なコンテンツがあるか判定します。
        *
        * @return       コミットが必要なコンテンツがある場合は true, そうでない場合は false が返ります。
        */
        bool IsCommitRequired() NN_NOEXCEPT;

    private:
        Result ExecuteImpl(int* outResultListCount, DownloadResult outResultList[], int resultListCount, ApplicationAsset assetList[], int count, bool deletesUnlistedContents, int64_t requiredFreeSpaceSize) NN_NOEXCEPT;
        void UpdateProgress(ApplicationUpdateProgress::State state, int64_t done = 0, int64_t total = 0) NN_NOEXCEPT;
        Result WaitDownload(ncm::ApplicationId id, NetworkConnection* connection) NN_NOEXCEPT;

        ApplicationUpdateProgress m_Progress;
        os::Mutex m_Mutex;
        bool m_IsCancelled;
        int64_t m_Current;
        int64_t m_Total;

        static const int MaxApplicationAocCount = 2048;
    };
}}
