﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "repair_CryptUtility.h"
#include "repair_FileSystem.h"
#include "repair_Types.h"
#include <nn/nn_Result.h>
#include <functional>

namespace nn
{
    namespace repair
    {
        struct AuthenticationPackage
        {
            static const int SIZE = 256;

            Sign2048 sign;
            uint8_t data[SIZE];
        };

        struct AuthenticationContent
        {
            Id128 sessionId;
            Key128 sessionKey;
        };

        struct BackupRequestMessage
        {
            Sign2048 sign;
            RsaEncryptedBlock sessionId;
            RsaEncryptedBlock sessionKey;
            char devIdHex[DeviceIdHexLength];

            static BackupRequestMessage MakeZero() { BackupRequestMessage ret = {}; return ret; }
        };

        struct BackupResponseMessage
        {
            Sign2048 sign;
            InitialVector iv;
            RsaEncryptedBlock sessionId;
            RsaEncryptedBlock sessionKey;
        };

        struct RestoreRequestMessage
        {
            Sign2048 sign;
            InitialVector iv;
            RsaEncryptedBlock sessionId;
            RsaEncryptedBlock sessionKey;
            char devIdHex[DeviceIdHexLength];

            static RestoreRequestMessage MakeZero() { RestoreRequestMessage ret = {}; return ret; }
        };

        struct RestoreResponseMessage
        {
            Sign2048 sign;
            RsaEncryptedBlock sessionId;
            RsaEncryptedBlock sessionKey;
        };

        struct BackupContext
        {
            Id128 sessionId;
            Key128 sessionKey;
            Key128 encryptedSessionKey;
        };

        struct AuthenticationArchiveContent
        {
            InitialVector iv;
            //Id128 encryptedSessionId;
            Key128 encryptedSessionKey;

            static AuthenticationArchiveContent MakeZero() { AuthenticationArchiveContent ret = {}; return ret; }
        };

        class IAuthenticationKeySource
        {
        public:
            virtual ~IAuthenticationKeySource() {};

            virtual nn::Result GetPrivateKey1(Key2048 *pOut) = 0;
            virtual nn::Result GetPublicKey1(Key2048 *pOut) = 0;

            virtual nn::Result GetPublicKey2(Key2048 *pOut) = 0;

            virtual nn::Result GetPublicKey3(Key2048 *pOut) = 0;
            virtual nn::Result GetPrivateKey3(Key2048 *pOut) = 0;

            virtual nn::Result GetPublicKey4(Key2048 *pOut) = 0;

            virtual nn::Result GenerateSessionId(Id128 *pOut) = 0;
            virtual nn::Result GenerateOaepSeed(Sha256Hash *pOut) = 0;
        };

        nn::Result CreateSplAuthenticationKeySource(std::shared_ptr<IAuthenticationKeySource> *pOut);

        class RepairAuthentication
        {
        public:
            RepairAuthentication() {}
            ~RepairAuthentication() {}

            nn::Result Initialize(std::shared_ptr<IAuthenticationKeySource> pKeyStorage);

            nn::Result MakeSessionId(Id128 *pOut);
            nn::Result MakeBackupRequestMessage(BackupRequestMessage *pOut, const Id128 &sessionId, const Key128 &protectedFileKey);
            nn::Result LoadBackupResponseMessage(AuthenticationArchiveContent * pOut, const BackupResponseMessage & response, const Id128 &sessionId);
            nn::Result MakeRestoreRequestMessage(RestoreRequestMessage *pOut, const Id128 &sessionId, const AuthenticationArchiveContent &content);
            nn::Result LoadRestoreResponseMessage(Key128 *pOut, const RestoreResponseMessage &response, const Id128 &sessionId);
        private:
            std::shared_ptr<IAuthenticationKeySource> m_pKeySource;
        };

        typedef nn::Result (*BackupRequestMessageWriter)(const BackupRequestMessage &message);
        typedef nn::Result (*BackupResponseMessageReader)(BackupResponseMessage *pOut);
        typedef nn::Result (*RestoreRequestMessageWriter)(const RestoreRequestMessage &message);
        typedef nn::Result (*RestoreResponseMessageReader)(RestoreResponseMessage *pOut);
        nn::Result RequestBackup(AuthenticationArchiveContent *pOut, Key128 inputKey, const std::string &requestFilePath, const std::string &responseFilePath, std::shared_ptr<FileSystem> fileSystem);
        nn::Result RequestRestore(Key128 *pOut, const AuthenticationArchiveContent &content, std::string restoreRequestPath, std::string restoreResponsePath, std::shared_ptr<FileSystem> fileSystem);
        nn::Result RequestBackupByCallback(AuthenticationArchiveContent *pOut, Key128 inputKey, BackupRequestMessageWriter requestWriter, BackupResponseMessageReader reader);
        nn::Result RequestRestoreByCallback(Key128 *pOut, const AuthenticationArchiveContent &content, RestoreRequestMessageWriter requestWriter, RestoreResponseMessageReader reader);
        nn::Result WriteBackupRequestMessage(const std::string requestFilePath, const BackupRequestMessage &message, std::shared_ptr<FileSystem> fileSystem);
        nn::Result WriteRestoreRequestMessage(const std::string requestFilePath, const RestoreRequestMessage &message, std::shared_ptr<FileSystem> fileSystem);
    }
}
