﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @file  repair_Api.h
* @brief Repair ライブラリの API リストです。
*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include "repair_Types.h"

namespace nn { namespace repair {

//! @name 初期化と終了処理
//! @{

/**
* @brief  Repair ライブラリを初期化します。
*
* @post   Repair ライブラリが初期化済み状態になります。
*         Repair のライブラリで使用するリソースが確保され、
*         ドライバ内部で必要なライブラリが使用可能な状態になります。
*
* @details Repair ライブラリを初期化します。
*/
void Initialize() NN_NOEXCEPT;

/**
* @brief  Repair ライブラリの終了処理をします。
*
* @pre    @ref nn::repair::Initialize() により
*         ライブラリが初期化済みの状態である必要があります
*
* @details Repair ライブラリの終了処理をします。
*/
void Finalize() NN_NOEXCEPT;

//! @}

//! @name バックアップとリストア
//! @{

/**
* @brief  セーブデータを抜き出します。
*
* @param[in]   path    抜き出したセーブデータを保存するディレクトリのパス
* @param[in]   pList   ブラックセーブデータリストの構造体
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   セーブデータの抜き出しに成功しました。
*
* @details セーブデータを抜き出します。
*/
nn::Result ExportSaveData(
        const char* path, const BlackListStruct* pList) NN_NOEXCEPT;

/**
* @brief  セーブデータを抜き出します。
*
* @param[in]   path        抜き出したセーブデータを保存するディレクトリのパス
* @param[in]   pList       ブラックセーブデータリストの構造体
* @param[in]   pCallback   メッセージ表示用の構造体
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   セーブデータの抜き出しに成功しました。
*
* @details セーブデータを抜き出します。
*/
nn::Result ExportSaveData(
        const char* path,
        const BlackListStruct* pList,
        const ReportCallbackStruct* pCallback) NN_NOEXCEPT;

/**
* @brief  セーブデータを安全に抜き出します。
*
* @param[in]   path    抜き出したセーブデータを保存するディレクトリのパス
* @param[in]   pList   ブラックセーブデータリストの構造体
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   セーブデータの抜き出しに成功しました。
*
* @details セーブデータを安全に抜き出します。
*/
nn::Result ExportSaveDataSecurely(
        const char* path, const BlackListStruct* pList) NN_NOEXCEPT;

/**
* @brief  セーブデータを安全に抜き出します。
*
* @param[in]   path        抜き出したセーブデータを保存するディレクトリのパス
* @param[in]   pList   ブラックセーブデータリストの構造体
* @param[in]   pCallback   メッセージ表示用の構造体
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   セーブデータの抜き出しに成功しました。
*
* @details セーブデータを安全に抜き出します。
*/
nn::Result ExportSaveDataSecurely(
        const char* path,
        const BlackListStruct* pList,
        const ReportCallbackStruct* pCallback) NN_NOEXCEPT;

/**
* @brief  バックアップリストア速度計測用の情報を得ます
*
* @param[in]   size    抜き出したセーブデータの総サイズ
* @param[in]   time    かかった時間
* @param[in]   count   処理したファイル数
*
* @pre         ExportSaveData(Securely) , ImportSaveData(Securely) の後で有効な値を得ることが出来ます。
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   処理に成功しました。
*
*/

nn::Result GetTransportRateInfo(uint64_t *pOutSize, int64_t *pOutElapsed, int64_t *pOutCount);


/**
* @brief  ユーザ引っ越し中であればステート情報を出力します
*
* @param[in]   spaceId   セーブデータ種別
* @param[in]   path    　抜き出したセーブデータを保存するディレクトリのパス
* @param[in]   pCallback メッセージ表示用の構造体
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   処理に成功しました。
*
* @details ユーザ引っ越し中でなければ出力は行いません
*/
nn::Result DetectMigration(
    const char* path,
    const ReportCallbackStruct* pCallback) NN_NOEXCEPT;

/**
* @brief  セーブデータを書き込みます。
*
* @param[in]   path    書き込むセーブデータが保存されているディレクトリのパス
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   セーブデータの書き込みに成功しました。
*
* @details セーブデータを書き込みます。
*/
nn::Result ImportSaveData(const char* path) NN_NOEXCEPT;

/**
* @brief  セーブデータを書き込みます。
*
* @param[in]   path        書き込むセーブデータが保存されているディレクトリのパス
* @param[in]   pCallback   メッセージ表示用の構造体
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   セーブデータの書き込みに成功しました。
*
* @details セーブデータを書き込みます。
*/
nn::Result ImportSaveData(
        const char* path, const ReportCallbackStruct* pCallback) NN_NOEXCEPT;

/**
* @brief  セーブデータを安全に書き込みます。
*
* @param[in]   path    書き込むセーブデータが保存されているディレクトリのパス
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   セーブデータの書き込みに成功しました。
*
* @details セーブデータを安全に書き込みます。
*/
nn::Result ImportSaveDataSecurely(const char* path) NN_NOEXCEPT;

/**
* @brief  セーブデータを安全に書き込みます。
*
* @param[in]   path        書き込むセーブデータが保存されているディレクトリのパス
* @param[in]   pCallback   メッセージ表示用の構造体
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   セーブデータの書き込みに成功しました。
*
* @details セーブデータを安全に書き込みます。
*/
nn::Result ImportSaveDataSecurely(
        const char* path, const ReportCallbackStruct* pCallback) NN_NOEXCEPT;

/**
* @brief       デバイス ID を返します。
*
* @param[out]  pOutDeviceId   デバイス ID
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   デバイス ID の取得に成功しました。
*
* @details     10 進数文字列としてデバイス ID を返します。
*              大きさ @ref nn::repair::DeviceIdLength の char 型のバッファを渡してください。
*/
nn::Result GetDeviceId(char* pOutDeviceId) NN_NOEXCEPT;

/**
* @brief       16進数のデバイス ID を返します。
*
* @param[out]  pOutDeviceIdHex   デバイス ID
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   デバイス ID の取得に成功しました。
*
* @details     16 進数文字列としてデバイス ID を返します。
*              大きさ @ref nn::repair::DeviceIdHexLength の char 型のバッファを渡してください。
*/
nn::Result GetDeviceIdHex(char* pOutDeviceId) NN_NOEXCEPT;

/**
* @brief       シリアルナンバーを返します。
*
* @param[out]  pOutSerialNumber   シリアルナンバー
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   シリアルナンバーの取得に成功しました。
*
* @details     シリアルナンバーを返します。
*              大きさ @ref nn::repair::SerialNumberLength の char 型のバッファを渡してください。
*/
nn::Result GetSerialNumber(char *pOutSerialNumber) NN_NOEXCEPT;

/**
* @brief  アカウントのバックアップ状況を調べます。
*
* @param[out]   pOut        バックアップ状況の格納先
* @param[in]    path        バックアップログが保存されているディレクトリのパス
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   アカウントのバックアップ状況取得に成功しました。
*
* @details アカウントのバックアップ状況を調べます。
*/

nn::Result IsAccountExported(bool *pOut, const char* path) NN_NOEXCEPT;

/**
* @brief  アカウントのリストア状況を調べます。
*
* @param[out]   pOut        リストア状況の格納先
* @param[in]    path        リストアログが保存されているディレクトリのパス
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   アカウントのリストア状況取得に成功しました。
*
* @details アカウントのリストア状況を調べます。
*/

nn::Result IsAccountImported(bool *pOut, const char* path) NN_NOEXCEPT;

/**
* @brief  バーチャルアカウントのバックアップ状況を調べます。
*
* @param[out]   pOut        バックアップ状況の格納先
* @param[in]    path        バックアップログが保存されているディレクトリのパス
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   バーチャルアカウントのバックアップ状況取得に成功しました。
*
* @details バーチャルアカウントのバックアップ状況を調べます。
*/

nn::Result IsVirtualAccountExported(bool *pOut, const char* path) NN_NOEXCEPT;

/**
* @brief       自動 CUP 可能な FW バージョンか否かを返します。
*
* @return      処理の結果を返します。
* @retval      true          自動 CUP 可能な FW バージョンです
*
* @details
*/
bool IsAvailableAutoCup() NN_NOEXCEPT;


/**
* @brief       破損していたデータを表示します。
*
* @param[out]  pOutHasCorruption    破損データの有無
* @param[in]   path                 破損データログが保存されているディレクトリのパス
* @param[in]   pFunction            メッセージ表示用の関数ポインタ
* @param[in]   pParameter           表示用関数に与える引数データのポインタ
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   破損データの表示に成功しました。。
*
* @details 破損していたデータを表示します。。
*/
nn::Result ShowCorruptedData(
        bool *pOutHasCorruption, const char* path, ReportFunctionPointer const pFunction, void* const pParameter) NN_NOEXCEPT;

/**
* @brief       ユーザ移行状態の 取得可能な FW バージョンか否かを返します。
*
* @return      処理の結果を返します。
* @retval      true          ユーザ移行状態取得の settings API を呼び出し可能です
*
* @details
*/
bool IsAvailableMigration() NN_NOEXCEPT;

/**
* @brief       再起動不要NUP Version を取得します
*
* @param[out]  pOutVersion   再起動不要NUP Version
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   取得に成功しました。
*
* @details     再起動不要NUP バージョン を返します。
*              システムバージョンの厳密な比較チェックに使用できます。
*              ncm システムセーブデータにアクセスします
*
*/
nn::Result GetSystemUpdateVersion(uint32_t* pOutVersion) NN_NOEXCEPT;

//! @}

}} // namespace nn::repair
