﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/ndd/ndd_Types.h>
#include <nn/ndd/detail/ndd_InternalTypes.h>
#include <nn/ndd/detail/ndd_Handle.h>

#include <nn/nifm/nifm_Request.h>
#include <nn/psc.h>

namespace nn { namespace ndd {

//複数のContextを格納し、Context単位もしくは全Contextを対象とした操作を提供する
//Context自体のInitialize,Finalizeはしない
class ContextContainer
{
public:
    ContextContainer();
    ~ContextContainer();
    void Add(Context* pContext, Handle handle);
    void Remove(Handle handle);
    void Signal(Handle handle, Context::EventId eventId);
    void Signal(Context::EventId eventId);
    void Reply(Handle handle);
    void Reply(Handle handle, const void* pReply);
private:
    static const int ContextCountMax = 3;
    Context* m_pContext[ContextCountMax];
    uint8_t m_ContextCount;
    uint8_t Search(Handle handle);//指定handleが無い場合は0xFFが返る
};

//Context識別用の、固有のハンドルを生成する
class HandleProvider {
public:
    HandleProvider();
    Handle Acquire();
    void Delete(Handle handle);
private:
    static const int HandleCountMax = 3;
    Handle m_Handle[HandleCountMax];
    uint8_t m_HandleCount;
    Handle m_Resource;//再利用によるハンドル値の被りを避けるため、この値をChange()して使い捨てする
    uint8_t Search(Handle handle);
};

//複数の送信データを格納する（現時点では1つのみ）
class SendDataDescriptionContainer {
public:
    SendDataDescriptionContainer();
    void Add(const SendDataDescription& sendDataDescription);
    void Clear();
    bool IsExist() const;
    const SendDataDescription* GetPtr() const;
private:
    SendDataDescription m_SendDataDescription;
    bool m_IsExist;
};

//ステートの操作、取得、遷移順の取得機能を提供する
class StateConfig {
public:
    StateConfig();
    void Awake();
    void Sleep();
    bool IsAwake() const;
    void EnableNetwork();
    void DisableNetwork();
    bool IsNetworkEnabled() const;
    void EnableAutoCommunication();
    void DisableAutoCommunication();
    bool IsAutoCommunicationEnabled() const;
    void AddSendData(const SendDataDescription& sendDataDescription);
    void ClearSendData();
    bool IsSendDataExist() const;
    const SendDataDescription* GetSendDataPtr() const;
    State Get() const;
    static State GetNext(State currentState, State targetState);
private:
    static State GetNextFromStop(State targetState);
    static State GetNextFromIdle(State targetState);
    static State GetNextFromReceive(State targetState);
    static State GetNextFromReceiveSend(State targetState);
    static State GetNextFromSleepStop(State targetState);
    static State GetNextFromSleepIdle(State targetState);
    static State GetNextFromSleepReceive(State targetState);
    static State GetNextFromSleepReceiveSend(State targetState);
    bool m_IsAwake;
    bool m_IsNetworkEnabled;
    bool m_IsAutoCommunication;
    SendDataDescriptionContainer m_SendDataDescriptionContainer;
};

//Scan機能の有効無効を管理する
class ScanConfig {
public:
    ScanConfig();
    bool Enable(Handle handle);//返り値は、Scan新規開始の場合にtrue
    void Disable();
    bool IsEnabled(Handle handle) const;
    uint8_t GetRequesterCount() const;
    Handle GetRequester(uint8_t index) const;
private:
    static const int RequesterCountMax = 3;
    uint8_t m_RequesterCount;
    Handle m_Requester[RequesterCountMax];
    uint8_t Search(Handle handle) const;//指定ハンドルが無い場合0xFFが返る
};

//NetworkRequestのラッパー
class NetworkRequest {
public:
    NetworkRequest();
    ~NetworkRequest();
    void Initialize();//他プロセスのAPIを呼ぶので、コンストラクタから分離
    nn::os::MultiWaitHolderType* GetMultiWaitHolderPtr();
    void ClearEvent();
    nn::nifm::RequestState GetState() const;
    void Submit();
    void Cancel();
private:
    nn::nifm::Request* m_pNetworkRequest;
    nn::os::MultiWaitHolderType m_MultiWaitHolder;
};

//PmModuleのラッパー
class PowerRequest {
public:
    void Initialize();//他プロセスのAPIを呼ぶので、コンストラクタから分離
    nn::os::MultiWaitHolderType* GetMultiWaitHolderPtr();
    void ClearEvent();
    nn::psc::PmState GetState();
    void Acknowledge(nn::psc::PmState pmState);
private:
    nn::psc::PmModule  m_PmModule;
    nn::os::MultiWaitHolderType m_MultiWaitHolder;
};

//MultiWait対象の保持と条件指定でのWait
class MultiWait {
public:
    MultiWait();
    void Link(nn::os::MultiWaitHolderType* pMultiWaitHolder);//MultiWaitIdは設定済み想定
    void Unlink(nn::os::MultiWaitHolderType* pMultiWaitHolder);
    MultiWaitId Wait();//イベントクリアはしない
private:
    nn::os::MultiWaitType m_MultiWait;
};

}}
