﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief  本体機能向けの型や値、演算子を定義します。
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace nd {

//! @name 本体機能向けの型や値、演算子の定義
//! @{

/**
    @brief  送受信データに含まれる本体機能用データの最大サイズ。
*/
static const size_t SystemDataSizeMax = 255;

/**
    @brief  送受信データに含まれる送信者が最近遊んだアプリケーションの最大数。
*/
static const int RecentlyPlayedApplicationCountMax = 3;


/**
    @brief      ユーザーアカウントに紐付いたユーザーの識別子。
    @details   データの送信者を識別するための識別子の1つです。
                ユーザーアカウント毎に本体上で生成され、同じユーザーアカウントを使用する限り同じ値が継続して利用されます。
                受信データ中のこの識別子の一致を調べることにより、同じユーザーアカウントからのデータかどうかを判定することができます。
    @see        NetworkUserId
*/
struct LocalUserId
{
    Bit8 _data[16];
};

//! @brief  LocalUserId の同値比較演算子
inline bool operator ==(const LocalUserId& lhs, const LocalUserId& rhs) NN_NOEXCEPT;


//! @brief LocalUserId の非同値比較演算子
inline bool operator !=(const LocalUserId& lhs, const LocalUserId& rhs) NN_NOEXCEPT;

/**
    @brief      ネットワークサービスアカウントに紐付いたユーザーの識別子。
    @details    データの送信者を識別するための識別子の1つです。
                ネットワークサービスアカウント毎に発行され、同じネットワークサービスアカウントが紐付いたユーザーアカウントには同じIDが与えられます。
                受信データ中のこの識別子の一致を調べることにより、同じネットワークサービスアカウントが紐付いたユーザーアカウントからのデータかどうかを判定することができます。
    @see        LocalUserId
*/
struct NetworkUserId
{
    Bit64 _id;
};

//! @brief  NetworkUserId の同値比較演算子
inline bool operator ==(const NetworkUserId& lhs, const NetworkUserId& rhs) NN_NOEXCEPT;


//! @brief NetworkUserId の非同値比較演算子
inline bool operator !=(const NetworkUserId& lhs, const NetworkUserId& rhs) NN_NOEXCEPT;

//! @}

// 実装

inline bool operator ==(const LocalUserId& lhs, const LocalUserId& rhs) NN_NOEXCEPT
{
    return std::memcmp(lhs._data, rhs._data, sizeof(LocalUserId::_data)) == 0;
}

inline bool operator !=(const LocalUserId& lhs, const LocalUserId& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

inline bool operator ==(const NetworkUserId& lhs, const NetworkUserId& rhs) NN_NOEXCEPT
{
    return lhs._id == rhs._id;
}
inline bool operator !=(const NetworkUserId& lhs, const NetworkUserId& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

}} // ~namespace nn::nd
