﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/migration/migration_AsyncContext.h>
#include <nn/migration/migration_UserMigrationClient.h>
#include <nn/migration/migration_UserMigrationServer.h>
#include <nn/migration/migration_UserMigrationTypes.h>

namespace nn { namespace migration {

/**
    @brief 最後に行ったユーザー移行の再開に関する状態を取得します。
    @pre 有効な UserMigrationServer, UserMigrationClient オブジェクトが存在しない。
*/
LastUserMigrationState GetLastUserMigrationState() NN_NOEXCEPT;

/**
    @brief 存在するすべてのユーザー移行に関する情報を削除します。

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::migration::ResultInternetConnectionUnavailable,
            インターネット接続を利用可能ではありません。
        }
    @endretresult

    @pre 次のそれぞれをみたす。
        - 有効な UserMigrationServer, UserMigrationClient オブジェクトが存在しない。
        - インターネット接続を利用可能。
    @post nn::migration::GetLastUserMigrationState() が nn::migration::LastUserMigrationState::Unresumable を返す。
    @details
        この本体のユーザー移行に関する情報を削除します。
        この本体がユーザー移行のクライアントとして中断状態を持つ場合、移行済みのデータも削除されます。

        nn::migration::GetLastUserMigrationState() が nn::migration::LastUserMigrationState::ServerResumeSuggested を返す場合、次のことに留意してください。

        - この本体の、中断中のユーザー移行の対象ユーザーアカウントに関するすべてのデータが削除されます。
        - 移行先本体で中断中のユーザー移行において UserMigrationClient::IsConnectionRequired() が true 返す場合に、ユーザー移行を再開できなくなります。

        nn::migration::GetLastUserMigrationState() が  nn::migration::LastUserMigrationState::ClientResumeSuggested を返す場合、次のことに留意してください。

        - 移行元本体で中断中のユーザー移行において UserMigrationServer::IsConnectionRequired() が true 返す場合に、ユーザー移行を再開できなくなります。
*/
Result CleanupUserMigration(void* buffer, size_t bufferSize) NN_NOEXCEPT;

/**
    @brief ユーザー移行のサーバーを新しく作成します。
    @pre 有効な UserMigrationServer, UserMigrationClient オブジェクトが存在しない。
    @details
        中断中のユーザー移行が存在する場合、この関数の実行は失敗します。
*/
UserMigrationServer CreateUserMigrationServer(const account::Uid& uid, const UserMigrationServerProfile& profile, void* buffer, size_t bufferSize) NN_NOEXCEPT;

/**
    @brief 中断中のユーザー移行のサーバーを再開します。
    @pre 有効な UserMigrationServer, UserMigrationClient オブジェクトが存在しない。
    @details
        中断中のユーザー移行が存在しない場合、この関数の実行は失敗します。
*/
UserMigrationServer ResumeUserMigrationServer(void* buffer, size_t bufferSize) NN_NOEXCEPT;

/**
    @brief ユーザー移行のクライアントを新しく作成します。
    @pre 有効な UserMigrationServer, UserMigrationClient オブジェクトが存在しない。
    @details
        中断中のユーザー移行が存在する場合、この関数の実行は失敗します。
*/
UserMigrationClient CreateUserMigrationClient(const UserMigrationClientProfile& profile, void* buffer, size_t bufferSize) NN_NOEXCEPT;

/**
    @brief 中断中のユーザー移行のクライアントを再開します。
    @pre 有効な UserMigrationServer, UserMigrationClient オブジェクトが存在しない。
    @details
        中断中のユーザー移行が存在しない場合、この関数の実行は失敗します。
*/
UserMigrationClient ResumeUserMigrationClient(void* buffer, size_t bufferSize) NN_NOEXCEPT;

}} // ~namespace nn::migration
