﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/migration/idc/migration_CommandTypes.h>
#include <nn/migration/idc/migration_ThreadedUserCommandMediator.h>
#include <nn/migration/idc/migration_UserCommandMediator.h>
#include <nn/migration/idc/migration_UserCommandProcessorHolder.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace migration { namespace idc {

struct ServerUserCommandErrorConfig
{
    typedef ResultInvalidRequestSizeForUser        ResultInvalidSize;
    typedef ResultInvalidRequestDataForUser        ResultInvalidData;
    typedef ResultInvalidRequestHeaderForUser      ResultInvalidHeader;
    typedef ResultInvalidRequestBlockSizeForUser   ResultInvalidBlockSize;
};

template <typename EncryptionPolicy, typename UserContextType>
class ServerUserCommandProducer
{
public:
    ServerUserCommandProducer(
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
        : m_Mediator(messageEncryptor, pUserContext, pWorkBuffer)
    {
    }
    Result Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT
    {
        return m_Mediator.Produce(pOutProducedSize, outStream, outStreamSize);
    }
private:
    typedef UserCommandProduceMediator<EncryptionPolicy, UserContextType> Mediator;
    Mediator m_Mediator;
};

template <typename EncryptionPolicy, typename UserContextType>
class ServerUserCommandConsumer
{
public:
    ServerUserCommandConsumer(
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
        : m_Mediator(messageEncryptor, pUserContext, pWorkBuffer)
    {
    }
    Result Consume(void* stream, size_t size) NN_NOEXCEPT
    {
        return m_Mediator.Consume(stream, size);
    }
private:
    typedef UserCommandConsumeMediator<EncryptionPolicy, UserContextType, ServerUserCommandErrorConfig> Mediator;
    Mediator m_Mediator;
};

template <typename EncryptionPolicy, typename UserContextType>
class ThreadedServerUserCommandProducer
{
public:
    ThreadedServerUserCommandProducer(
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        migration::detail::ThreadResourceType* pThreadResource,
        ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
        : m_Mediator(messageEncryptor, pUserContext, pThreadResource, pWorkBuffer)
    {
    }
    Result Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT
    {
        return m_Mediator.Produce(pOutProducedSize, outStream, outStreamSize);
    }
private:
    typedef ThreadedUserCommandProduceMediator<EncryptionPolicy, UserContextType> Mediator;
    Mediator m_Mediator;
};

template <typename EncryptionPolicy, typename UserContextType>
class ThreadedServerUserCommandConsumer
{
public:
    ThreadedServerUserCommandConsumer(
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        migration::detail::ThreadResourceType* pThreadResource,
        ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
        : m_Mediator(messageEncryptor, pUserContext, pThreadResource, pWorkBuffer)
    {
    }
    Result Consume(void* stream, size_t size) NN_NOEXCEPT
    {
        return m_Mediator.Consume(stream, size);
    }
private:
    typedef ThreadedUserCommandConsumeMediator<EncryptionPolicy, UserContextType, ServerUserCommandErrorConfig> Mediator;
    Mediator m_Mediator;
};

template <typename EncryptionPolicy, typename UserContextType>
using ServerUserCommandProducerHolder = UserCommandProcessorHolder<ServerUserCommandProducer<EncryptionPolicy, UserContextType>>;

template <typename EncryptionPolicy, typename UserContextType>
using ServerUserCommandConsumerHolder = UserCommandProcessorHolder<ServerUserCommandConsumer<EncryptionPolicy, UserContextType>>;

template <typename EncryptionPolicy, typename UserContextType>
using ThreadedServerUserCommandConsumerHolder = UserCommandProcessorHolder<ThreadedServerUserCommandConsumer<EncryptionPolicy, UserContextType>>;

template <typename EncryptionPolicy, typename UserContextType>
using ThreadedServerUserCommandProducerHolder = UserCommandProcessorHolder<ThreadedServerUserCommandProducer<EncryptionPolicy, UserContextType>>;

/**
* @brief    サーバー（移行元）の状態遷移を管理するコンテキスト。
*/
template <typename EncryptionPolicyType>
class ServerContext
{
public:
    typedef EncryptionPolicyType EncryptionPolicy;

    ServerContext() NN_NOEXCEPT;
    ~ServerContext() NN_NOEXCEPT;

    bool IsConnected() const NN_NOEXCEPT;
    void SetErrorResponse(ErrorKind error) NN_NOEXCEPT;
    bool HasErrorResponse() const NN_NOEXCEPT;

    Result Consume(void* stream, size_t size) NN_NOEXCEPT;
    Result Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT;

    ServerContext& GetConsumerRef() NN_NOEXCEPT
    {
        return *this;
    }

    ServerContext& GetProducerRef() NN_NOEXCEPT
    {
        return *this;
    }

    template <typename UserContextType>
    ServerUserCommandProducerHolder<EncryptionPolicy, UserContextType> AcquireProducerHolder(
        UserContextType* pContext, UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    {
        typedef ServerUserCommandProducer<EncryptionPolicy, UserContextType> ProducerType;
        NN_STATIC_ASSERT(sizeof(ProducerType) <= sizeof(m_UserCommandProcessorStorage));
        NN_STATIC_ASSERT(std::alignment_of<UserCommandProcessorStorage>::value % std::alignment_of<ProducerType>::value == 0u);

        NN_SDK_REQUIRES_NOT_NULL(pContext);
        NN_ABORT_UNLESS(m_UserCommandProcessorStorageMutex.TryLock());

        auto pProducer = new (&m_UserCommandProcessorStorage) ProducerType(m_MessageEncryptor, pContext, pWorkBuffer);
        return ServerUserCommandProducerHolder<EncryptionPolicy, UserContextType>(pProducer, &m_UserCommandProcessorStorageMutex);
    }

    template <typename UserContextType>
    ServerUserCommandConsumerHolder<EncryptionPolicy, UserContextType> AcquireConsumerHolder(
        UserContextType* pContext, UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    {
        typedef ServerUserCommandConsumer<EncryptionPolicy, UserContextType> ConsumerType;
        NN_STATIC_ASSERT(sizeof(ConsumerType) <= sizeof(m_UserCommandProcessorStorage));
        NN_STATIC_ASSERT(std::alignment_of<UserCommandProcessorStorage>::value % std::alignment_of<ConsumerType>::value == 0u);

        NN_SDK_REQUIRES_NOT_NULL(pContext);
        NN_ABORT_UNLESS(m_UserCommandProcessorStorageMutex.TryLock());

        auto pConsumer = new (&m_UserCommandProcessorStorage) ConsumerType(m_MessageEncryptor, pContext, pWorkBuffer);
        return ServerUserCommandConsumerHolder<EncryptionPolicy, UserContextType>(pConsumer, &m_UserCommandProcessorStorageMutex);
    }

    template <typename UserContextType>
    ThreadedServerUserCommandProducerHolder<EncryptionPolicy, UserContextType> AcquireThreadedProducerHolder(
        UserContextType* pContext, migration::detail::ThreadResourceType* pThreadResource, ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    {
        typedef ThreadedServerUserCommandProducer<EncryptionPolicy, UserContextType> ProducerType;
        NN_STATIC_ASSERT(sizeof(ProducerType) <= sizeof(m_UserCommandProcessorStorage));
        NN_STATIC_ASSERT(std::alignment_of<UserCommandProcessorStorage>::value % std::alignment_of<ProducerType>::value == 0u);

        NN_SDK_REQUIRES_NOT_NULL(pContext);
        NN_SDK_REQUIRES_NOT_NULL(pThreadResource);
        NN_ABORT_UNLESS(m_UserCommandProcessorStorageMutex.TryLock());

        auto pProducer = new (&m_UserCommandProcessorStorage) ProducerType(m_MessageEncryptor, pContext, pThreadResource, pWorkBuffer);
        return ThreadedServerUserCommandProducerHolder<EncryptionPolicy, UserContextType>(pProducer, &m_UserCommandProcessorStorageMutex);
    }

    template <typename UserContextType>
    ThreadedServerUserCommandConsumerHolder<EncryptionPolicy, UserContextType> AcquireThreadedConsumerHolder(
        UserContextType* pContext, migration::detail::ThreadResourceType* pThreadResource, ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    {
        typedef ThreadedServerUserCommandConsumer<EncryptionPolicy, UserContextType> ConsumerType;
        NN_STATIC_ASSERT(sizeof(ConsumerType) <= sizeof(m_UserCommandProcessorStorage));
        NN_STATIC_ASSERT(std::alignment_of<UserCommandProcessorStorage>::value % std::alignment_of<ConsumerType>::value == 0u);

        NN_SDK_REQUIRES_NOT_NULL(pContext);
        NN_SDK_REQUIRES_NOT_NULL(pThreadResource);
        NN_ABORT_UNLESS(m_UserCommandProcessorStorageMutex.TryLock());

        auto pConsumer = new (&m_UserCommandProcessorStorage) ConsumerType(m_MessageEncryptor, pContext, pThreadResource, pWorkBuffer);
        return ThreadedServerUserCommandConsumerHolder<EncryptionPolicy, UserContextType>(pConsumer, &m_UserCommandProcessorStorageMutex);
    }

private:
    Result AcceptCommand(CommandKind command) NN_NOEXCEPT;

    void ClearBufferedConsumableData() NN_NOEXCEPT;
    void ClearClientChallenge() NN_NOEXCEPT;
    void ClearServerChallenge() NN_NOEXCEPT;

    Result ConsumeInitiate0Request(const Initiate0Request& request) NN_NOEXCEPT;
    Result ConsumeInitiate1Request(const Initiate1Request& request) NN_NOEXCEPT;
    Result ConsumeResume0Request(const Resume0Request& request) NN_NOEXCEPT;
    Result ConsumeResume1Request(const Resume1Request& request) NN_NOEXCEPT;
    Result ConsumeTerminateRequest(const TerminateRequest& request) NN_NOEXCEPT;

    void ProduceInitiate0Response(Initiate0Response* pOut) NN_NOEXCEPT;
    void ProduceInitiate1Response(Initiate1Response* pOut) NN_NOEXCEPT;
    void ProduceResume0Response(Resume0Response* pOut) NN_NOEXCEPT;
    void ProduceResume1Response(Resume1Response* pOut) NN_NOEXCEPT;
    void ProduceTerminateResponse(TerminateResponse* pOut) NN_NOEXCEPT;
    void ProduceErrorResponse(ErrorResponse* pOut) NN_NOEXCEPT;

    enum State
    {
        WaitConnection,
        WaitInitiation,
        WaitResume,
        Connected,
    };

    State m_State;
    util::optional<ErrorInfo> m_ErrorInfo;

    CommandKind     m_CommandKind;
    bool            m_IsKeyExchanged;

    typename EncryptionPolicy::KeyEncryptor     m_KeyEncryptor;
    typename EncryptionPolicy::MessageEncryptor m_MessageEncryptor;

    // Optionable にするための wrapper 構造体。
    struct ChallengeStruct
    {
        KeyExchangeCommandConfig::Challenge data;
    };

    util::optional<ChallengeStruct> m_ClientChallenge;
    util::optional<ChallengeStruct> m_ServerChallenge;

    typedef ServerUserCommandConsumer<EncryptionPolicy, void> DummyUserCommandConsumerType;
    typedef ThreadedServerUserCommandConsumer<EncryptionPolicy, void> DummyThreadedUserCommandConsumerType;
    typedef ThreadedServerUserCommandProducer<EncryptionPolicy, void> DummyThreadedUserCommandProducerType;
    NN_STATIC_ASSERT(sizeof(DummyUserCommandConsumerType) <= sizeof(DummyThreadedUserCommandConsumerType));
    NN_STATIC_ASSERT(sizeof(DummyThreadedUserCommandProducerType) <= sizeof(DummyThreadedUserCommandConsumerType));
    typedef typename std::aligned_storage<sizeof(DummyThreadedUserCommandConsumerType), std::alignment_of<DummyThreadedUserCommandConsumerType>::value>::type UserCommandProcessorStorage;
    UserCommandProcessorStorage m_UserCommandProcessorStorage;
    os::Mutex m_UserCommandProcessorStorageMutex;

    util::optional<CommandKind> m_ConsumeCommandKind;
    Bit8   m_ConsumableData[ReqeustCommandSizeMax];
    size_t m_AvailableConsumableDataSize;
    size_t m_ExpectedConsumableDataSize;
};

}}} // ~nn::migration::idc

// 以下実装

#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_Log.h>
#include <nn/migration/idc/migration_CommandApi.h>
#include <nn/migration/idc/migration_Result.h>
#include <nn/migration/idc/detail/migration_Result.h>
#include <nn/migration/idc/detail/migration_Util.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace idc {

// ServerContext

template <typename EncryptionPolicy>
ServerContext<EncryptionPolicy>::ServerContext() NN_NOEXCEPT
    : m_State(State::WaitConnection)
    , m_ErrorInfo()
    , m_CommandKind(CommandKind::Error)
    , m_IsKeyExchanged(false)
    , m_ClientChallenge(nullptr)
    , m_ServerChallenge(nullptr)
    , m_UserCommandProcessorStorageMutex(false)
    , m_ConsumeCommandKind(nullptr)
    , m_AvailableConsumableDataSize(0u)
    , m_ExpectedConsumableDataSize(0u)
{
}

template <typename EncryptionPolicy>
ServerContext<EncryptionPolicy>::~ServerContext() NN_NOEXCEPT
{
    if( m_ClientChallenge )
    {
        ClearClientChallenge();
    }
    if( m_ServerChallenge )
    {
        ClearServerChallenge();
    }
}

template <typename EncryptionPolicy>
bool ServerContext<EncryptionPolicy>::IsConnected() const NN_NOEXCEPT
{
    return (m_State == State::Connected);
}

template <typename EncryptionPolicy>
void ServerContext<EncryptionPolicy>::SetErrorResponse(ErrorKind errorId) NN_NOEXCEPT
{
    m_ErrorInfo.emplace();
    m_ErrorInfo->errorId = errorId;
    std::memset(m_ErrorInfo->padding, 0, sizeof(m_ErrorInfo->padding));
}

template <typename EncryptionPolicy>
bool ServerContext<EncryptionPolicy>::HasErrorResponse() const NN_NOEXCEPT
{
    return static_cast<bool>(m_ErrorInfo);
}

template <typename EncryptionPolicy>
Result ServerContext<EncryptionPolicy>::AcceptCommand(CommandKind command) NN_NOEXCEPT
{
    switch( m_State )
    {
    case State::WaitConnection:
        NN_RESULT_THROW_UNLESS(command == CommandKind::Initiate0 || command == CommandKind::Resume0, ResultInvalidRequestKindForStateWaitConnection());
        break;
    case State::WaitInitiation:
        NN_RESULT_THROW_UNLESS(command == CommandKind::Initiate1, ResultInvalidRequestKindForStateWaitInitiation());
        break;
    case State::WaitResume:
        NN_RESULT_THROW_UNLESS(command == CommandKind::Resume1, ResultInvalidRequestKindForStateWaitResume());
        break;
    case State::Connected:
        NN_RESULT_THROW_UNLESS(command == CommandKind::User || command == CommandKind::Terminate, ResultInvalidRequestKindForStateConnected());
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    m_ConsumeCommandKind = command;
    m_AvailableConsumableDataSize = 0;
    m_ExpectedConsumableDataSize = GetRequestCommandSize(*m_ConsumeCommandKind);
    m_CommandKind = *m_ConsumeCommandKind;
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
Result ServerContext<EncryptionPolicy>::ConsumeInitiate0Request(const Initiate0Request& request) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_IsKeyExchanged, ResultInvalidRequestKeyAlreadyExchangedForInitiate0());
    NN_RESULT_THROW_UNLESS(!m_ClientChallenge, ResultInvalidRequestClientChallengeAlreadyStoredForInitiate0());

    m_ClientChallenge.emplace();
    NN_RESULT_THROW_UNLESS(ParseInitiate0Request(m_ClientChallenge->data, request, m_KeyEncryptor), ResultInvalidRequestDataForInitiate0());
    m_State = State::WaitInitiation;
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
Result ServerContext<EncryptionPolicy>::ConsumeInitiate1Request(const Initiate1Request& request) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ServerChallenge, ResultInvalidRequestServerChallengeNotStoredForInitiate1());

    NN_RESULT_THROW_UNLESS(ParseInitiate1Request(request, m_ServerChallenge->data, m_MessageEncryptor), ResultInvalidRequestDataForInitiate1());

    ClearServerChallenge();
    m_IsKeyExchanged = true;
    m_State = State::Connected;
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
Result ServerContext<EncryptionPolicy>::ConsumeResume0Request(const Resume0Request& request) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsKeyExchanged, ResultInvalidRequestKeyNotExchangedForResume0());
    NN_RESULT_THROW_UNLESS(!m_ClientChallenge, ResultInvalidRequestClientChallengeAlreadyStoredForResume0());

    m_ClientChallenge.emplace();
    NN_RESULT_THROW_UNLESS(ParseResume0Request(m_ClientChallenge->data, request, m_MessageEncryptor), ResultInvalidRequestDataForResume0());
    m_State = State::WaitResume;
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
Result ServerContext<EncryptionPolicy>::ConsumeResume1Request(const Resume1Request& request) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ServerChallenge, ResultInvalidRequestServerChallengeNotStoredForResume1());

    NN_RESULT_THROW_UNLESS(ParseResume1Reqeust(request, m_ServerChallenge->data, m_MessageEncryptor), ResultInvalidRequestDataForResume1());

    ClearServerChallenge();
    m_IsKeyExchanged = true;
    m_State = State::Connected;
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
Result ServerContext<EncryptionPolicy>::ConsumeTerminateRequest(const TerminateRequest& request) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsKeyExchanged, ResultInvalidRequestKeyNotExchangedForTerminate());

    NN_RESULT_THROW_UNLESS(ParseTerminateRequest(request, m_MessageEncryptor), ResultInvalidRequestDataForTerminate());
    m_State = State::WaitConnection;
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
void ServerContext<EncryptionPolicy>::ProduceInitiate0Response(Initiate0Response* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES(!m_IsKeyExchanged);
    NN_SDK_REQUIRES(m_ClientChallenge);
    NN_SDK_REQUIRES(!m_ServerChallenge);

    m_ServerChallenge.emplace();
    detail::GenerateRandomBytes(m_ServerChallenge->data, KeyExchangeCommandConfig::ChallengeSize);
    KeyExchangeCommandConfig::Iv0 iv0;
    detail::GenerateRandomBytes(iv0, KeyExchangeCommandConfig::Iv0Size);
    KeyExchangeCommandConfig::Passphrase passphrase;
    detail::GenerateRandomBytes(passphrase, KeyExchangeCommandConfig::PassphraseSize);

    m_MessageEncryptor.Initialize(passphrase, sizeof(passphrase), EncryptionPolicy::KeyStretchingSalt, sizeof(EncryptionPolicy::KeyStretchingSalt), EncryptionPolicy::KeyStretchingIteration, 0);

    CreateInitiate0Response(pOut, iv0, passphrase, m_ClientChallenge->data, m_ServerChallenge->data, m_KeyEncryptor);
    ClearClientChallenge();
}

template <typename EncryptionPolicy>
void ServerContext<EncryptionPolicy>::ProduceInitiate1Response(Initiate1Response* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    CreateInitiate1Response(pOut, m_MessageEncryptor);
}

template <typename EncryptionPolicy>
void ServerContext<EncryptionPolicy>::ProduceResume0Response(Resume0Response* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES(m_IsKeyExchanged);
    NN_SDK_REQUIRES(m_ClientChallenge);
    NN_SDK_REQUIRES(!m_ServerChallenge);

    m_ServerChallenge.emplace();
    detail::GenerateRandomBytes(m_ServerChallenge->data, KeyExchangeCommandConfig::ChallengeSize);

    CreateResume0Response(pOut, m_ClientChallenge->data, m_ServerChallenge->data, m_MessageEncryptor);
    ClearClientChallenge();
}

template <typename EncryptionPolicy>
void ServerContext<EncryptionPolicy>::ProduceResume1Response(Resume1Response* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES(m_IsKeyExchanged);

    CreateResume1Response(pOut, m_MessageEncryptor);
}

template <typename EncryptionPolicy>
void ServerContext<EncryptionPolicy>::ProduceTerminateResponse(TerminateResponse* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES(m_IsKeyExchanged);

    CreateTerminateResponse(pOut, m_MessageEncryptor);
}

template <typename EncryptionPolicy>
void ServerContext<EncryptionPolicy>::ProduceErrorResponse(ErrorResponse* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES(m_IsKeyExchanged);

    CreateErrorResponse(pOut, *m_ErrorInfo, m_MessageEncryptor);
}

template <typename EncryptionPolicy>
Result ServerContext<EncryptionPolicy>::Consume(void* stream, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(stream);
    NN_SDK_REQUIRES_GREATER(size, 0u);

    if( !m_ConsumeCommandKind )
    {
        auto command = static_cast<CommandKind>(reinterpret_cast<Bit8*>(stream)[0]);
        NN_DETAIL_MIGRATION_TRACE("ServerContext::Consume : %s (0x%02X)\n", detail::GetCommandKindString(command), static_cast<Bit8>(command));
        NN_RESULT_DO(AcceptCommand(command));
    }

    std::memcpy(m_ConsumableData + m_AvailableConsumableDataSize, stream, std::min(size, sizeof(m_ConsumableData) - m_AvailableConsumableDataSize));
    m_AvailableConsumableDataSize += size;
    NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize >= m_ExpectedConsumableDataSize, ResultConsumeCommandContinue());

    switch( *m_ConsumeCommandKind )
    {
    case CommandKind::Initiate0:
        {
            NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize == sizeof(Initiate0Request), ResultInvalidRequestSizeForInitiate0());
            NN_UTIL_SCOPE_EXIT{ ClearBufferedConsumableData(); };
            auto pRequest = reinterpret_cast<Initiate0Request*>(m_ConsumableData);
            return ConsumeInitiate0Request(*pRequest);
        }
    case CommandKind::Initiate1:
        {
            NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize == sizeof(Initiate1Request), ResultInvalidRequestSizeForInitiate1());
            NN_UTIL_SCOPE_EXIT{ ClearBufferedConsumableData(); };
            auto pRequest = reinterpret_cast<Initiate1Request*>(m_ConsumableData);
            return ConsumeInitiate1Request(*pRequest);
        }
    case CommandKind::Resume0:
        {
            NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize == sizeof(Resume0Request), ResultInvalidRequestSizeForResume0());
            NN_UTIL_SCOPE_EXIT{ ClearBufferedConsumableData(); };
            auto pRequest = reinterpret_cast<Resume0Request*>(m_ConsumableData);
            return ConsumeResume0Request(*pRequest);
        }
    case CommandKind::Resume1:
        {
            NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize == sizeof(Resume1Request), ResultInvalidRequestSizeForResume1());
            NN_UTIL_SCOPE_EXIT{ ClearBufferedConsumableData(); };
            auto pRequest = reinterpret_cast<Resume1Request*>(m_ConsumableData);
            return ConsumeResume1Request(*pRequest);
        }
    case CommandKind::Terminate:
        {
            NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize == sizeof(TerminateRequest), ResultInvalidRequestSizeForTerminate());
            NN_UTIL_SCOPE_EXIT{ ClearBufferedConsumableData(); };
            auto pRequest = reinterpret_cast<TerminateRequest*>(m_ConsumableData);
            return ConsumeTerminateRequest(*pRequest);
        }
    case CommandKind::User:  // ユーザーコマンドは ServerUserCommandConsumer/ThreadedServerUserCommandConsumer で処理する。
        {
            NN_RESULT_THROW(ResultInvalidRequestConsumerForUser());
        }
    case CommandKind::Error: // クライアントからエラーコマンドが送信されることはない。
    default:
        // AcceptCommand で引っかかるのでここには到達しない。
        NN_UNEXPECTED_DEFAULT;
    }
}

template <typename EncryptionPolicy>
Result ServerContext<EncryptionPolicy>::Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutProducedSize);
    NN_SDK_REQUIRES_NOT_NULL(outStream);
    NN_SDK_REQUIRES_GREATER(outStreamSize, 0u);
    NN_UNUSED(outStreamSize);

    if( m_State == State::Connected && m_ErrorInfo )
    {
        NN_DETAIL_MIGRATION_TRACE("ServerContext::Produce : %s (0x%02X)\n", detail::GetCommandKindString(CommandKind::Error), static_cast<Bit8>(CommandKind::Error));
        NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(ErrorResponse));
        auto pResponse = reinterpret_cast<ErrorResponse*>(outStream);
        ProduceErrorResponse(pResponse);
        *pOutProducedSize = sizeof(ErrorResponse);
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_MIGRATION_TRACE("ServerContext::Produce : %s (0x%02X)\n", detail::GetCommandKindString(m_CommandKind), static_cast<Bit8>(m_CommandKind));

    switch( m_CommandKind )
    {
    case CommandKind::Initiate0:
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(Initiate0Response));
            auto pResponse = reinterpret_cast<Initiate0Response*>(outStream);
            ProduceInitiate0Response(pResponse);
            *pOutProducedSize = sizeof(Initiate0Response);
            NN_RESULT_SUCCESS;
        }
    case CommandKind::Initiate1:
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(Initiate1Response));
            auto pResponse = reinterpret_cast<Initiate1Response*>(outStream);
            ProduceInitiate1Response(pResponse);
            *pOutProducedSize = sizeof(Initiate1Response);
            NN_RESULT_SUCCESS;
        }
    case CommandKind::Resume0:
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(Resume0Response));
            auto pResponse = reinterpret_cast<Resume0Response*>(outStream);
            ProduceResume0Response(pResponse);
            *pOutProducedSize = sizeof(Resume0Response);
            NN_RESULT_SUCCESS;
        }
    case CommandKind::Resume1:
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(Resume1Response));
            auto pResponse = reinterpret_cast<Resume1Response*>(outStream);
            ProduceResume1Response(pResponse);
            *pOutProducedSize = sizeof(Resume1Response);
            NN_RESULT_SUCCESS;
        }
    case CommandKind::Terminate:
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(TerminateResponse));
            auto pResponse = reinterpret_cast<TerminateResponse*>(outStream);
            ProduceTerminateResponse(pResponse);
            *pOutProducedSize = sizeof(TerminateResponse);
            NN_RESULT_SUCCESS;
        }
    case CommandKind::User:  // ユーザーコマンドは ServerUserCommandProducer で処理する。
    case CommandKind::Error: // エラーは受信したコマンドに関係なく返すので switch 文に入る前に処理をする。
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

template <typename EncryptionPolicy>
void ServerContext<EncryptionPolicy>::ClearBufferedConsumableData() NN_NOEXCEPT
{
    m_ConsumeCommandKind = nullptr;
    m_ExpectedConsumableDataSize = 0;
    m_AvailableConsumableDataSize = 0;
}

template <typename EncryptionPolicy>
void ServerContext<EncryptionPolicy>::ClearClientChallenge() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_ClientChallenge);
    detail::SecureMemoryZero(m_ClientChallenge->data, sizeof(m_ClientChallenge->data));
    m_ClientChallenge = nullptr;
}

template <typename EncryptionPolicy>
void ServerContext<EncryptionPolicy>::ClearServerChallenge() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_ServerChallenge);
    detail::SecureMemoryZero(m_ServerChallenge->data, sizeof(m_ServerChallenge->data));
    m_ServerChallenge = nullptr;
}

}}} // ~nn::migration::idc
