﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/migration/detail/migration_SharedResource.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace detail {

struct ThreadResourceType
{
    char* stack;
    size_t stackSize;
    int priority;
    const char* name;
};

typedef UniqueResource<ThreadResourceType> ThreadResource;
inline Result CreateThread(os::ThreadType* pOut, void(f)(void*), void* arg, const ThreadResource& resource) NN_NOEXCEPT
{
    auto& r = resource.Get();
    NN_RESULT_DO(os::CreateThread(pOut, f, arg, r.stack, r.stackSize, r.priority));
    os::SetThreadNamePointer(pOut, r.name);
    NN_RESULT_SUCCESS;
}

class ThreadResourceAllocator
{
    NN_DISALLOW_COPY(ThreadResourceAllocator);
    NN_DISALLOW_MOVE(ThreadResourceAllocator);

private:
    static const size_t ResourceCountMax = 4u;

    os::Mutex m_Lock;
    int m_Priority;
    const char* m_Name;

    ThreadResourceType m_Resources[ResourceCountMax];
    Counter m_Counters[ResourceCountMax];
    size_t m_Count;

    void InitializeImpl(void* buffer, size_t unit, size_t count) NN_NOEXCEPT;

public:
    ThreadResourceAllocator(int priority, const char* name) NN_NOEXCEPT;

    template <size_t ThreadStackSize>
    void Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        InitializeImpl(buffer, ThreadStackSize, bufferSize / ThreadStackSize);
    }

    ThreadResource Allocate() NN_NOEXCEPT;
};

}}} // ~namespace nn::migration::detail

