﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/account/account_Types.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/ncm/ncm_ApplicationInfo.h>

namespace nn { namespace migration { namespace detail {

struct MigrationListHeader
{
    uint32_t version;
    uint32_t dataCount;
};

// 上限なし
struct DataInfo
{
    enum Type : uint8_t
    {
        Type_System,
        Type_User,
    } type;

    fs::SaveDataSpaceId spaceId;
    fs::SaveDataId id;
    int64_t rawSize;
    int64_t codedSize;

    union
    {
        struct
        {
            fs::SystemSaveDataId systemSaveDataId;
            account::Uid user;

            void Set(fs::SystemSaveDataId _systemSaveDataId, const account::Uid& _user) NN_NOEXCEPT
            {
                systemSaveDataId = _systemSaveDataId;
                user = _user;
            }
        } system;
        struct
        {
            ncm::ApplicationId applicationId;
            uint32_t applicationVersion;
            account::Uid user;

            void Set(const ncm::ApplicationId& _appId, uint32_t _appVer) NN_NOEXCEPT
            {
                Set(_appId, _appVer, account::InvalidUid);
            }
            void Set(const ncm::ApplicationId& _appId, uint32_t _appVer, const account::Uid& _user) NN_NOEXCEPT
            {
                applicationId = _appId;
                applicationVersion = _appVer;
                user = _user;
            }
        } user;
    } attribute;
};

// POD チェック
NN_STATIC_ASSERT(std::is_pod<MigrationListHeader>::value);
NN_STATIC_ASSERT(std::is_pod<DataInfo>::value);

struct MigrationList
{
    struct AcceptableVersion
    {
        static const uint32_t Min = 0;
        static const uint32_t Max = 0;
    };
    static bool IsAccaptable(const MigrationListHeader& header) NN_NOEXCEPT
    {
        return NN_STATIC_CONDITION(header.version >= AcceptableVersion::Min)
            && header.version <= AcceptableVersion::Max;
    }
    static size_t GetMigrationListSize(const MigrationListHeader& header) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsAccaptable(header));
        return sizeof(header)
            + header.dataCount * sizeof(DataInfo);
    }
    static size_t GetDataInfoOffsetInBytes(const MigrationListHeader& header, size_t offset) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsAccaptable(header));
        NN_SDK_ASSERT_LESS_EQUAL(offset, header.dataCount);
        NN_UNUSED(header);
        return sizeof(header)
            + offset * sizeof(DataInfo);
    }
};

}}} // ~namespace nn::migration::detail
