﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ldn/ldn_PrivateTypes.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/ldn_SessionManager.h>
#include <nn/ldn/detail/Advertise/ldn_AdvertiseManager.h>
#include <nn/ldn/detail/Authentication/ldn_ChallengeResponseAuthentication.h>
#include <nn/ldn/detail/NetworkInterface/ldn_FrameReceiver.h>
#include <nn/ldn/detail/NetworkInterface/ldn_FrameSender.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterface.h>
#include <nn/os.h>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    /**
     * @brief           Core の初期化に使用するバッファです。
     */
    struct CoreBuffer
    {
        //! セッション管理に使用するバッファです。
        SessionManagerBuffer sessionManager;

        //! 接続状態監視スレッドのスレッドスタックです。
        char networkMonitorThreadStack[8 * 1024];

        //! Advertise の配信に使用するバッファです。
        char advertiseDistribution[4 * 1024];

        //! Advertise のスキャンに使用するバッファです。
        char advertiseScanner[40 * 1024];

        //! アドバータイズの監視に使用するバッファです。
        char advertiseMonitor[8 * 1024];

        //! 認証用のバッファです。
        char authentication[8 * 1024];

        //! 受信バッファです
        Bit8 receive[8 * 1024];

        //! 送信バッファです
        Bit8 send[8 * 1024];

        //! 切断通知専用の受信バッファです。
        Bit8 rejectReceiveBuffer[4 * 1024];
    };

    /**
     * @brief           Core が管理するノード情報です。
     */
    struct CoreNodeInfo
    {
        //! MAC アドレスです。
        MacAddress macAddress;

        //! ノードの接続状態です。
        bool isConnected;
        NN_PADDING1;

        //! 認証用の識別子です。
        int32_t authenticationId;
        NN_PADDING4;

        //! 接続時刻です。
        int64_t connectedAt;
    };

}}}} // namespace nn::ldn::detail::impl

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief           ノード種別です。
     */
    enum NodeType
    {
        //! ノード種別を選択していません。
        NodeType_None,

        //! アクセスポイントです。
        NodeType_AccessPoint,

        //! ステーションです。
        NodeType_Station,
    };

    /**
     * @brief           ローカル通信のコア機能を提供します。
     */
    class Core
    {
    public:

        /**
         * @brief       初期化に必要なバッファサイズです。
         */
        static const size_t RequiredBufferSize = sizeof(impl::CoreBuffer);

        /**
         * @brief       コンストラクタです。
         * @param[in]   buffer                  初期化に使用するバッファです。
         * @param[in]   bufferSize              初期化に使用するバッファのサイズです。
         */
        Core(void* buffer, size_t bufferSize) NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        ~Core() NN_NOEXCEPT;

        /**
         * @brief       ローカル通信機能の利用を開始します。
         * @param[in]   ローカル通信に使用するネットワーク・インタフェースです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result StartLocalCommunication(INetworkInterface* pNetworkInterface) NN_NOEXCEPT;

        /**
         * @brief       ローカル通信機能の使用を終了します。
         * @param[in]   isTriggeredBySystem     システムによる終了か否かのフラグです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result StopLocalCommunication(bool isTriggeredBySystem) NN_NOEXCEPT;

        /**
         * @brief       現在の状態を取得します。
         * @return      現在の状態です。
         */
        State GetState() const NN_NOEXCEPT;

        /**
         * @brief       周囲のネットワークを探索します。
         * @param[out]  pOutScanResultArray     スキャン結果の出力先です。
         * @param[out]  pOutCount               pOutScanResultArray に出力されたスキャン結果の数です。
         * @param[in]   bufferCount             pOutScanResultArray に格納できるスキャン結果の数です。
         * @param[in]   filter                  スキャンの条件です
         * @param[in]   channel                 スキャン対象のチャンネルです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result Scan(
            NetworkInfo* pOutScanResultArray,
            int* pOutCount,
            int bufferCount,
            const ScanFilter& filter,
            int channel) NN_NOEXCEPT;

        /**
         * @brief       現在参加しているネットワークの情報を取得します。
         * @param[out]  pOutNetwork             ネットワーク情報の出力先です。
         * @param[out]  pOutUpdates             ネットワークに参加しているノードの変化の出力先です。
         * @param[in]   bufferCount             pOutUpdates に格納できる NodeLatestUpdate の数です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result GetNetworkInfo(
            NetworkInfo* pOutNetwork, NodeLatestUpdate* outUpdates, int bufferCount) NN_NOEXCEPT;

        /**
         * @brief       現在参加しているネットワークの情報を取得します。
         * @param[out]  pOutNetwork             ネットワーク情報の出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result GetNetworkInfo(NetworkInfo* pOutNetwork) const NN_NOEXCEPT;

        /**
         * @brief       自身の IP アドレスを取得します。
         * @param[out]  pOutAddress             IP アドレスの出力先です。
         * @param[out]  pOutMask                サブネットマスクの出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result GetIpv4Address(Ipv4Address* pOutAddress, SubnetMask* pOutMask) const NN_NOEXCEPT;

        /**
         * @brief       現在参加しているネットワークのセキュリティ・パラメータを取得します。
         * @param[out]  pOutParam               セキュリティ・パラメータの出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result GetSecurityParameter(SecurityParameter* pOutParam) const NN_NOEXCEPT;

       /**
        * @brief        現在参加しているネットワークの設定情報を取得します。
        * @param[out]   pOutNetworkConfig       ネットワークの設定情報の出力先です。
        * @retresult
        *  @handleresult{ResultSuccess}
        *  @handleresult{ResultInvalidState}
        *  @handleresult{ResultDeviceOccupied}
        * @endretresult
        */
        Result GetNetworkConfig(NetworkConfig* pOutNetworkConfig) const NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントモードで通信を開始します。
         * @param[in]   pStateChangeEvent       状態遷移時にシグナルするイベントです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result OpenAccessPoint(nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントモードを終了します。
         * @param[in]   isTriggeredBySystem     システムによる終了か否かのフラグです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result CloseAccessPoint(bool isTriggeredBySystem) NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントとして新規にネットワークを構築します。
         * @param[in]   network                 構築するネットワークの設定情報です。
         * @param[in]   securityConfig          構築するネットワークのセキュリティ設定です。
         * @param[in]   securityParam           構築するネットワークのセキュリティ・パラメータです。
         * @param[in]   user                    ネットワークを構築するユーザの情報です。
         * @param[in]   addressEntryCount       静的に割り当てる IPv4 アドレスの数です。
         * @param[in]   addressEntries          静的に割り当てる IPv4 アドレスです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result CreateNetwork(
            const NetworkConfig& network,
            const SecurityConfig& securityConfig,
            const SecurityParameter& securityParam,
            const UserConfig& user,
            int addressEntryCount,
            const AddressEntry* addressEntries) NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントとして構築したネットワークを破棄します。
         * @param[in]   isTriggeredBySystem     システムによる終了か否かのフラグです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result DestroyNetwork(bool isTriggeredBySystem) NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントが配信する任意データを設定します。
         * @param[in]   data                    配信するデータです。
         * @param[in]   dataSize                配信するデータのサイズです。
         *                                      @ref AdvertiseDataSizeMax 以下の値を設定します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result SetAdvertiseData(const void* data, size_t dataSize) NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントにおけるステーションの接続受付ポリシーを変更します。
         * @param[in]   policy                  新しい接続受付ポリシーです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result SetStationAcceptPolicy(AcceptPolicy policy) NN_NOEXCEPT;

        /**
         * @brief       指定したステーションをネットワークから追放します。
         * @param[in]   ipv4Address             対象のステーションの IPv4 アドレスです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultNodeNotFound}
         * @endretresult
         */
        Result Reject(Ipv4Address ipv4Address) NN_NOEXCEPT;

        /**
         * @brief       ステーションの接続を制限するフィルタにエントリを追加します。
         * @param[in]   macAddress             対象のステーションの MAC アドレスです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result AddAcceptFilterEntry(MacAddress macAddress) NN_NOEXCEPT;

        /**
         * @brief       フィルタに登録されているエントリを全て削除します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result ClearAcceptFilter() NN_NOEXCEPT;

        /**
         * @brief       ステーションモードで通信を開始します。
         * @param[in]   pStateChangeEvent       状態遷移時にシグナルするイベントです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result OpenStation(nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT;

        /**
         * @brief       ステーションモードを終了します。
         * @param[in]   isTriggeredBySystem     システムによる終了か否かのフラグです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result CloseStation(bool isTriggeredBySystem) NN_NOEXCEPT;

        /**
         * @brief       ステーションとして指定されたネットワークに接続します。
         * @param[in]   network                 接続先のネットワークの情報です。
         * @param[in]   securityConfig          接続に必要なセキュリティパラメータです。
         * @param[in]   securityParam           接続先のネットワークのセキュリティ・パラメータです。
         * @param[in]   user                    ネットワークに参加するユーザの情報です。
         * @param[in]   version                 ローカル通信バージョンです。
         * @param[in]   option                  オプションです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultNetworkNotFound}
         *  @handleresult{ResultConnectionTimeout}
         *  @handleresult{ResultConnectionRejected}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result Connect(
            const NetworkConfig& network,
            const SecurityConfig& securityConfig,
            const SecurityParameter& securityParam,
            const UserConfig& user,
            int version,
            ConnectOption option) NN_NOEXCEPT;

        /**
         * @brief       ステーションとして接続中のネットワークから切断します。
         * @param[in]   isTriggeredBySystem     システムによる終了か否かのフラグです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result Disconnect(bool isTriggeredBySystem) NN_NOEXCEPT;

        /**
         * @brief       切断理由を取得します。
         * @param[out:  pOutReason              切断理由の出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result GetDisconnectReason(DisconnectReason* pOutReason) const NN_NOEXCEPT;

        /**
         * @brief       ローカル通信の動作モードを設定します。
         * @param[in]   mode                    ローカル通信の動作モードです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result SetOperationMode(OperationMode mode) NN_NOEXCEPT;

        /**
         * @brief       無線コントローラーの接続制限を有効化あるいは無効化します。
         * @param[in]   restriction             接続制限の設定です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result SetWirelessControllerRestriction(
            WirelessControllerRestriction restriction) NN_NOEXCEPT;

    private:

        // 接続状態の変化を監視するスレッドです。
        static void AccessPointMonitor(void* pArg) NN_NOEXCEPT;
        static void StationMonitor(void* pArg) NN_NOEXCEPT;
        void StartNetworkMonitoring(nn::os::ThreadFunction func) NN_NOEXCEPT;
        void StopNetworkMonitoring() NN_NOEXCEPT;
        void OnAccessPointStateChanged() NN_NOEXCEPT;
        void OnStationConnected(const L2StationInfo& l2) NN_NOEXCEPT;
        void OnStationDisconnected(int aid) NN_NOEXCEPT;
        void OnAuthenticationProcessed() NN_NOEXCEPT;
        void OnStationStateChanged() NN_NOEXCEPT;
        void OnRejectRequestReceived() NN_NOEXCEPT;
        void OnAdvertiseReceived() NN_NOEXCEPT;

        // アドバータイズで配信するデータを更新します。
        Result UpdateAdvertise() NN_NOEXCEPT;

        // 実装の詳細です。
        void OpenCommon(
            nn::os::ThreadFunction func, nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT;
        void CloseCommon() NN_NOEXCEPT;
        Result Authenticate(
            MacAddress bssid, const NetworkId& networkId, const Bit8 (&serverRandom)[RandomSize],
            const UserConfig& user, int localCommunicationVersion) NN_NOEXCEPT;
        Result DisconnectImpl(DisconnectReason reason) NN_NOEXCEPT;
        Result SendRejectRequest(int aid, DisconnectReason reason) NN_NOEXCEPT;

        nn::os::ThreadType m_NetworkMonitorThread;
        nn::os::Event m_CloseEvent;
        nn::os::Event m_RejectEvent;
        mutable nn::os::Mutex m_NetworkInterfaceMutex;
        mutable nn::os::Mutex m_SessionMutex;
        FrameQueue m_RejectQueue;

        impl::CoreBuffer* m_Buffer;
        INetworkInterface* m_pNetworkInterface;
        FrameReceiver m_Receiver;
        FrameSender m_Sender;
        SessionManager m_SessionManager;
        AdvertiseDistributor m_AdvertiseDistributor;
        AdvertiseScanner m_AdvertiseScanner;
        AdvertiseMonitor m_AdvertiseMonitor;
        ChallengeResponseAuthenticationServer m_AuthenticationServer;
        ChallengeResponseAuthenticationClient m_AuthenticationClient;

        State m_State;
        DisconnectReason m_DisconnectReason;
        impl::CoreNodeInfo m_Stations[StationCountMax];
    };

}}} // namespace nn::ldn::detail
