﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_Common.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         OUI です。
     */
    struct Oui
    {
        Bit8 raw[3];
    };

    /**
     * @brief         OUI を生成します。
     *
     * @param[in]     byte1       OUI の 1 バイト目
     * @param[in]     byte2       OUI の 2 バイト目
     * @param[in]     byte3       OUI の 3 バイト目
     *
     * @return        生成された OUI です。
     */
    inline const Oui MakeOui(Bit8 byte1, Bit8 byte2, Bit8 byte3) NN_NOEXCEPT
    {
        Oui oui = { {byte1, byte2, byte3} };
        return oui;
    }

    /**
     * @brief         OUI を生成します。
     *
     * @param[in]     raw         OUI の生の値
     *
     * @return        生成された OUI です。
     */
    inline const Oui MakeOui(const Bit8 (&raw)[3]) NN_NOEXCEPT
    {
        Oui oui = { {raw[0], raw[1], raw[2]} };
        return oui;
    }

    /**
     * @brief         OUI を比較します。
     *
     * @param[in]     lhs         比較対象の OUI
     * @param[in]     rhs         比較対象の OUI
     *
     * @return        OUI が等しい場合に true です。
     */
    inline bool operator == (Oui lhs, Oui rhs) NN_NOEXCEPT
    {
        return std::memcmp(lhs.raw, rhs.raw, sizeof(Oui)) == 0;
    }

    /**
     * @brief         OUI を比較します。
     *
     * @param[in]     lhs         比較対象の OUI
     * @param[in]     rhs         比較対象の OUI
     *
     * @return        OUI が等しくない場合に true です。
     */
    inline bool operator != (Oui lhs, Oui rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

}}} // namespace nn::ldn::server::detail
