﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/ldn_MacAddress.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief           MTU の最小値です。
     */
    const size_t MtuMin = 576;

    /**
     * @brief           ネットワークインタフェース名の最大文字数です。終端文字は含みません。
     */
    const int NetworkInterfaceNameLengthMax = 256;

    /**
     * @brief           サポートするチャンネルの最大数です。
     */
    const int SupportedChannelCountMax = 32;

    /**
     * @brief           自動選択の候補となるチャンネルの最大数です。
     */
    const int AutoSelectableChannelCountMax = 16;

    /**
     * @brief           ネットワークインタフェース要求時に指定できる優先度の最小値です。
     */
    const int NetworkInterfaceRequestPriorityMin = 0;

    /**
     * @brief           ネットワークインタフェース要求時に指定できる優先度の最大値です。
     */
    const int NetworkInterfaceRequestPriorityMax = 255;

    /**
     * @brief           ネットワークインタフェースの種別を表す定数です。
     */
    enum NetworkInterfaceType
    {
        //! 無効なネットワークインタフェースです。
        NetworkInterfaceType_Invalid,

        //! ループバック・インタフェースです。
        NetworkInterfaceType_Loopback,

        //! Ethernet (IEEE 802.3) です。
        NetworkInterfaceType_Ethernet,

        //! 無線インタフェース (IEEE 802.11) です。
        NetworkInterfaceType_Wireless80211
    };

    /**
     * @brief           ネットワークインタフェースのケイパビリティです。
     */
    enum NetworkInterfaceCapability
    {
        //! ハードウェア・レベルの暗号化に対応しています。
        NetworkInterfaceCapability_HardwareEncryption = 1 << 0,

        //! 自動チャンネル選択に対応しています
        NetworkInterfaceCapability_AutoChannelSelection = 1 << 1,

        //! 全てのケイパビリティです。
        NetworkInterfaceCapability_All = NetworkInterfaceCapability_HardwareEncryption
    };

    /**
     * @brief           ネットワークインタフェースの情報を格納する構造体です。
     */
    struct NetworkInterfaceProfile
    {
        //! MTU をバイト単位で指定します。
        uint32_t        mtu;

        //! @ref NetworkInterfaceCapability の論理和です。
        Bit32           capability;

        //! ネットワークに接続可能なノードの数です。
        int16_t         nodeCountMax;

        //! @ref NetworkInterfaceType です。
        Bit8            type;

        //! ネットワークインタフェースの名称です。
        char            name[NetworkInterfaceNameLengthMax + 1];

        //! 物理アドレスです。
        MacAddress      physicalAddress;

        //! ビーコンデータの最大サイズです。
        uint16_t        beaconDataSizeMax;
        Bit8            _reserved[10];

        //! サポートする無線チャンネルの総数です。
        int8_t          supportedChannelCount;

        //! 自動チャンネル選択で候補となるチャンネルの総数です。
        int8_t          autoSelectableChannelCount;

        //! サポートする無線チャンネルです。昇順にソートされています。
        int16_t         supportedChannels[SupportedChannelCountMax];

        //! 自動チャンネル選択で選択されうる無線チャンネルです。昇順にソートされています。
        int16_t         autoSelectableChannels[AutoSelectableChannelCountMax];
    };

}}} // namespace nn::ldn::detail
