﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief       フレームを格納するキューです。
     */
    class FrameQueue
    {
    public:

        /**
         * @brief       コンストラクタです。
         */
        FrameQueue() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        ~FrameQueue() NN_NOEXCEPT;

        /**
         * @brief       初期化に必要なバッファサイズを計算します。
         * @param[in]   size            要素のサイズです。
         * @param[in]   capacity        キューの要素数です。
         * @return      バッファサイズです。
         */
        static size_t GetRequiredBufferSize(size_t size, int capacity) NN_NOEXCEPT;

        /**
         * @brief       キューを初期化します。
         * @param[in]   buffer          受信バッファです。
         * @param[in]   bufferSize      受信バッファのバイトサイズです。
         * @param[in]   size            要素のサイズです。
         * @param[in]   capacity        キューの要素数です。
         */
        void Initialize(void* buffer, size_t bufferSize, size_t size, int capacity) NN_NOEXCEPT;

        /**
         * @brief       キューを破棄します。
         */
        void Finalize() NN_NOEXCEPT;

        /**
         * @brief       キューにデータを追加します。
         * @param[in]   data            対象のデータです。
         * @param[in]   dataSize        データサイズです。
         * @param[in]   macAddress      送信先あるいは送信元の MAC アドレスです。
         */
        void Enqueue(const void* data, size_t dataSize) NN_NOEXCEPT;

        /**
         * @brief       キューにデータを追加します。
         * @param[in]   data            対象のデータです。
         * @param[in]   dataSize        データサイズです。
         * @retval      true            成功しました。
         * @retval      false           キューに空きがないため失敗しました。
         */
        bool TryEnqueue(const void* data, size_t dataSize) NN_NOEXCEPT;

        /**
         * @brief       キューの先頭のデータを取得して削除します。
         * @param[out]  buffer          データの出力先です。
         * @param[out]  pOutSize        データサイズの出力先です。
         * @param[in]   bufferSize      バッファサイズです。
         */
        void Dequeue(void* buffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT;

        /**
         * @brief       キューの先頭のデータを取得して削除します。
         * @param[out]  buffer          データの出力先です。
         * @param[out]  pOutSize        データサイズの出力先です。
         * @param[in]   bufferSize      バッファサイズです。
         * @retval      true            成功しました。
         * @retval      false           データが無いため失敗しました。
         */
        bool TryDequeue(void* buffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT;

        /**
         * @brief       キューの先頭のデータを取得せずに削除します。
         */
        void Dequeue() NN_NOEXCEPT;

        /**
         * @brief       キューの先頭のデータを取得せずに削除します。
         * @retval      true            成功しました。
         * @retval      false           データが無いため失敗しました。
         */
        bool TryDequeue() NN_NOEXCEPT;

        /**
         * @brief       キューを空にします。
         */
        void Clear() NN_NOEXCEPT;

        /**
         * @brief       キューが空か否かを調べます。
         * @return      キューが空の場合に true です。
         */
        bool IsEmpty() const NN_NOEXCEPT;

        /**
         * @brief       先頭のデータを削除せずに取得します。
         * @param[out]  pOutSize        先頭のデータのバイトサイズです。
         * @return      キューの先頭のデータです。
         */
        const void* GetFront(size_t* pOutSize) const NN_NOEXCEPT;

    private:

        struct FrameQueueItemHeader
        {
            int16_t  index;
            uint16_t dataSize;
            Bit8 _reserved[4];
        };

        struct FrameQueueItem
        {
            FrameQueueItemHeader header;
            Bit8 data[2040];
        };

        FrameQueueItem& GetItem(int index) NN_NOEXCEPT;
        const FrameQueueItem& GetItem(int index) const NN_NOEXCEPT;

        Bit8* m_Buffer;
        size_t m_Size;
        int m_Capacity;
        int m_Head;
        int m_Tail;
        int m_Count;
        nn::os::Mutex m_Mutex;
    };

}}} // namespace nn::ldn::detail
