﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/ldn/ldn_MacAddress.h>
#include <nn/ldn/detail/NetworkInterface/ldn_Oui.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         Ethertype です。
     */
    struct Ethertype
    {
        Bit8 raw[2];
    };

    /**
     * @brief         Ethertype を生成します。
     *
     * @param[in]     byte1       Ethertype の 1 バイト目
     * @param[in]     byte2       Ethertype の 2 バイト目
     *
     * @return        生成された Ethertype です。
     */
    inline const Ethertype MakeEthertype(Bit8 byte1, Bit8 byte2) NN_NOEXCEPT
    {
        Ethertype ethertype = { {byte1, byte2} };
        return ethertype;
    }

    /**
     * @brief         Ethertype を生成します。
     *
     * @param[in]     raw         Ethertype の生の値
     *
     * @return        生成された Ethertype です。
     */
    inline const Ethertype MakeEthertype(const Bit8 (&raw)[2]) NN_NOEXCEPT
    {
        Ethertype ethertype = { {raw[0], raw[1]} };
        return ethertype;
    }

    /**
     * @brief         Ethertype を比較します。
     *
     * @param[in]     lhs         比較対象の Ethertype
     * @param[in]     rhs         比較対象の Ethertype
     *
     * @return        Ethertype が等しい場合に true です。
     */
    inline bool operator == (Ethertype lhs, Ethertype rhs) NN_NOEXCEPT
    {
        return std::memcmp(lhs.raw, rhs.raw, sizeof(Ethertype)) == 0;
    }

    /**
     * @brief         Ethertype を比較します。
     *
     * @param[in]     lhs         比較対象の Ethertype
     * @param[in]     rhs         比較対象の Ethertype
     *
     * @return        Ethertype が等しくない場合に true です。
     */
    inline bool operator != (Ethertype lhs, Ethertype rhs) NN_NOEXCEPT
    {
        return std::memcmp(lhs.raw, rhs.raw, sizeof(Ethertype)) != 0;
    }

    /**
     * @brief         OUI Extended Ethertype です。
     */
    const Ethertype OuiExtendedEthertype = MakeEthertype(0x88, 0xB7);

    /**
     * @brief         Ethernet フレームの最大ペイロードサイズです。
     */
    const size_t EthernetFramePayloadSizeMax = 1500;

    /**
     * @brief         Ethernet のフレームヘッダです。
     */
    struct EthernetHeader
    {
        MacAddress                  dst;
        MacAddress                  src;
        Ethertype                   ethertype;
    };

    /**
     * @brief         Ethernet のフレームです。
     */
    struct EthernetFrame
    {
        EthernetHeader              ethernetHeader;
        Bit8                        data[EthernetFramePayloadSizeMax];
    };

    /**
     * @brief         Ethernet フレームの最小サイズです。
     */
    const size_t EthernetFrameSizeMin = sizeof(EthernetHeader);

    /**
     * @brief         OUI Extended Ethertype のヘッダです。
     */
    struct OuiExtendedEthernetHeader
    {
        Oui                         oui;
    };

    /**
     * @brief         OUI Extended Ethernet フレームの最大ペイロードサイズです。
     */
    const size_t OuiExtendedEthernetFramePayloadSizeMax =
        EthernetFramePayloadSizeMax - sizeof(OuiExtendedEthernetHeader);

    /**
     * @brief         OUI Extended Ethertype のフレームです。
     */
    struct OuiExtendedEthernetFrame
    {
        EthernetHeader              ethernetHeader;
        OuiExtendedEthernetHeader   ouiExtendedEthernetHeader;
        Bit8                        data[OuiExtendedEthernetFramePayloadSizeMax];
    };
    NN_STATIC_ASSERT(sizeof(EthernetFrame) == sizeof(OuiExtendedEthernetFrame));

    /**
     * @brief         OUI Extended Ethertype フレームの最小サイズです。
     */
    const size_t OuiExtendedEthernetFrameSizeMin =
        sizeof(OuiExtendedEthernetFrame) - OuiExtendedEthernetFramePayloadSizeMax;

    /**
     * @brief         情報エレメントのペイロードの最大サイズです。
     */
    const size_t IePayloadSizeMax = 255;

    /**
     * @brief         ベンダ定義の情報エレメントのペイロードの最大サイズです。
     */
    const size_t VendorSpecificIePayloadSizeMax
        = IePayloadSizeMax - sizeof(Oui);

    /**
     * @brief         情報エレメントのヘッダです。
     */
    struct IeHeader
    {
        Bit8                        elementId;
        uint8_t                     length;
    };

    /**
     * @brief         情報エレメントです。
     */
    struct Ie
    {
        IeHeader                    header;
        Bit8                        payload[IePayloadSizeMax];
    };

    /**
     * @brief         Vendor Specific Information Element のヘッダです。
     */
    struct VendorSpecificIeHeader
    {
        Bit8                        elementId;
        uint8_t                     length;
        Oui                         oui;
    };

    /**
     * @brief         ベンダ定義の情報エレメントの Body です。
     */
    struct VendorSpecificIe
    {
        VendorSpecificIeHeader      header;
        Bit8                        payload[VendorSpecificIePayloadSizeMax];
    };
    NN_STATIC_ASSERT(sizeof(Ie) == sizeof(VendorSpecificIe));

    /**
     * @brief         Action Frame のカテゴリです。
     */
    enum ActionFrameCategory
    {
        ActionFrameCategory_VendorSpecific  = 127
    };

    /**
     * @brief         Action Frame のヘッダです。
     */
    struct ActionFrameHeader
    {
        MacAddress                  dst;
        MacAddress                  src;
        MacAddress                  bssid;
        uint16_t                    seq;
    };

    /**
     * @brief         Action Frame の最大サイズです。
     *
     * @details       規格における最大サイズではなく、独自の制限です。
     *                多少の余裕をもって TCP の MSS に収まるように設定されています。
     */
    const size_t ActionFrameSizeMax = 1440;

    /**
     * @brief         Action Frame のペイロードの最大サイズです。
     */
    const size_t ActionFramePayloadSizeMax =
        ActionFrameSizeMax - sizeof(ActionFrameHeader);

    /**
     * @brief         Action Frame です。
     */
    struct ActionFrame
    {
        ActionFrameHeader           header;
        Bit8                        payload[ActionFramePayloadSizeMax];
    };

}}} // namespace nn::ldn::server::detail
